import os
from gi.repository import GLib, GdkPixbuf

from ubatar_lib.ubatarconfig import get_data_file

def pixbuf_normalize(pixbuf, size):
    """
    Crop and scale a GdkPixbuf object to adjust to the given size and aspect ratio.
    Return a GdkPixbuf
    """
    width = pixbuf.get_width()
    height = pixbuf.get_height()
    src_x = 0
    src_y = 0
    if width < height:
        src_x = 0
        src_y = (height - width) / 2
        new_pixbuf = pixbuf.new_subpixbuf(src_x, src_y, width, width)
    elif width > height:
        src_x = (width - height) / 2
        src_y = 0
        new_pixbuf = pixbuf.new_subpixbuf(src_x, src_y, height, height)
    else:
        new_pixbuf = pixbuf
    new_pixbuf = new_pixbuf.scale_simple(size, size, GdkPixbuf.InterpType.BILINEAR)
    return new_pixbuf


def find_ribbons_on_dir(dirname):
    """
    Find valid image files with aspect ratio 1
    """
    ribbons = []
    files = os.listdir(dirname)
    for filename in files:
        file_path = os.path.join(dirname, filename)
        try:
            pixbuf = GdkPixbuf.Pixbuf.new_from_file(file_path)
            width = float(pixbuf.get_width())
            height = float(pixbuf.get_height())
            ratio = width / height
            if ratio == 1.0:
                ribbons.append(file_path)
        except GLib.GError:
            # Ignore unknown file image format
            pass
    return ribbons


def get_ribbons_files():
    """
    Return a dict of ribbon files with aspect ratio 1.
    """
    user_dir = os.path.join(GLib.get_user_data_dir(), 'ubatar/ribbons')
    try:
        os.makedirs(user_dir)
    except OSError:
        pass

    default_dir = get_data_file('media', 'ribbons')

    empty_ribbon = get_data_file('media', 'empty.png')
    ribbons = [empty_ribbon]

    for dirname in [default_dir, user_dir]:
        ribbons_found = find_ribbons_on_dir(dirname)
        ribbons.extend(ribbons_found)

    ribbons_dict = {}
    for index, value in enumerate(ribbons):
        ribbons_dict[index] = value
    return ribbons_dict


