import os.path
import logging
import urllib2

from gi.repository import GLib, GdkPixbuf # pylint: disable=E0611
from ubatar.plugins import BaseService

from launchpadlib.launchpad import Launchpad

logger = logging.getLogger('ubatar')


class LaunchpadService(BaseService):
    """Service to set profile picture for Launchpad."""
    name = 'launchpad'
    title = 'Launchpad'
    use_auth = True
    icon = 'launchpad.png'
    provides = ['get', 'set']

    def __init__(self, key='', secret=''):
        # Mugshot image size 192x192 pixels
        self.mugshot_size = 192

        # Logo image size 192x192 pixels
        self.logo_size = 64

        self.img_mimetype = 'image/png'
        self.img_format = 'png'
        self.img_name = '%s.%s' % (self.name, self.img_format)
       
    def set_image(self, pixbuf):
        """Set image from a GdkPixbuf object."""
        launchpad = Launchpad.login_with('Ubatar', 'production')

        logger.debug('%s: updating mugshot' % self.name)
        image_path = self._normalize_image(pixbuf, self.mugshot_size)
        mugshot = launchpad.me.mugshot
        with open(image_path) as f:
            mugshot_handle = mugshot.open('w', self.img_mimetype, self.img_name)
            mugshot_handle.write(f.read())
            mugshot_handle.close()

        logger.debug('%s: updating logo' % self.name)
        image_path = self._normalize_image(pixbuf, self.logo_size)
        logo = launchpad.me.logo
        with open(image_path) as f:
            logo_handle = logo.open('w', self.img_mimetype, self.img_name)
            logo_handle.write(f.read())
            logo_handle.close()

    def get_image(self):
        """Get image from Launchpad and return its path."""
        launchpad = Launchpad.login_with('Ubatar', 'production')

        image_path = os.path.join(GLib.get_tmp_dir(), self.img_name)
        mugshot_link = launchpad.me.mugshot_link
        image_data = urllib2.urlopen(mugshot_link)
        with open(image_path, 'w') as f:
            f.write(image_data.read())
            return image_path

    def _normalize_image(self, pixbuf, pixels):
        """
        Normalize image used for Launchpad mugshot's profile picture
        (PNG format 192x192 pixels) and return the path to a temporary file.
        """
        pixbuf_resized = pixbuf.scale_simple(pixels, pixels, GdkPixbuf.InterpType.BILINEAR)
        image_path = os.path.join(GLib.get_tmp_dir(), self.img_name)
        pixbuf_resized.savev(image_path, self.img_format, [], [])
        return image_path
