import os.path
import logging
import hashlib
import xmlrpclib
import base64
import urllib2

from gi.repository import GLib, GdkPixbuf # pylint: disable=E0611
from ubatar.plugins import BaseService

logger = logging.getLogger('ubatar')


class GravatarService(BaseService):
    """Service to set profile picture for Gravatar."""
    name = 'gravatar'
    title = 'Gravatar'
    description = 'Used by AskUbuntu (StackExchange), Github, Disqus, Wordpress'
    use_auth = True
    icon = 'gravatar.png'
    provides = ['get', 'set']


    def __init__(self, email, password):
        self.img_format = 'jpeg'
        self.img_name = '%s.%s' % (self.name, self.img_format)
        self.image_path = os.path.join(GLib.get_tmp_dir(), self.img_name)
        api_uri = 'https://secure.gravatar.com/xmlrpc?user={0}'
        self.email = email.lower().strip()
        self.password = password

        self._user_hash = hashlib.md5(self.email).hexdigest()
        self._server = xmlrpclib.ServerProxy(api_uri.format(self._user_hash))
       
    def set_image(self, pixbuf):
        """Set image from a GdkPixbuf object."""
        pixbuf.savev(self.image_path, self.img_format, [], [])

        logger.debug('%s: updating image' % self.name)
        self._upload_image(self.image_path)

    def get_image(self):
        """Get image from Gravatar website."""
        image_url = 'http://www.gravatar.com/avatar/%s?s=512' % self._user_hash
        image_data = urllib2.urlopen(image_url)
        with open(self.image_path, 'w') as f:
            f.write(image_data.read())
            return self.image_path

    def _upload_image(self, image_path):
        """Upload image to Gravatar."""
        with open(image_path) as f:
            image_encoded = base64.b64encode(f.read())
            params = {
                'data': image_encoded,
                'rating': 0,
            }
            image_hash = self._call('saveData', params)
            logger.debug('%s: image uploaded' % self.name)
            params = {
                'userimage': image_hash,
                'addresses': [self.email],
            }
            self._call('useUserimage', params)
            logger.debug('%s: image selected' % self.name)

    def _call(self, method, params={}):
        """Call a method from the API, gets 'grav.' prepended to it."""
        args = params
        args['password'] = self.password
        return getattr(self._server, 'grav.' + method, None)(args)
