import os.path
import logging

from gi.repository import GLib, Gio, GdkPixbuf # pylint: disable=E0611
from ubatar.plugins import BaseService

logger = logging.getLogger('ubatar')


class FreeDesktopService(BaseService):
    """
    This class implement the specification AccountsService proposed by freedesktop.org
    http://www.freedesktop.org/wiki/Software/AccountsService
    """
    name = 'freedesktop'
    title = 'Ubuntu'
    use_auth = False
    icon = 'ubuntu.png'
    provides = ['set']
    is_default = True

    def __init__(self):
        # Default image size and format.
        self.width = 96
        self.heigth = 96
        self.img_format = 'png'

    def set_image(self, pixbuf):
        """Set image from a GdkPixbuf object."""
        image_path = self._normalize_image(pixbuf)
        current_user = GLib.get_user_name ()
        bus = Gio.bus_get_sync (Gio.BusType.SYSTEM, None)
        result = bus.call_sync ('org.freedesktop.Accounts',
                                '/org/freedesktop/Accounts',
                                'org.freedesktop.Accounts',
                                'FindUserByName',
                                GLib.Variant ('(s)', (current_user,)),
                                GLib.VariantType.new ('(o)'),
                                Gio.DBusCallFlags.NONE,
                                -1,
                                None)
        (path,) = result.unpack ()

        logger.debug('%s: updating user image' % self.name)
        result = bus.call_sync ('org.freedesktop.Accounts',
                               path,
                               'org.freedesktop.Accounts.User',
                               'SetIconFile',
                               GLib.Variant ('(s)', (image_path,)),
                               GLib.VariantType.new ('()'),
                               Gio.DBusCallFlags.NONE,
                               -1,
                               None)

    def _normalize_image(self, pixbuf):
        """
        Normalize image used for GNOME3 user's profile picture
        (PNG format 96x96) and return the path to a temporary file.
        """
        pixbuf_resized = pixbuf.scale_simple(self.width, self.heigth,
                                             GdkPixbuf.InterpType.BILINEAR)
        image_path = os.path.join(GLib.get_tmp_dir(),
                                  '%s.%s' % (self.name, self.img_format))
        pixbuf_resized.savev(image_path, self.img_format, [], [])
        return image_path
