# -*- Mode: Python; coding: utf-8; indent-tabs-mode: nil; tab-width: 4 -*-
### BEGIN LICENSE
# Copyright (C) 2012 David Planella <david.planella@ubuntu.com>
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
### END LICENSE

TEXTDOMAIN = 'qreator'
LOCALEDIR = '/opt/extras.ubuntu.com/qreator/share/locale/'
import locale
from locale import gettext as _
locale.bindtextdomain(TEXTDOMAIN, LOCALEDIR)
locale.bind_textdomain_codeset(TEXTDOMAIN, 'UTF-8')
locale.textdomain(TEXTDOMAIN)

import cairo
import math
from gi.repository import Gtk, Gdk  # pylint: disable=E0611
from gi.repository import GtkChamplain, Clutter, Champlain
from gi.repository import NetworkManager, NMClient
import logging
logger = logging.getLogger('qreator')

from qreator_lib import Window

import Geoclue
from QRCode import QRCode as QRCode
from QRCode import QRCodeOutput as QRCodeOutput

COL_DESC = 0
COL_PIXBUF = 1
COL_ID = 2
COL_CALLBACK = 3

PAGE_NEW = 0
#PAGE_HISTORY = 1
#PAGE_SETTINGS = 2
PAGE_ABOUT = 1
PAGE_QR = 2

POS_PROVIDER = 'Ubuntu GeoIP'


# See qreator_lib.Window.py for more details about how this class works
class QreatorWindow(Window):
    __gtype_name__ = "QreatorWindow"

    def finish_initializing(self, builder):  # pylint: disable=E1002
        """Set up the main window"""
        super(QreatorWindow, self).finish_initializing(builder)

        # Code for other initialization actions should be added here.

        # Initialize the clipboard
        self.clipboard = Gtk.Clipboard.get(Gdk.SELECTION_CLIPBOARD)

        # Initialize the style for the main toolbar
        context = self.ui.toolbar1.get_style_context()
        context.add_class(Gtk.STYLE_CLASS_PRIMARY_TOOLBAR)

        # Initialize the Cairo surface that will contain the QR code
        self.surface = None

        # Initialize the icon view widget
        self.ui.iconview1.set_text_column(COL_DESC)
        self.ui.iconview1.set_pixbuf_column(COL_PIXBUF)

        # Hide the notebook tabs
        self.ui.notebook1.set_show_tabs(False)

        # Initialize combo boxes
        self.ui.comboboxtextSecurity.set_active(0)
        self.ui.comboboxtextProtocol.set_active(0)

        # Initialize placeholder text (we need to do that because due to
        # a Glade bug they are otherwise not marked as translatable)
        self.ui.entryURL.set_placeholder_text(_('[URL]'))
        self.ui.entrySSID.set_placeholder_text(
            _('[Network SSID - expand for autodetection]'))
        self.ui.entryPassword.set_placeholder_text(_('[Password]'))

        # Initialize about dialog
        about = Gtk.AboutDialog()
        about.set_program_name("Qreator")
        about.set_copyright("Copyright (c) 2012 David Planella\n" +
                            "http://about.me/david.planella")
        about.set_website("https://launchpad.net/qreator")
        #about.set_logo(GdkPixbuf.Pixbuf.new_from_file(
        #                    get_media_file("qreator-192.png")))

        about.set_version('12.05.4')
        about.set_authors(['David Planella <david.planella@ubuntu.com>',
                           'Michael Hall <mhall119@ubuntu.com>'])
        #about.set_comments(_('Create your own QR codes'))
        about.set_license(_('Distributed under the GPL v3 license.\n' +
                'http://www.opensource.org/licenses/gpl-3.0.html'))

        about.set_translator_credits(_("translator-credits"))

        box = builder.get_object("about_box")
        about.vbox.reparent(box)

        self.qr_code_placeholder = 'qreator'

    def on_entryText_changed(self, widget, data=None):
        self.update_qr_code(widget.get_text(widget.get_start_iter(),
                                            widget.get_end_iter(),
                                            False))

    def on_map_widget_button_press_event(self, actor, event, view):
        x, y = event.get_coords()
        lat, lon = view.x_to_longitude(x), view.y_to_latitude(y)

        self.ui.lat_entry.set_text(str(lat))
        self.ui.lon_entry.set_text(str(lon))

        self.update_qr_code('geo:{0},{1}'.format(str(lon), str(lat)))

        return True

    def on_toolbuttonNew_clicked(self, widget, data=None):
        '''Shows the home page'''
        self.ui.notebook1.set_current_page(PAGE_NEW)

    def on_toolbuttonHistory_clicked(self, widget, data=None):
        '''Shows the history page'''
        pass  # self.ui.notebook1.set_current_page(PAGE_HISTORY)

    def on_toolbuttonSettings_clicked(self, widget, data=None):
        '''Shows the settings page'''
        pass  # self.ui.notebook1.set_current_page(PAGE_SETTINGS)

    def on_toolbuttonAbout_clicked(self, widget, data=None):
        '''Shows the about page'''
        self.ui.notebook1.set_current_page(PAGE_ABOUT)

    def _hide_children(self, widget):
        for child in widget.get_children():
            child.hide()

    def on_qr_url_clicked(self):
        self._hide_children(self.ui.qr_input)
        self.ui.qr_url.show_all()

    def on_qr_wifi_clicked(self):
        self._hide_children(self.ui.qr_input)
        self.ui.qr_wifi.show_all()

        ssids = get_ssids()
        for ssid in ssids:
            self.ui.comboboxtextSSID.append_text(ssid)

    def on_qr_text_clicked(self):
        # Initialize multi-line text entry
        textbuffer = self.ui.entryText.get_buffer()
        textbuffer.connect("changed", self.on_entryText_changed, None)

        self._hide_children(self.ui.qr_input)
        self.ui.qr_text.show_all()

    def on_qr_location_clicked(self):
        # We're now loading the UI for the location QR type
        self._hide_children(self.ui.qr_input)
        self.ui.qr_location.show_all()

        map_widget = GtkChamplain.Embed()
        map_widget.set_hexpand(True)
        map_widget.set_vexpand(True)

        map_grid = self.ui.qr_location

        map_grid.set_size_request(-1, 250)
        map_grid.attach(map_widget, 0, 0, 1, 1)

        self.map_view = map_widget.get_view()
        self.map_view.set_reactive(True)
        map_widget.connect('button-release-event',
                           self.on_map_widget_button_press_event,
                           self.map_view)

        # Get the current location, center the map on it, and initialize
        # other map features
        latitude, longitude = get_current_location()
        self.map_view.center_on(latitude, longitude)
        if latitude == 0 and longitude == 0:
            # In case something went wrong in getting the current location
            self.map_view.set_zoom_level(1)
        else:
            self.map_view.set_zoom_level(15)
        self.map_view.set_kinetic_mode(True)

        scale = Champlain.Scale()
        scale.connect_view(self.map_view)
        self.map_view.bin_layout_add(scale, Clutter.BinAlignment.START,
                                     Clutter.BinAlignment.END)

        self.ui.qr_location.show_all()

    def on_iconview1_item_activated(self, widget, item):
        '''Loads the UI for the appropriate QR type'''

        model = widget.get_model()
        qr_callback = getattr(QreatorWindow, model[item][COL_CALLBACK])

        qr_callback(self)

        self.ui.notebook1.set_current_page(PAGE_QR)

    def get_pixbuf_from_drawing_area(self):
        window = self.ui.qr_drawingarea.get_window()

        src_x, src_y = self.get_centered_coordinates(self.ui.qr_drawingarea,
                                                     self.surface)
        image_height = self.surface.get_height()
        image_width = self.surface.get_width()

        return Gdk.pixbuf_get_from_window(window, src_x, src_y,
                                          image_width, image_height)

    def on_toolbuttonSave_clicked(self, widget, data=None):
        if not self.surface:
            return

        dialog = Gtk.FileChooserDialog(_("Please choose a file"), self,
            Gtk.FileChooserAction.SAVE,
            (Gtk.STOCK_CANCEL, Gtk.ResponseType.CANCEL,
             Gtk.STOCK_SAVE, Gtk.ResponseType.OK))

        filter_png = Gtk.FileFilter()
        filter_png.set_name(_("PNG images"))
        filter_png.add_mime_type("image/png")
        dialog.add_filter(filter_png)

        response = dialog.run()

        if response == Gtk.ResponseType.OK:
            # We cannot write directly from the surface, as the
            # Surface.write_to_png() method writes the image in the original
            # size returned by qrencode (i.e. non-scaled), and the
            # SurfacePattern does not have any methods to write to disk.
            # So we read the contents from the Gtk.DrawingArea, put them into
            # a Gdk.Pixbuf and use its 'savev' method to write to disk.

            pixbuf = self.get_pixbuf_from_drawing_area()

            pixbuf.savev(dialog.get_filename(), 'png', [], [])

        dialog.destroy()

    def on_toolbuttonCopy_clicked(self, widget, data=None):
        if not self.surface:
            return

        pixbuf = self.get_pixbuf_from_drawing_area()
        self.clipboard.set_image(pixbuf)

    def update_qr_code(self, text):
        self.qr_code_placeholder = text
        self.ui.qr_drawingarea.queue_draw()

    def on_entryURL_changed(self, widget, data=None):
        self.update_url_qr_code(www=widget.get_text())

    def on_comboboxtextProtocol_changed(self, widget, data=None):
        self.update_url_qr_code(protocol=widget.get_active_text())

    def update_url_qr_code(self, protocol=None, www=None):
        if not protocol:
            protocol = self.ui.comboboxtextProtocol.get_active_text()
        if not www:
            www = self.ui.entryURL.get_text()

        if not protocol or www == '':
            return

        self.update_qr_code(protocol + www)

    def update_wifi_qr_code(self, security=None, ssid=None, password=None):
        if not security:
            security = self.ui.comboboxtextSecurity.get_active_text()
        if not ssid:
            ssid = self.ui.entrySSID.get_text()
        if not password:
            password = self.ui.entryPassword.get_text()

        if not security or ssid == '' or password == '':
            return

        wifi_qr_code = 'WIFI:T:{0};S:{1};P:{2};;'.format(security, ssid,
                                                         password)
        self.update_qr_code(wifi_qr_code)

    def on_comboboxtextSecurity_changed(self, widget, data=None):
        self.update_wifi_qr_code(security=widget.get_active_text())

    def on_entrySSID_changed(self, widget, data=None):
        self.update_wifi_qr_code(ssid=widget.get_text())

    def on_comboboxtextSSID_changed(self, widget, data=None):
        self.update_wifi_qr_code(ssid=widget.get_active_text())

    def on_entryPassword_changed(self, widget, data=None):
        self.update_wifi_qr_code(password=widget.get_text())

    def on_checkbutton1_clicked(self, widget):
        if widget.get_active():
            self.ui.entryPassword.set_visibility(True)
        else:
            self.ui.entryPassword.set_visibility(False)

    def get_centered_coordinates(self, drawing_area, surface):

        drawing_area_height = drawing_area.get_allocated_height()
        drawing_area_width = drawing_area.get_allocated_width()
        image_height = surface.get_height()
        image_width = surface.get_width()

        return (drawing_area_width / 2 - image_width / 2,
                drawing_area_height / 2 - image_height / 2)

    def on_qr_drawingarea_draw(self, widget, ctx, data=None):
        text = self.qr_code_placeholder
        if text == '':
            return

        ## Fill the background

        # Create a rounded rectanble
        x = 0.0
        y = 0.0
        width = widget.get_allocated_width()
        height = widget.get_allocated_height()
        aspect = 1.0
        corner_radius = height / 50.0

        radius = corner_radius / aspect
        degrees = math.pi / 180.0

        ctx.new_sub_path()
        ctx.arc(x + width - radius, y + radius, radius, -90 * degrees,
                0 * degrees)
        ctx.arc(x + width - radius, y + height - radius, radius, 0 * degrees,
                90 * degrees)
        ctx.arc(x + radius, y + height - radius, radius, 90 * degrees,
                180 * degrees)
        ctx.arc(x + radius, y + radius, radius, 180 * degrees, 270 * degrees)
        ctx.close_path()

        # Fill the rounded rectangle with a linear gradient
        lg = cairo.LinearGradient(0.0, 0.0, 500.0, 500.0)
        lg.add_color_stop_rgba(0, 1, 1, 1, 1)
        lg.add_color_stop_rgba(1, 0.678, 0.678, 0.678, 1)

        ctx.set_source(lg)
        ctx.fill()

        ## Create the QR code
        qr_code = QRCode()
        self.surface = qr_code.encode(text, QRCodeOutput.CAIRO_SURFACE)

        # Center the image in the drawing area
        centered_width, centered_height = \
            self.get_centered_coordinates(widget, self.surface)
        ctx.translate(centered_width, centered_height)

        # Set the surface as the context's source
        ctx.set_source_surface(self.surface)

        # Render the image
        ctx.paint()


def get_current_location():
    '''Gets the current location from geolocation via IP (only method
       currently supported)'''

    location = Geoclue.DiscoverLocation()
    location.init()
    location.set_position_provider(POS_PROVIDER)
    position = location.get_location_info()

    return position['latitude'], position['longitude']


def get_ssids():
    nmc = NMClient.Client.new()
    devs = nmc.get_devices()

    ssids = []
    for dev in devs:
        if dev.get_device_type() == NetworkManager.DeviceType.WIFI:
            for ap in dev.get_access_points():
                ssids.append(ap.get_ssid())

    return ssids
