/*
 *      smsdb.c
 *      
 *      Copyright 2012 Alex <alex@linuxonly.ru>
 *      
 *      This program is free software: you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation; either version 3 of the License, or
 *      (at your option) any later version.
 *      
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *      
 *      You should have received a copy of the GNU General Public License
 *      along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdlib.h>
#include <glib.h>
#include <glib/gprintf.h>
#include <string.h>
#include <gdbm.h>

//#include "dbus-api.h"
#include "encoding.h"
#include "smsdb.h"

#define SMSDB_SMS_MESSAGE_XML "<sms>\n\t<number>%s</number>\n\t<time>%lu</time>\n\t<text>%s</text>\n</sms>\n\n"

enum _gmm_smsdb_xml_elements {
	SMSDB_XML_PARAM_NUMBER = 0,
	SMSDB_XML_PARAM_TIME,
	SMSDB_XML_PARAM_TEXT,
	SMSDB_XML_PARAM_NULL
};

static gint gmm_smsdb_xml_parameter = SMSDB_XML_PARAM_NULL;

static gint gmm_smsdb_sms_message_sort_compare(gconstpointer a, gconstpointer b);
static void gmm_smsdb_free_sms_list_foreach(gpointer data, gpointer user_data);
static sms_full_message_t gmm_smsdb_xml_parse(gchar *xml, gsize size);
static void gmm_smsdb_xml_get_element(GMarkupParseContext *context, const gchar *element, const gchar **attr_names, const gchar **attr_values, gpointer data, GError **error);
static void gmm_smsdb_xml_get_value(GMarkupParseContext *context, const gchar *text, gsize size, gpointer data, GError **error);
static void gmm_smsdb_xml_end_element(GMarkupParseContext *context, const gchar *element, gpointer data, GError **error);


smsdb_t gmm_smsdb_open(gchar *appname, gchar *deviceid)
{
	smsdb_t smsdb;
	const gchar *homepath; 
	gchar filename[64];
	gchar *confpath;
	
	if ((appname == NULL) || (deviceid == NULL)) return NULL;
	
	homepath = g_get_home_dir();
	
	if (homepath == NULL) return NULL;
	
	confpath = g_build_filename(homepath, ".config", appname, NULL);
	
	if (g_mkdir_with_parents(confpath, 0755) != 0) {
		g_warning("Cant create program settings directory");
		g_free(confpath);
		return NULL;
	}
	
	g_free(confpath);
	
	smsdb = g_new(struct _smsdb, 1);
	
	g_snprintf(filename, sizeof(filename), "sms-%s.gdbm", deviceid);
	
	smsdb->filepath = g_build_filename(homepath, ".config", appname, filename, NULL);
	
	return smsdb;
}

gboolean gmm_smsdb_close(smsdb_t smsdb)
{
	if (smsdb == NULL) return FALSE;
	
	if (smsdb->filepath != NULL) {
		g_free(smsdb->filepath);
	}
	
	g_free(smsdb);
	
	return TRUE;
}

gboolean gmm_smsdb_add_sms(smsdb_t smsdb, sms_full_message_t sms)
{
	GDBM_FILE db;
	gchar smsid[64];
	gulong idvalue;
	gint idlen;
	datum key, data;
	gchar *smsxml;
		
	if ((smsdb == NULL) || (sms == NULL)) return FALSE;
	if (smsdb->filepath == NULL) return FALSE;
	
	db = gdbm_open(smsdb->filepath, 0, GDBM_WRCREAT, 0755, 0);
	
	if (db == NULL) return FALSE;
	
	do {
		idvalue = (gulong)random();
		memset(smsid, 0, sizeof(smsid));
		idlen = snprintf(smsid, sizeof(smsid), "%lu", idvalue);
		key.dptr = (gchar *)smsid;
		key.dsize = idlen;
	} while (gdbm_exists(db, key));
	
	sms->dbid = idvalue;
	
	smsxml = g_strdup_printf(SMSDB_SMS_MESSAGE_XML, sms->number, sms->timestamp, sms->message->str);
	
	data.dptr = smsxml;
	data.dsize = strlen(smsxml);
	
	if (gdbm_store(db, key, data, GDBM_REPLACE) == -1) {
		g_warning("Unable to write to database");
		gdbm_close(db);
		g_free(smsxml);
		return FALSE;
	}
	
	gdbm_sync(db);
	gdbm_close(db);
	
	g_free(smsxml);
	
	return TRUE;
}

static gint gmm_smsdb_sms_message_sort_compare(gconstpointer a, gconstpointer b)
{
	sms_full_message_t sms1, sms2;
		
	sms1 = (sms_full_message_t)a;
	sms2 = (sms_full_message_t)b;
	
	if (sms1->timestamp < sms2->timestamp) {
		return -1;
	} else if (sms1->timestamp > sms2->timestamp) {
		return 1;
	} else {
		return 0;
	}
}

GSList *gmm_smsdb_read_sms_list(smsdb_t smsdb)
{
	GDBM_FILE db;
	GSList *list;
	sms_full_message_t sms;
	datum key, data;
	gchar smsid[64];
	
	if (smsdb == NULL) return NULL;
	if (smsdb->filepath == NULL) return NULL;
	
	db = gdbm_open(smsdb->filepath, 0, GDBM_READER, 0755, 0);
	
	if (db == NULL) return NULL;
	
	list = NULL;
	
	key = gdbm_firstkey(db);
	
	if (key.dptr != NULL) {
		do {
			data = gdbm_fetch(db, key);
			if (data.dptr != NULL) {
				sms = gmm_smsdb_xml_parse(data.dptr, data.dsize);
				if (sms != NULL) {
					memset(smsid, 0, sizeof(smsid));
					strncpy(smsid, key.dptr, key.dsize);
					sms->dbid = strtoul(smsid, NULL, 0);
					list = g_slist_prepend(list, sms);
				}
			}
			key = gdbm_nextkey(db, key);
		} while (key.dptr != NULL);
	}
	
	gdbm_close(db);
	
	if (list != NULL) {
		list = g_slist_sort(list, gmm_smsdb_sms_message_sort_compare);
	} 
	
	return list;
}

static void gmm_smsdb_free_sms_list_foreach(gpointer data, gpointer user_data)
{
	sms_full_message_t sms;
	
	if (data != NULL) return;
	
	sms = (sms_full_message_t)data;
	
	if (sms->number != NULL) {
		g_free(sms->number);
		sms->number = NULL;
	}
	
	if (sms->idents != NULL) {
		g_array_free(sms->idents, TRUE);
		sms->idents = NULL;
	}
	
	if (sms->message != NULL) {
		g_string_free(sms->message, TRUE);
		sms->message = NULL;
	}
}

void gmm_smsdb_free_sms_list(GSList *smslist)
{
	if (smslist == NULL) return;
	
	g_slist_foreach(smslist, gmm_smsdb_free_sms_list_foreach, NULL);
	g_slist_free(smslist);
}

sms_full_message_t gmm_smsdb_read_sms_message(smsdb_t smsdb, gulong idvalue)
{
	GDBM_FILE db;
	gchar smsid[64];
	gint idlen;
	datum key, data;
	sms_full_message_t sms;
		
	if (smsdb == NULL) return NULL;
	if (smsdb->filepath == NULL) return NULL;
	
	db = gdbm_open(smsdb->filepath, 0, GDBM_READER, 0755, 0);
	
	if (db == NULL) return NULL;
	
	sms = NULL;
	
	memset(smsid, 0, sizeof(smsid));
	idlen = snprintf(smsid, sizeof(smsid), "%lu", idvalue);
	key.dptr = (gchar *)smsid;
	key.dsize = idlen;
	
	if (gdbm_exists(db, key)) {
		data = gdbm_fetch(db, key);
		if (data.dptr != NULL) {
			sms = gmm_smsdb_xml_parse(data.dptr, data.dsize);
			sms->dbid = idvalue;
		}
	}
	
	gdbm_close(db);
	
	return sms;
}

gboolean gmm_smsdb_remove_sms_message(smsdb_t smsdb, gulong idvalue)
{
	GDBM_FILE db;
	gchar smsid[64];
	gint idlen;
	datum key, data;
	
	if (smsdb == NULL) return FALSE;
	if (smsdb->filepath == NULL) return FALSE;
	
	db = gdbm_open(smsdb->filepath, 0, GDBM_WRCREAT, 0755, 0);
	
	if (db == NULL) return FALSE;
	
	memset(smsid, 0, sizeof(smsid));
	idlen = g_snprintf(smsid, sizeof(smsid), "%lu", idvalue);
	key.dptr = (gchar *)smsid;
	key.dsize = idlen;
	
	if (gdbm_exists(db, key)) {
		if (gdbm_delete(db, key) == 0) {
			gdbm_sync(db);
			gdbm_close(db);
			return TRUE;
		}
	}
	
	gdbm_close(db);
	
	return FALSE;
}

static sms_full_message_t gmm_smsdb_xml_parse(gchar *xml, gsize size)
{
	sms_full_message_t sms;
	GMarkupParser mp;
	GMarkupParseContext *mpc;
	GError *error = NULL;
	
	sms = g_new(struct _sms_full_message, 1);
	
	mp.start_element = gmm_smsdb_xml_get_element;
	mp.end_element = gmm_smsdb_xml_end_element;
	mp.text = gmm_smsdb_xml_get_value;
	mp.passthrough = NULL;
	mp.error = NULL;
	
	mpc = g_markup_parse_context_new(&mp, 0, (gpointer)sms, NULL);
	g_markup_parse_context_parse(mpc, xml, size, &error);
	if (error != NULL) {
		//g_warning(error->message);
		g_error_free(error);
		g_markup_parse_context_free(mpc);
		return NULL;
	}
	g_markup_parse_context_free(mpc);
	
	return sms;
}

static void gmm_smsdb_xml_get_element(GMarkupParseContext *context, const gchar *element, const gchar **attr_names, const gchar **attr_values, gpointer data, GError **error)
{
	if (g_str_equal(element, "number")) gmm_smsdb_xml_parameter = SMSDB_XML_PARAM_NUMBER;
	else if (g_str_equal(element, "time")) gmm_smsdb_xml_parameter = SMSDB_XML_PARAM_TIME;
	else if (g_str_equal(element, "text")) gmm_smsdb_xml_parameter = SMSDB_XML_PARAM_TEXT;
}

static void gmm_smsdb_xml_get_value(GMarkupParseContext *context, const gchar *text, gsize size, gpointer data, GError **error)
{
	sms_full_message_t sms;
	gchar *numstr;
	gsize numlen;
	
	sms = (sms_full_message_t)data;
	
	if (gmm_smsdb_xml_parameter == SMSDB_XML_PARAM_NULL) return;
	
	switch (gmm_smsdb_xml_parameter) {
		case SMSDB_XML_PARAM_NUMBER:
			numstr = g_markup_escape_text(text, size);
			sms->number = bcd_to_utf8_ascii_part(numstr, size, &numlen);
			if (sms->number == NULL) {
				sms->number = numstr;
			} else {
				g_free(numstr); 
			}
			break;
		case SMSDB_XML_PARAM_TIME:
			sms->timestamp = (time_t)atol(text);
			break;
		case SMSDB_XML_PARAM_TEXT:
			sms->message = g_string_new(g_markup_escape_text(text, size));
			break;
		default:
			break;
	}
}

static void gmm_smsdb_xml_end_element(GMarkupParseContext *context, const gchar *element, gpointer data, GError **error)
{
	if (!g_str_equal(element, "sms")) {
		gmm_smsdb_xml_parameter = SMSDB_XML_PARAM_NULL;
	}
}
