/*
 *      main.c
 *      
 *      Copyright 2012 Alex <alex@linuxonly.ru>
 *      
 *      This program is free software: you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation; either version 3 of the License, or
 *      (at your option) any later version.
 *      
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *      
 *      You should have received a copy of the GNU General Public License
 *      along with this program. If not, see <http://www.gnu.org/licenses/>.
 */


#include <gtk/gtk.h>
#include <locale.h>
#include <libintl.h>
#include <stdlib.h>
#include <glib/gi18n.h>

#include <libnotify/notify.h>

#ifndef _NO_LIBINDICATE
	#include <libindicate/server.h>
	#include <libindicate/indicator.h>
#endif

#include "settings.h"
#include "dbus-api.h"
#include "smsdb.h"
#include "resources.h"


enum _mainwindow_pages {
	MAINWINDOW_PAGE_DEVICES = 0,
	MAINWINDOW_PAGE_SMS,
	MAINWINDOW_PAGE_USSD,
	MAINWINDOW_PAGE_INFO,
	MAINWINDOW_PAGE_SCAN,
	MAINWINDOW_PAGE_TRAFFIC
};

enum _devlist_columns {
	DEVLIST_ENABLED = 0,
	DEVLIST_DESCRIPTION,
	DEVLIST_ID,
	DEVLIST_IDENTIFIER,
	DEVLIST_COLUMNS
};

enum _smslist_columns {
	SMSLIST_SMS = 0,
	SMSLIST_ID,
	SMSLIST_COLUMNS
};

enum _scanlist_columns {
	SCANLIST_OPERATOR = 0,
	SCANLIST_COLUMNS
};

enum _trafficlist_columns {
	TRAFFICLIST_PARAMETER = 0,
	TRAFFICLIST_VALUE,
	TRAFFICLIST_ID,
	TRAFFICLIST_COLUMNS
};

enum _trafficlist_id {
	TRAFFICLIST_ID_RXDATA = 0,
	TRAFFICLIST_ID_TXDATA,
	TRAFFICLIST_ID_RXSPEED,
	TRAFFICLIST_ID_TXSPEED,
	TRAFFICLIST_ID_TIME,
	TRAFFICLIST_ID_DATALIMIT,
	TRAFFICLIST_ID_TIMELIMIT
};

enum _indicator_id {
	INDICATOR_ID_SERVER = 0,
	INDICATOR_ID_NEW,
	INDICATOR_ID_UNREAD
};

enum _new_sms_validation {
	NEW_SMS_VALID = 0x00,
	NEW_SMS_WRONG_NUMBER = 0x01,
	NEW_SMS_WRONG_TEXT = 0x02
};

enum _traffic_limits_validation {
	LIMIT_TRAFFIC = 0x00,
	LIMIT_TIME = 0x01,
	LIMIT_BOTH = 0x02
};

struct _selectdetails {
	guint id;
	gchar *identifier;
	gboolean selected;
	gboolean selectfirst;
};

typedef struct _selectdetails *selectdetails_t;

static struct _mainwindow {
	GtkApplication *application;
	
	GtkWidget *window;
	GtkWidget *toolbar;
	GtkWidget *statusbar;
	guint sbcontext;
	
	GtkWidget *aboutdialog;
	GtkWidget *questiondialog;
	GtkWidget *errordialog;
	GtkWidget *progressdialog;
	GtkWidget *progressbar;
	
	GtkWidget *nodevbar;
	
	GtkWidget *newsmsdialog;
	GtkWidget *smsnumberentry;
	GtkWidget *smstextview;
	GtkWidget *sendsmsbutton;
		
	GtkWidget *devbutton;
	GtkWidget *smsbutton;
	GtkWidget *ussdbutton;
	GtkWidget *infobutton;
	GtkWidget *scanbutton;
	GtkWidget *trafficbutton;
	
	GtkWidget *notebook;
	GtkWidget *devlist;
	
	GtkWidget *smslist;
	GtkWidget *smstext;
	GtkWidget *newsmsbutton;
	GtkWidget *removesmsbutton;
	
	GtkWidget *devicevlabel;
	GtkWidget *operatorvlabel;
	GtkWidget *modevlabel;
	GtkWidget *imeivlabel;
	GtkWidget *imsivlabel;
	GtkWidget *signallevelvlabel;
	
	GtkWidget *ussdentry;
	GtkWidget *ussdencoding;
	GtkWidget *ussdsend;
	GtkWidget *ussdtext;
	
	GtkWidget *scanlist;
	GtkWidget *startscanbutton;
	
	GtkWidget *trafficparamslist;
	GtkWidget *trafficdrawingarea;
	
	GtkWidget *trafficlimitsdialog;
	
	GtkWidget *trafficlimitcheckbutton;
	GtkWidget *trafficamount;
	GtkWidget *trafficunits;
	GtkWidget *trafficmessage;
	GtkWidget *trafficaction;
	
	GtkWidget *timelimitcheckbutton;
	GtkWidget *timeamount;
	GtkWidget *timeunits;
	GtkWidget *timemessage;
	GtkWidget *timeaction;
	
	guint progresstimeout;
	
	#ifdef _NO_LIBINDICATE
		GtkStatusIcon *statusicon;
		GtkWidget *traymenu;
		GtkWidget *showwin_tm, *sep1_tm, *newsms_tm, *sep2_tm, *quit_tm;
		gulong traysigid;
	#else
		IndicateServer *indserver;
		IndicateIndicator *indnew;
		IndicateIndicator *indunread;
	#endif
	
	guint unreadsms;
} mainwindow;

static dbus_api_t dbus;
static settings_t settings;
static cli_options_t clioptions;
static traffic_limits_t limits;

static gboolean gmm_question_dialog(gchar *caption, gchar *text);
static gboolean gmm_error_dialog(gchar *caption, gchar *text);
static gboolean gmm_progress_dialog_change_progress(gpointer user_data);
static void gmm_progress_dialog_start(void);
static void gmm_progress_dialog_stop(void);
static void gmm_new_sms_dialog_number_changed_signal(GtkEditable *editable, gpointer data);
static gboolean gmm_new_sms_dialog(gchar *number, gchar *text);
static void gmm_clear_device_data(void);
static void gmm_lock_sms_controls(gboolean lock);
static void gmm_lock_ussd_controls(gboolean lock);
static void gmm_lock_scan_controls(gboolean lock);
static gboolean gmm_remove_sms_foreach(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data);
static void gmm_init_scan_list(GtkWidget *list);
static void gmm_device_get_scan_result_foreach(gpointer data, gpointer user_data);
static void gmm_device_get_scan_result(device_t device);
static void gmm_ussd_send_start(gchar *request, gboolean ucs2);
static void gmm_ussd_send_end(device_t device);
static void gmm_init_sms_list(GtkWidget *list);
static void gmm_on_sms_list_cursor_changed(GtkTreeView *tree_view, gpointer data);
static void gmm_add_sms(sms_full_message_t sms, GtkTreeModel *model);
static void gmm_get_sms_foreach(gpointer data, gpointer user_data);
static void gmm_device_get_sms(void);
static void gmm_device_update_info(device_t device);
static void gmm_device_open(guint id);
static gboolean gmm_devlist_item_unselect_foreach(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data);static void on_devlist_item_selected(GtkCellRendererToggle *cell_renderer, gchar *path, gpointer data);
static void gmm_init_device_list(GtkWidget *list);
static void gmm_add_device(device_t device, GtkTreeModel *model);
static void gmm_add_devices_foreach(gpointer data, gpointer user_data);
static gboolean gmm_remove_device_foreach(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data);
static void gmm_remove_device(guint id);
static gboolean gmm_select_device_foreach(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data);
static void gmm_select_device(gchar *identifier);
static void gmm_buttons_lock(gboolean lock);
static gboolean gmm_add_sms_from_thread(gpointer data);
static gboolean gmm_update_statusbar_from_thread(gpointer data);
static gboolean gmm_update_traffic_page_from_thread_foreach(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data);
static gboolean gmm_update_traffic_page_from_thread(gpointer data);
static gboolean gmm_show_limit_message_from_thread(gpointer data);
static gchar *gmm_format_speed(gfloat speed, gchar *buffer, gsize bufsize, gboolean small);
static gchar *gmm_format_time_number(guchar number, gchar *buffer, gsize bufsize);
static gchar *gmm_format_time(guint64 seconds, gchar *buffer, gsize bufsize, gboolean small);
static gchar *gmm_format_bytes(guint64 bytes, gchar *buffer, gsize bufsize, gboolean small);
#ifdef _NO_LIBINDICATE
static void gmm_tray_action_signal(GtkStatusIcon *status_icon, gpointer data);
static gint gmm_tray_window_show_signal(GtkWidget *widget, GdkEvent *event, gpointer data);
static void gmm_tray_new_sms_signal(GtkMenuItem *menuitem, gpointer data);
static void gmm_tray_popup_signal(GtkStatusIcon *status_icon, guint button, guint activate_time, gpointer data);
#else
static void gmm_indicator_server_show_signal(IndicateServer *arg0, guint arg1, gpointer user_data);
#endif
static void gmm_load_settings(settings_t settings, traffic_limits_t limits);
static void gmm_application_activate_signal(GtkApplication *application);
static void gmm_application_shutdown_signal(GtkApplication *application, gpointer user_data);


static gboolean gmm_question_dialog(gchar *caption, gchar *text)
{
	gint response;
	
	if ((caption == NULL) || (text == NULL)) return FALSE;
	
	gtk_message_dialog_set_markup(GTK_MESSAGE_DIALOG(mainwindow.questiondialog), caption);
	gtk_message_dialog_format_secondary_markup(GTK_MESSAGE_DIALOG(mainwindow.questiondialog), "%s", text);
	
	response = gtk_dialog_run(GTK_DIALOG(mainwindow.questiondialog));
	
	gtk_widget_hide(mainwindow.questiondialog);
	
	return (response == GTK_RESPONSE_YES);
}

static gboolean gmm_error_dialog(gchar *caption, gchar *text)
{
	gint response;
	
	if ((caption == NULL) || (text == NULL)) return FALSE;
	
	gtk_message_dialog_set_markup(GTK_MESSAGE_DIALOG(mainwindow.errordialog), caption);
	gtk_message_dialog_format_secondary_markup(GTK_MESSAGE_DIALOG(mainwindow.errordialog), "%s", text);
	
	response = gtk_dialog_run(GTK_DIALOG(mainwindow.errordialog));
	
	gtk_widget_hide(mainwindow.errordialog);
	
	return (response == GTK_RESPONSE_CLOSE);
}

static gboolean gmm_progress_dialog_change_progress(gpointer user_data)
{
	GtkProgressBar *pb;
	
	pb = (GtkProgressBar *)user_data;
	
	if (pb != NULL) gtk_progress_bar_pulse(pb);
	
	return TRUE;
}

static void gmm_progress_dialog_start(void)
{
	mainwindow.progresstimeout = g_timeout_add(100, gmm_progress_dialog_change_progress, mainwindow.progressbar);
	
	gtk_dialog_run(GTK_DIALOG(mainwindow.progressdialog));
}

static void gmm_progress_dialog_stop(void)
{
	g_source_remove(mainwindow.progresstimeout);
	
	mainwindow.progresstimeout = 0;
	
	gtk_widget_hide(mainwindow.progressdialog);
}

static void gmm_new_sms_dialog_number_changed_signal(GtkEditable *editable, gpointer data)
{
	const gchar *number;
	GtkTextBuffer *buffer;
	gboolean newnumvalid;
	gint bufferchars;
	gint *smsvalidflags;
	gint newsmsvalidflags;
	
	number = gtk_entry_get_text(GTK_ENTRY(mainwindow.smsnumberentry));
	buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(mainwindow.smstextview));
	smsvalidflags = (gint *)data;
	
	newnumvalid = gmm_dbus_api_validate_sms_number((gchar *)number);
	if (buffer != NULL) {
		bufferchars = gtk_text_buffer_get_char_count(buffer);
	} else {
		bufferchars = 0;
	}
	
	newsmsvalidflags = NEW_SMS_VALID;
	if (!newnumvalid) newsmsvalidflags &= NEW_SMS_WRONG_NUMBER;
	if (bufferchars == 0) newsmsvalidflags &= NEW_SMS_WRONG_TEXT;
	
	if (((!newnumvalid) || (bufferchars == 0)) && ((*smsvalidflags == NEW_SMS_VALID) || (*smsvalidflags != newsmsvalidflags))) {
		gtk_entry_set_icon_from_stock(GTK_ENTRY(mainwindow.smsnumberentry), GTK_ENTRY_ICON_SECONDARY, GTK_STOCK_CAPS_LOCK_WARNING);
		if (!newnumvalid) {
			gtk_entry_set_icon_tooltip_markup(GTK_ENTRY(mainwindow.smsnumberentry), GTK_ENTRY_ICON_SECONDARY, _("<b>SMS number is not valid</b>\n<small>Only numbers from 2 to 20 digits without\nletters and symbols can be used</small>"));
		} else if (bufferchars == 0) {
			gtk_entry_set_icon_tooltip_markup(GTK_ENTRY(mainwindow.smsnumberentry), GTK_ENTRY_ICON_SECONDARY, _("<b>SMS text is not valid</b>\n<small>Please write some text to send</small>"));
		}
		gtk_widget_set_sensitive(mainwindow.sendsmsbutton, FALSE);
		*smsvalidflags = newsmsvalidflags;
	} else if ((newnumvalid) && (bufferchars > 0)) {
		gtk_entry_set_icon_from_stock(GTK_ENTRY(mainwindow.smsnumberentry), GTK_ENTRY_ICON_SECONDARY, NULL);
		gtk_entry_set_icon_tooltip_markup(GTK_ENTRY(mainwindow.smsnumberentry), GTK_ENTRY_ICON_SECONDARY, NULL);
		gtk_widget_set_sensitive(mainwindow.sendsmsbutton, TRUE);
		*smsvalidflags = newsmsvalidflags;
	}
}

static gboolean gmm_new_sms_dialog(gchar *number, gchar *text)
{
	GtkTextBuffer *buffer;
	gint response;
	gulong editnumsignal, edittextsignal;
	gint smsvalidflags;
	
	smsvalidflags = NEW_SMS_VALID;
	
	editnumsignal = g_signal_connect(G_OBJECT(mainwindow.smsnumberentry), "changed", G_CALLBACK(gmm_new_sms_dialog_number_changed_signal), &smsvalidflags);
	
	if (number != NULL) {
		gtk_entry_set_text(GTK_ENTRY(mainwindow.smsnumberentry), number);
		g_signal_emit_by_name(G_OBJECT(mainwindow.smsnumberentry), "changed");
	}
	
	buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(mainwindow.smstextview));
	if (buffer != NULL) {
		edittextsignal = g_signal_connect(G_OBJECT(buffer), "changed", G_CALLBACK(gmm_new_sms_dialog_number_changed_signal), &smsvalidflags);
		if (text != NULL) {
			gtk_text_buffer_set_text(buffer, text, -1);
			g_signal_emit_by_name(G_OBJECT(buffer), "changed");
		}
	}
	
	response = gtk_dialog_run(GTK_DIALOG(mainwindow.newsmsdialog));
	
	g_signal_handler_disconnect(G_OBJECT(mainwindow.smsnumberentry), editnumsignal);
	if (buffer != NULL) {
		g_signal_handler_disconnect(G_OBJECT(buffer), edittextsignal);
	}
	
	gtk_widget_hide(mainwindow.newsmsdialog);
	
	return (response > 0);
}

static void gmm_clear_device_data(void)
{
	GtkTreeModel *model;
	GtkTextBuffer *buffer;
	GtkTextIter siter, eiter;
	
	model = gtk_tree_view_get_model(GTK_TREE_VIEW(mainwindow.smslist));
	if (model != NULL) {
		gtk_list_store_clear(GTK_LIST_STORE(model));
	}
	
	buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(mainwindow.smstext));
	if (buffer != NULL) {
		gtk_text_buffer_get_bounds(buffer, &siter, &eiter);
		gtk_text_buffer_delete(buffer, &siter, &eiter);
	}
	
	buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(mainwindow.ussdtext));
	if (buffer != NULL) {
		gtk_text_buffer_get_bounds(buffer, &siter, &eiter);
		gtk_text_buffer_delete(buffer, &siter, &eiter);
	}
	
	gtk_label_set_text(GTK_LABEL(mainwindow.devicevlabel), "");
	gtk_label_set_text(GTK_LABEL(mainwindow.operatorvlabel), "");
	gtk_label_set_text(GTK_LABEL(mainwindow.modevlabel), "");
	gtk_label_set_text(GTK_LABEL(mainwindow.imeivlabel), "");
	gtk_label_set_text(GTK_LABEL(mainwindow.imsivlabel), "");
	gtk_label_set_text(GTK_LABEL(mainwindow.signallevelvlabel), "");
	
	model = gtk_tree_view_get_model(GTK_TREE_VIEW(mainwindow.scanlist));
	if (model != NULL) {
		gtk_list_store_clear(GTK_LIST_STORE(model));
	}
} 

static void gmm_lock_sms_controls(gboolean lock)
{
	gtk_widget_set_sensitive(mainwindow.newsmsbutton, !lock);
	gtk_widget_set_sensitive(mainwindow.removesmsbutton, !lock);
}

static void gmm_lock_ussd_controls(gboolean lock)
{
	gtk_widget_set_sensitive(mainwindow.ussdentry, !lock);
	gtk_widget_set_sensitive(mainwindow.ussdencoding, !lock);
	gtk_widget_set_sensitive(mainwindow.ussdsend, !lock);
	
	if (!lock) {
		g_signal_emit_by_name(G_OBJECT(mainwindow.ussdentry), "changed", NULL);
	}
}

static void gmm_lock_scan_controls(gboolean lock)
{
	gtk_widget_set_sensitive(mainwindow.startscanbutton, !lock);
}

void on_window_destroy(GObject *object, gpointer user_data)
{
	g_application_quit(G_APPLICATION(mainwindow.application));
}

void on_exitmenuitem_activate(GObject *object, gpointer user_data)
{
	g_application_quit(G_APPLICATION(mainwindow.application));
}

void gmm_on_devbutton_toggled(GObject *object, gpointer user_data)
{
	gtk_notebook_set_current_page(GTK_NOTEBOOK(mainwindow.notebook), MAINWINDOW_PAGE_DEVICES);
}

void gmm_on_smsbutton_toggled(GObject *object, gpointer user_data)
{
	gboolean enabled;
	
	if (gtk_toggle_tool_button_get_active(GTK_TOGGLE_TOOL_BUTTON(mainwindow.smsbutton))) {
		if (!gmm_dbus_api_device_blocked(dbus)) {
			enabled = gmm_dbus_api_device_enabled(dbus);
			if (!enabled) {
				if (gmm_question_dialog(_("<b>Enable modem</b>"), _("Modem must be enabled to read SMS. Enable modem?"))) {
					if (gmm_dbus_api_set_device_enabled_async(dbus, TRUE)) {
						gmm_progress_dialog_start();
					} else {
						gmm_error_dialog(_("<b>Error enabling device</b>"), gmm_dbus_api_get_error_message(dbus));
					}
				} else {
					gmm_lock_sms_controls(TRUE);
				}
			} else {
				gmm_lock_sms_controls(FALSE);
			}
		} else {
			gmm_lock_sms_controls(TRUE);
		}
		
		gtk_notebook_set_current_page(GTK_NOTEBOOK(mainwindow.notebook), MAINWINDOW_PAGE_SMS);
		
		if (mainwindow.unreadsms > 0) {
			mainwindow.unreadsms = 0;
			#ifdef _NO_LIBINDICATE
				gtk_status_icon_set_tooltip_text(mainwindow.statusicon, _("No unread messages"));
			#else
				indicate_indicator_set_property(mainwindow.indunread, "draw_attention", "false");
				indicate_indicator_set_property(mainwindow.indunread, "count", "");
			#endif
		}
	}
}

void gmm_on_ussdbutton_toggled(GObject *object, gpointer user_data)
{
	gboolean enabled;
	
	if (gtk_toggle_tool_button_get_active(GTK_TOGGLE_TOOL_BUTTON(mainwindow.ussdbutton))) {
		if (!gmm_dbus_api_device_blocked(dbus)) {
			enabled = gmm_dbus_api_device_enabled(dbus);
			if (!enabled) {
				if (gmm_question_dialog(_("<b>Enable modem</b>"), _("Modem must be enabled to send USSD. Enable modem?"))) {
					if (gmm_dbus_api_set_device_enabled_async(dbus, TRUE)) {
						gmm_progress_dialog_start();
					} else {
						gmm_error_dialog(_("<b>Error enabling device</b>"), gmm_dbus_api_get_error_message(dbus));
					}
				} else {
					gmm_lock_ussd_controls(TRUE);
				}
			} else {
				gmm_lock_ussd_controls(FALSE);
			}
		} else {
			gmm_lock_ussd_controls(TRUE);
		}
		gtk_notebook_set_current_page(GTK_NOTEBOOK(mainwindow.notebook), MAINWINDOW_PAGE_USSD);
	}
}

void gmm_on_infobutton_toggled(GObject *object, gpointer user_data)
{
	if (gtk_toggle_tool_button_get_active(GTK_TOGGLE_TOOL_BUTTON(mainwindow.infobutton))) {
		if (gmm_dbus_api_update_device_info(dbus)) {
			gmm_device_update_info(dbus->curdevice);
		}
		gtk_notebook_set_current_page(GTK_NOTEBOOK(mainwindow.notebook), MAINWINDOW_PAGE_INFO);
	}
}

void gmm_on_scanbutton_toggled(GObject *object, gpointer user_data)
{
	gboolean enabled;
	
	if (gtk_toggle_tool_button_get_active(GTK_TOGGLE_TOOL_BUTTON(mainwindow.scanbutton))) {
		if (gmm_dbus_api_device_connected(dbus)) {
			gmm_error_dialog(_("<b>Wrong device state</b>"), _("Modem is connected now. Please disconnect to scan."));
			gmm_lock_scan_controls(TRUE);
		} else {
			if (!gmm_dbus_api_device_blocked(dbus)) {
				enabled = gmm_dbus_api_device_enabled(dbus);
				if (!enabled) {
					if (gmm_question_dialog(_("<b>Enable modem</b>"), _("Modem must be enabled to scan for available networks. Enable modem?"))) {
						if (gmm_dbus_api_set_device_enabled_async(dbus, TRUE)) {
							gmm_progress_dialog_start();
						} else {
							gmm_error_dialog(_("<b>Error enabling device</b>"), gmm_dbus_api_get_error_message(dbus));
						}
					} else {
						gmm_lock_scan_controls(TRUE);
					}
				} else {
					gmm_lock_scan_controls(FALSE);
				}
			} else {
				gmm_lock_scan_controls(TRUE);
			}
		}
		gtk_notebook_set_current_page(GTK_NOTEBOOK(mainwindow.notebook), MAINWINDOW_PAGE_SCAN);
	}
}

void gmm_on_trafficbutton_toggled(GObject *object, gpointer user_data)
{
	gtk_notebook_set_current_page(GTK_NOTEBOOK(mainwindow.notebook), MAINWINDOW_PAGE_TRAFFIC);
}

static void gmm_traffic_limits_dialog_disable(GtkToggleButton *togglebutton, gpointer data)
{
	gboolean sensitive;
	gint criteria;
	
	sensitive = gtk_toggle_button_get_active(togglebutton);
	criteria = GPOINTER_TO_INT(data);
	
	switch (criteria) {
		case LIMIT_TRAFFIC:
			gtk_widget_set_sensitive(mainwindow.trafficamount, sensitive);
			gtk_widget_set_sensitive(mainwindow.trafficunits, sensitive);
			gtk_widget_set_sensitive(mainwindow.trafficmessage, sensitive);
			gtk_widget_set_sensitive(mainwindow.trafficaction, sensitive);
			break;
		case LIMIT_TIME:
			gtk_widget_set_sensitive(mainwindow.timeamount, sensitive);
			gtk_widget_set_sensitive(mainwindow.timeunits, sensitive);
			gtk_widget_set_sensitive(mainwindow.timemessage, sensitive);
			gtk_widget_set_sensitive(mainwindow.timeaction, sensitive);
			break;
		default:
			break;
	}
}

static gboolean gmm_traffic_limits_dialog(void)
{
	gint response;
	gulong trafficboxsignal, timeboxsignal;
	
	trafficboxsignal = g_signal_connect(G_OBJECT(mainwindow.trafficlimitcheckbutton), "toggled", G_CALLBACK(gmm_traffic_limits_dialog_disable), GINT_TO_POINTER(LIMIT_TRAFFIC));
	timeboxsignal = g_signal_connect(G_OBJECT(mainwindow.timelimitcheckbutton), "toggled", G_CALLBACK(gmm_traffic_limits_dialog_disable), GINT_TO_POINTER(LIMIT_TIME));
	
	g_signal_emit_by_name(G_OBJECT(mainwindow.trafficlimitcheckbutton), "toggled", NULL);
	g_signal_emit_by_name(G_OBJECT(mainwindow.timelimitcheckbutton), "toggled", NULL);
	
	response = gtk_dialog_run(GTK_DIALOG(mainwindow.trafficlimitsdialog));
	
	g_signal_handler_disconnect(G_OBJECT(mainwindow.trafficlimitcheckbutton), trafficboxsignal);
	g_signal_handler_disconnect(G_OBJECT(mainwindow.timelimitcheckbutton), timeboxsignal);
	
	gtk_widget_hide(mainwindow.trafficlimitsdialog);
	
	return (response > 0);
}

void on_trafficsetlimitbutton_clicked(GObject *object, gpointer user_data)
{
	gchar realtrafficbuf[64], settrafficbuf[64], realtimebuf[64], settimebuf[64];
	gchar *message;
	
	if (limits == NULL) return;
	
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(mainwindow.trafficlimitcheckbutton), limits->trafficenabled);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(mainwindow.trafficamount), (gdouble)limits->trafficamount);
	gtk_combo_box_set_active(GTK_COMBO_BOX(mainwindow.trafficunits), limits->trafficunits);
	gtk_entry_set_text(GTK_ENTRY(mainwindow.trafficmessage), limits->trafficmessage);
	gtk_combo_box_set_active(GTK_COMBO_BOX(mainwindow.trafficaction), limits->trafficaction);
	
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(mainwindow.timelimitcheckbutton), limits->timeenabled);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(mainwindow.timeamount), (gdouble)limits->timeamount);
	gtk_combo_box_set_active(GTK_COMBO_BOX(mainwindow.timeunits), limits->timeunits);
	gtk_entry_set_text(GTK_ENTRY(mainwindow.timemessage), limits->timemessage);
	gtk_combo_box_set_active(GTK_COMBO_BOX(mainwindow.timeaction), limits->timeaction);
	
	if (gmm_traffic_limits_dialog()) {
		if (limits != NULL) {
			limits->trafficenabled = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(mainwindow.trafficlimitcheckbutton));
			limits->trafficamount = (guint)gtk_spin_button_get_value(GTK_SPIN_BUTTON(mainwindow.trafficamount));
			limits->trafficunits = gtk_combo_box_get_active(GTK_COMBO_BOX(mainwindow.trafficunits));
			if (limits->trafficmessage != NULL) g_free(limits->trafficmessage);
			limits->trafficmessage = g_strdup(gtk_entry_get_text(GTK_ENTRY(mainwindow.trafficmessage)));
			limits->trafficaction = gtk_combo_box_get_active(GTK_COMBO_BOX(mainwindow.trafficaction));
			
			switch (limits->trafficunits) {
				case 0:
					limits->trafficfull = limits->trafficamount*1024*1024;
					break;
				case 1:
					limits->trafficfull = limits->trafficamount*1024*1024*1024;
					break;
				case 2:
					limits->trafficfull = limits->trafficamount*1024*1024*1024*1024;
					break;
				default:
					limits->trafficfull = limits->trafficamount*1024*1024;
					break;
			}
			
			limits->trafficexecuted = FALSE;
			
			if (dbus->curdevice != NULL) {
				if ((dbus->curdevice->connected) && (limits->trafficenabled) && (limits->trafficfull < (dbus->curdevice->rxbytes + dbus->curdevice->txbytes))) {
					limits->trafficexecuted = TRUE;
				}
			}
					
			limits->timeenabled = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(mainwindow.timelimitcheckbutton));
			limits->timeamount = (guint)gtk_spin_button_get_value(GTK_SPIN_BUTTON(mainwindow.timeamount));
			limits->timeunits = gtk_combo_box_get_active(GTK_COMBO_BOX(mainwindow.timeunits));
			if (limits->timemessage != NULL) g_free(limits->timemessage);
			limits->timemessage = g_strdup(gtk_entry_get_text(GTK_ENTRY(mainwindow.timemessage)));
			limits->timeaction = gtk_combo_box_get_active(GTK_COMBO_BOX(mainwindow.timeaction));
			
			switch (limits->timeunits) {
				case 0:
					limits->timefull = limits->timeamount*60;
					break;
				case 1:
					limits->timefull = limits->timeamount*60*60;
					break;
				default:
					limits->timefull = limits->timeamount*60;
					break;
			}
			
			limits->timeexecuted = FALSE;
			
			if (dbus->curdevice != NULL) {
				if ((dbus->curdevice->connected) && (limits->timeenabled) && (limits->timefull < dbus->curdevice->sessiontime)) {
					limits->timeexecuted = TRUE;
				}
			}
			
			gmm_settings_set_boolean(settings, "limits_traffic_enabled", limits->trafficenabled);
			gmm_settings_set_int(settings, "limits_traffic_amount", (gint)limits->trafficamount);
			gmm_settings_set_int(settings, "limits_traffic_units", (guint)limits->trafficunits);
			gmm_settings_set_string(settings, "limits_traffic_message", limits->trafficmessage);
			gmm_settings_set_int(settings, "limits_traffic_action", (guint)limits->trafficaction);
			
			gmm_settings_set_boolean(settings, "limits_time_enabled", limits->timeenabled);
			gmm_settings_set_int(settings, "limits_time_amount", (guint)limits->timeamount);
			gmm_settings_set_int(settings, "limits_time_units", (guint)limits->timeunits);
			gmm_settings_set_string(settings, "limits_time_message", limits->timemessage);
			gmm_settings_set_int(settings, "limits_time_action", (guint)limits->timeaction);
			
			if (dbus->curdevice != NULL) {
				if ((limits->trafficexecuted) || (limits->timeexecuted)) {
					if ((limits->trafficexecuted) && (limits->timeexecuted)) {
						message = g_strdup_printf(_("Traffic: %s, limit set to: %s\nTime: %s, limit set to: %s\nPlease check entered values and try once more"),
													gmm_format_bytes(dbus->curdevice->rxbytes + dbus->curdevice->txbytes, realtrafficbuf, sizeof(realtrafficbuf), TRUE),
													gmm_format_bytes(limits->trafficfull, settrafficbuf, sizeof(settrafficbuf), TRUE),
													gmm_format_time(dbus->curdevice->sessiontime, realtimebuf, sizeof(realtimebuf), TRUE),
													gmm_format_time(limits->timefull, settimebuf, sizeof(settimebuf), TRUE));
						gmm_error_dialog(_("Wrong traffic and time limit values"), message);
						g_free(message);
					} else if (limits->trafficexecuted) {
						message = g_strdup_printf(_("Traffic: %s, limit set to: %s\nPlease check entered values and try once more"),
													gmm_format_bytes(dbus->curdevice->rxbytes + dbus->curdevice->txbytes, realtrafficbuf, sizeof(realtrafficbuf), TRUE),
													gmm_format_bytes(limits->trafficfull, settrafficbuf, sizeof(settrafficbuf), TRUE));
						gmm_error_dialog(_("Wrong traffic limit value"), message);
						g_free(message);
					} else if (limits->timeexecuted) {
						message = g_strdup_printf(_("Time: %s, limit set to: %s\nPlease check entered values and try once more"),
													gmm_format_time(dbus->curdevice->sessiontime, realtimebuf, sizeof(realtimebuf), TRUE),
													gmm_format_time(limits->timefull, settimebuf, sizeof(settimebuf), TRUE));
						gmm_error_dialog(_("Wrong time limit value"), message);
						g_free(message);
					}
				}
			}
		}
	}
}

static void gmm_init_traffic_list(GtkWidget *list)
{
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column;
	GtkListStore *store;
	GtkTreeIter iter;
	
	renderer = gtk_cell_renderer_text_new();
	//g_object_set(renderer, "ellipsize", PANGO_ELLIPSIZE_END, "ellipsize-set", TRUE, NULL);
	column = gtk_tree_view_column_new_with_attributes(_("Parameter"), renderer, "markup", TRAFFICLIST_PARAMETER, NULL);
	gtk_tree_view_append_column(GTK_TREE_VIEW(list), column);
	
	renderer = gtk_cell_renderer_text_new();
	g_object_set(renderer, "ellipsize", PANGO_ELLIPSIZE_END, "ellipsize-set", TRUE, NULL);
	column = gtk_tree_view_column_new_with_attributes(_("Value"), renderer, "markup", TRAFFICLIST_VALUE, NULL);
	gtk_tree_view_append_column(GTK_TREE_VIEW(list), column);
	
	store = gtk_list_store_new(TRAFFICLIST_COLUMNS, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_INT);
		
	gtk_list_store_append(store, &iter);
	gtk_list_store_set(store, &iter, TRAFFICLIST_PARAMETER, _("<small><b>Received data</b></small>"), TRAFFICLIST_ID, TRAFFICLIST_ID_RXDATA, -1);
	
	gtk_list_store_append(store, &iter);
	gtk_list_store_set(store, &iter, TRAFFICLIST_PARAMETER, _("<small><b>Transmitted data</b></small>"), TRAFFICLIST_ID, TRAFFICLIST_ID_TXDATA, -1);
	
	gtk_list_store_append(store, &iter);
	gtk_list_store_set(store, &iter, TRAFFICLIST_PARAMETER, _("<small><b>Receive speed</b></small>"), TRAFFICLIST_ID, TRAFFICLIST_ID_RXSPEED, -1);
	
	gtk_list_store_append(store, &iter);
	gtk_list_store_set(store, &iter, TRAFFICLIST_PARAMETER, _("<small><b>Transmit speed</b></small>"), TRAFFICLIST_ID, TRAFFICLIST_ID_TXSPEED, -1);
	
	gtk_list_store_append(store, &iter);
	gtk_list_store_set(store, &iter, TRAFFICLIST_PARAMETER, _("<small><b>Session time</b></small>"), TRAFFICLIST_ID, TRAFFICLIST_ID_TIME, -1);
	
	gtk_list_store_append(store, &iter);
	gtk_list_store_set(store, &iter, TRAFFICLIST_PARAMETER, _("<small><b>Traffic left</b></small>"), TRAFFICLIST_ID, TRAFFICLIST_ID_DATALIMIT, -1);
	
	gtk_list_store_append(store, &iter);
	gtk_list_store_set(store, &iter, TRAFFICLIST_PARAMETER, _("<small><b>Time left</b></small>"), TRAFFICLIST_ID, TRAFFICLIST_ID_TIMELIMIT, -1);
	
	gtk_tree_view_set_model(GTK_TREE_VIEW(list), GTK_TREE_MODEL(store));
	g_object_unref(store);
}

static void gmm_draw_traffic_speed_plot(GtkWidget *widget, cairo_t *cr, gpointer data)
{
	gint width, height;
	gint i, c, graphlen;
	gfloat maxvalue;
	gchar strbuffer[32];
	const gdouble dashed[1] = {1.0};
	dbus_api_t dbusapi;
	device_t device;
	
	dbusapi = (dbus_api_t)data;
	
	if (dbusapi == NULL) return;
	
	device = dbusapi->curdevice;
	
	maxvalue = 100.0;
	
	if ((device != NULL) && (device->connected) && (device->speedindex > 0)) {
		for (i=device->speedindex-1; i>=0; i--) {
			if (device->speedvalues[0][i] > maxvalue) {
				maxvalue = device->speedvalues[0][i];
			}
			if (device->speedvalues[1][i] > maxvalue) {
				maxvalue = device->speedvalues[1][i];
			}
		}
	}
	
	if (maxvalue < 100.0) maxvalue = 100.0;
	
	width = gtk_widget_get_allocated_width(widget);
	height = gtk_widget_get_allocated_height(widget);
	
	cairo_set_source_rgba(cr, 0, 0, 0, 1);
	cairo_set_line_width(cr, 1.5);
	
	graphlen = 19*(gint)((width-50)/19.0);
	
	cairo_move_to(cr, 25, 25);
	cairo_line_to(cr, 25, height-25);
	cairo_line_to(cr, 25+graphlen, height-25);
	cairo_line_to(cr, 25+graphlen, 25);
	cairo_line_to(cr, 25, 25);
	
	cairo_stroke(cr);
	
	cairo_set_source_rgba(cr, 0.5, 0.5, 0.5, 1);
	cairo_set_line_width(cr, 1.0);
	cairo_set_dash(cr, dashed, 1, 0);
	
	for (i=1; i<10; i++) {
		cairo_move_to(cr, 25, height-25-(i*(gint)((height-50)/10.0)));
		cairo_line_to(cr, 25+graphlen, height-25-(i*(gint)((height-50)/10.0)));
	}
	
	for (i=1; i<19; i++) {
		cairo_move_to(cr, 25+(i*(gint)((width-50)/19.0)), 25);
		cairo_line_to(cr, 25+(i*(gint)((width-50)/19.0)), height-25);
	}
	
	cairo_stroke(cr);
	
	cairo_set_dash(cr, dashed, 0, 0);
	cairo_select_font_face(cr, "Sans", CAIRO_FONT_SLANT_NORMAL, CAIRO_FONT_WEIGHT_BOLD);
	cairo_set_font_size(cr, 8);
	
	for (i=0; i<=10; i++) {
		cairo_move_to(cr, 0, height-25+3-(i*(gint)((height-50)/10.0)));
		memset(strbuffer, 0, sizeof(strbuffer));
		g_snprintf(strbuffer, sizeof(strbuffer), "%4.0f", i*(maxvalue/10.0));
		cairo_show_text(cr, strbuffer);
	}
	
	cairo_move_to(cr, 0, 15);
	cairo_show_text(cr, _("kbps"));
	
	for (i=0; i<19; i++) {
		cairo_move_to(cr, 25-5+(i*(gint)((width-50)/19.0)), height-8);
		memset(strbuffer, 0, sizeof(strbuffer));
		g_snprintf(strbuffer, sizeof(strbuffer), "%i", (i+1)*DBUS_API_THREAD_SLEEP_INTERVAL);
		cairo_show_text(cr, strbuffer);
	}
	
	cairo_move_to(cr, width-35, height-8);
	cairo_show_text(cr, _("sec"));
	
	cairo_stroke(cr);
	
	if ((device != NULL) && (device->connected) && (device->speedindex > 0)) {
		cairo_set_source_rgba(cr, 0.598, 0.0664, 0.301, 1.0);
		cairo_set_line_width(cr, 2.5);
		cairo_set_line_join(cr, CAIRO_LINE_JOIN_ROUND); 
		c = 1;
		for (i=device->speedindex-1; i>=0; i--) {
			if (i == device->speedindex-1) {
				cairo_arc(cr, 25, height-25-(gint)(device->speedvalues[1][i]*((height-50)/maxvalue)), 2.0, 0*(3.14/180.0), 360*(3.14/180.0));
				cairo_move_to(cr, 25, height-25-(gint)(device->speedvalues[1][i]*((height-50)/maxvalue)));
			} else {
				cairo_line_to(cr, 25+(c*(gint)((width-50)/19.0)), height-25-(gint)(device->speedvalues[1][i]*((height-50)/maxvalue)));
				cairo_arc(cr, 25+(c*(gint)((width-50)/19.0)), height-25-(gint)(device->speedvalues[1][i]*((height-50)/maxvalue)), 2.0, 0*(3.14/180.0), 360*(3.14/180.0));
				cairo_move_to(cr, 25+(c*(gint)((width-50)/19.0)), height-25-(gint)(device->speedvalues[1][i]*((height-50)/maxvalue)));
				c++;
			}
		}
		cairo_stroke(cr);
		
		cairo_set_source_rgba(cr, 0.0273, 0.543, 0.176, 1.0);
		cairo_set_line_width(cr, 2.5);
		cairo_set_line_join(cr, CAIRO_LINE_JOIN_ROUND); 
		c = 1;
		for (i=device->speedindex-1; i>=0; i--) {
			if (i == device->speedindex-1) {
				cairo_arc(cr, 25, height-25-(gint)(device->speedvalues[0][i]*((height-50)/maxvalue)), 2.0, 0*(3.14/180.0), 360*(3.14/180.0));
				cairo_move_to(cr, 25, height-25-(gint)(device->speedvalues[0][i]*((height-50)/maxvalue)));
			} else {
				cairo_line_to(cr, 25+(c*(gint)((width-50)/19.0)), height-25-(gint)(device->speedvalues[0][i]*((height-50)/maxvalue)));
				cairo_arc(cr, 25+(c*(gint)((width-50)/19.0)), height-25-(gint)(device->speedvalues[0][i]*((height-50)/maxvalue)), 2.0, 0*(3.14/180.0), 360*(3.14/180.0));
				cairo_move_to(cr, 25+(c*(gint)((width-50)/19.0)), height-25-(gint)(device->speedvalues[0][i]*((height-50)/maxvalue)));
				c++;
			}
		}
		cairo_stroke(cr);
	}
	
	cairo_set_source_rgba(cr, 0.0273, 0.543, 0.176, 1.0);
	cairo_set_line_width(cr, 2.5);
	
	cairo_arc(cr, width-180, 12, 2.0, 0*(3.14/180.0), 360*(3.14/180.0));
	cairo_move_to(cr, width-172, 12);
	cairo_line_to(cr, width-188, 12);
	
	cairo_stroke(cr);
	
	cairo_set_source_rgba(cr, 0.5, 0.5, 0.5, 1);
	cairo_select_font_face(cr, "Sans", CAIRO_FONT_SLANT_NORMAL, CAIRO_FONT_WEIGHT_NORMAL);
	cairo_set_font_size(cr, 10);
	
	cairo_move_to(cr, width-170, 15);
	cairo_show_text(cr, _("RX speed"));
	
	cairo_stroke(cr);
	
	cairo_set_source_rgba(cr, 0.598, 0.0664, 0.301, 1.0);
	cairo_set_line_width(cr, 2.5);
	
	cairo_arc(cr, width-90, 12, 2.0, 0*(3.14/180.0), 360*(3.14/180.0));
	cairo_move_to(cr, width-82, 12);
	cairo_line_to(cr, width-98, 12);
	
	cairo_stroke(cr);
	
	cairo_set_source_rgba(cr, 0.5, 0.5, 0.5, 1);
	cairo_select_font_face(cr, "Sans", CAIRO_FONT_SLANT_NORMAL, CAIRO_FONT_WEIGHT_NORMAL);
	cairo_set_font_size(cr, 10);
	
	cairo_move_to(cr, width-80, 15);
	cairo_show_text(cr, _("TX speed"));
	
	cairo_stroke(cr);
}

static gboolean gmm_remove_sms_foreach(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
	gulong id;
	gulong delid;
	
	delid = *((gulong *)data);
	
	gtk_tree_model_get(model, iter, SMSLIST_ID, &id, -1);
	if (id == delid) {
		gtk_list_store_remove(GTK_LIST_STORE(model), iter);
		return TRUE;
	}
	
	return FALSE;
}

void on_aboutmenuitem_activate(GObject *object, gpointer user_data)
{
	gtk_dialog_run(GTK_DIALOG(mainwindow.aboutdialog));
	
	gtk_widget_hide(mainwindow.aboutdialog);
}

void on_removesmsbutton_clicked(GObject *object, gpointer user_data)
{
	GtkTreeModel *model;
	GtkTreeSelection *selection;
	GtkTreeIter iter;
	gulong id;
	
	model = gtk_tree_view_get_model(GTK_TREE_VIEW(mainwindow.smslist));
	selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(mainwindow.smslist));
	
	if ((model != NULL) && (dbus->curdevice != NULL)) {
		if (gtk_tree_selection_get_selected(selection, &model, &iter)) {
			gtk_tree_model_get(model, &iter, SMSLIST_ID, &id, -1);
			
			if (gmm_question_dialog(_("<b>Remove message</b>"), _("Really want to remove message?"))) {
				if (gmm_smsdb_remove_sms_message(dbus->curdevice->smsdb, id)) {
					gtk_tree_model_foreach(model, gmm_remove_sms_foreach, &id);
				} else {
					gmm_error_dialog(_("<b>Error removing SMS</b>"), _("Message not found"));
				}
			}
			
		} else {
			gmm_error_dialog(_("<b>Error removing SMS</b>"), _("Message not selected"));
		}
	}
}

void on_newsmsbutton_clicked(GObject *object, gpointer user_data)
{
	GtkTextBuffer *buffer;
	GtkTextIter start, end;
	gchar *number, *text;
	gchar *oldnumber;
	
	oldnumber = gmm_settings_get_string(settings, "sms_number", "8888");
		
	if (gmm_new_sms_dialog(oldnumber, "")) {
		buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(mainwindow.smstextview));
		if (buffer != NULL) {
			number = (gchar *)gtk_entry_get_text(GTK_ENTRY(mainwindow.smsnumberentry));
			gtk_text_buffer_get_bounds(buffer, &start, &end);
			text = gtk_text_buffer_get_text(buffer, &start, &end, FALSE);
			
			if (gmm_dbus_api_send_sms(dbus, number, text)) {
				gmm_settings_set_string(settings, "sms_number", number);
				gmm_progress_dialog_start();
			} else {
				gmm_error_dialog(_("<b>Error sending message</b>"), _("Wrong number or device not ready"));
			}
		
		}
	}
	
	g_free(oldnumber);
}

void gmm_on_startscanbutton_clicked(GObject *object, gpointer user_data)
{
	if (ggmm_dbus_api_scan_networks_async(dbus)) {
		gmm_progress_dialog_start();
	} else {
		gmm_error_dialog(_("<b>Error sending message</b>"), _("Device error"));
	}
}

static void gmm_init_scan_list(GtkWidget *list)
{
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column;
	GtkListStore *store;
	
	renderer = gtk_cell_renderer_text_new();
	g_object_set(renderer, "ellipsize", PANGO_ELLIPSIZE_END, "ellipsize-set", TRUE, NULL);
	column = gtk_tree_view_column_new_with_attributes(_("Operator"), renderer, "markup", SCANLIST_OPERATOR, NULL);
	gtk_tree_view_append_column(GTK_TREE_VIEW(list), column);
	
	store = gtk_list_store_new(SCANLIST_COLUMNS, G_TYPE_STRING);
	gtk_tree_view_set_model(GTK_TREE_VIEW(list), GTK_TREE_MODEL(store));
	g_object_unref(store);
}


static void gmm_device_get_scan_result_foreach(gpointer data, gpointer user_data)
{
	network_t network;
	GtkTreeModel *model;
	GtkTreeIter iter;
	gchar *markup;
		
	network = (network_t)data;
	model = (GtkTreeModel *)user_data;
	
	if ((network == NULL) || (model == NULL)) return;
	
	markup = g_strdup_printf(_("<b>%s</b>\n<small>%s ID: %u Availability: %s Access tech: %s</small>"), 
							network->operator_long,
							network->operator_short,
							network->operator_num,
							gmm_dbus_api_get_na_status_string(network->status),
							gmm_dbus_api_get_access_tech_string(network->access_tech));
	
	gtk_list_store_append(GTK_LIST_STORE(model), &iter);
	gtk_list_store_set(GTK_LIST_STORE(model), &iter, SCANLIST_OPERATOR, markup, -1);
	
	g_free(markup);
}

static void gmm_device_get_scan_result(device_t device)
{
	GSList *netlist;
	GtkTreeModel *model;
	
	if (device == NULL) return;
	
	netlist = gmm_dbus_api_get_networks(dbus);
	
	if (netlist != NULL) {
		model = gtk_tree_view_get_model(GTK_TREE_VIEW(mainwindow.scanlist));
		if (model != NULL) {
			g_object_ref(model);
			gtk_tree_view_set_model(GTK_TREE_VIEW(mainwindow.scanlist), NULL);
			gtk_list_store_clear(GTK_LIST_STORE(model));
			g_slist_foreach(netlist, gmm_device_get_scan_result_foreach, model);
			gtk_tree_view_set_model(GTK_TREE_VIEW(mainwindow.scanlist), model);
			g_object_unref(model);
		}
	} else {
		 gmm_error_dialog(_("<b>Error scanning networks</b>"), gmm_dbus_api_get_error_message(dbus));
	}
}

void gmm_ussdentry_changed_signal(GtkEditable *editable, gpointer data)
{
	const gchar *request;
	guint validationid, sessionstate;
	
	request = gtk_entry_get_text(GTK_ENTRY(mainwindow.ussdentry));
	validationid = gmm_dbus_api_validate_ussd_request((gchar *)request);
	
	if (validationid == DBUS_API_USSD_VALIDATION_REQUEST) {
		gtk_entry_set_icon_from_stock(GTK_ENTRY(mainwindow.ussdentry), GTK_ENTRY_ICON_SECONDARY, NULL);
		gtk_entry_set_icon_tooltip_markup(GTK_ENTRY(mainwindow.ussdentry), GTK_ENTRY_ICON_SECONDARY, NULL);
		gtk_widget_set_sensitive(mainwindow.ussdsend, TRUE);
	} else if (validationid == DBUS_API_USSD_VALIDATION_INVALID) {
		gtk_entry_set_icon_from_stock(GTK_ENTRY(mainwindow.ussdentry), GTK_ENTRY_ICON_SECONDARY, GTK_STOCK_CAPS_LOCK_WARNING);
		gtk_entry_set_icon_tooltip_markup(GTK_ENTRY(mainwindow.ussdentry), GTK_ENTRY_ICON_SECONDARY, _("<b>USSD request is not valid</b>\n<small>Request must be 160 symbols long\nstarted with '*' and ended with '#'</small>"));
		gtk_widget_set_sensitive(mainwindow.ussdsend, FALSE);
	} else if (validationid == DBUS_API_USSD_VALIDATION_RESPONSE) {
		sessionstate = gmm_dbus_api_get_ussd_state(dbus);
		if (sessionstate == DBUS_API_USSD_STATE_USER_RESPONSE) {
			gtk_entry_set_icon_from_stock(GTK_ENTRY(mainwindow.ussdentry), GTK_ENTRY_ICON_SECONDARY, NULL);
			gtk_entry_set_icon_tooltip_markup(GTK_ENTRY(mainwindow.ussdentry), GTK_ENTRY_ICON_SECONDARY, NULL);
			gtk_widget_set_sensitive(mainwindow.ussdsend, TRUE);
		} else {
			gtk_entry_set_icon_from_stock(GTK_ENTRY(mainwindow.ussdentry), GTK_ENTRY_ICON_SECONDARY, GTK_STOCK_CAPS_LOCK_WARNING);
			gtk_entry_set_icon_tooltip_markup(GTK_ENTRY(mainwindow.ussdentry), GTK_ENTRY_ICON_SECONDARY, _("<b>USSD request is not valid</b>\n<small>Request must be 160 symbols long\nstarted with '*' and ended with '#'</small>"));
			gtk_widget_set_sensitive(mainwindow.ussdsend, FALSE);
		}
	}
}

void gmm_ussdentry_activate_signal(GtkEntry *entry, gpointer  user_data)
{
	gboolean ucs2;
	gchar *request;
	
	if (gtk_entry_get_text_length(GTK_ENTRY(mainwindow.ussdentry)) > 0) {
		ucs2 = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(mainwindow.ussdencoding));
		request = (gchar *)gtk_entry_get_text(GTK_ENTRY(mainwindow.ussdentry));
		gmm_ussd_send_start(request, ucs2);
	}
	
}

void gmm_ussdsend_clicked_signal(GtkButton *button, gpointer user_data)
{
	gboolean ucs2;
	gchar *request;
	
	if (gtk_entry_get_text_length(GTK_ENTRY(mainwindow.ussdentry)) > 0) {
		ucs2 = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(mainwindow.ussdencoding));
		request = (gchar *)gtk_entry_get_text(GTK_ENTRY(mainwindow.ussdentry));
		gmm_ussd_send_start(request, ucs2);
	}
}

void gmm_ussdencoding_toggled_signal(GObject *object, gpointer user_data)
{
	gboolean active;
	
	active = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(object));
	
	gmm_settings_set_boolean(settings, "ussd_ucs2_encode", active);
}

static void gmm_ussd_send_start(gchar *request, gboolean ucs2)
{
	gchar *answer;
	guint encoding;
	GtkTextBuffer *buffer;
	
	if (ucs2) {
		encoding = DBUS_API_ENCODING_UCS2;
	} else {
		encoding = DBUS_API_ENCODING_GSM7;
	}
	
	if (gmm_dbus_api_send_ussd_async(dbus, request, encoding)) {
		if (settings != NULL) {
			gmm_settings_set_string(settings, "ussd_request", request);
		}
		gmm_progress_dialog_start();
	} else {
		gmm_error_dialog(_("<b>Error sending USSD</b>"), _("Wrong USSD request or device not ready"));
	}
}

static void gmm_ussd_send_end(device_t device)
{
	GtkTextBuffer *buffer;
	
	if (device->ussdanswer != NULL) {
		buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(mainwindow.ussdtext));
		if (buffer != NULL) {
			gtk_text_buffer_set_text(buffer, device->ussdanswer, -1);
		}
		g_free(device->ussdanswer);
		device->ussdanswer = NULL;
	} else {
		gmm_error_dialog(_("<b>Error sending USSD</b>"), gmm_dbus_api_get_error_message(dbus));
	}
}

static void gmm_init_sms_list(GtkWidget *list)
{
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column;
	GtkListStore *store;
	
	renderer = gtk_cell_renderer_text_new();
	g_object_set(renderer, "ellipsize", PANGO_ELLIPSIZE_END, "ellipsize-set", TRUE, NULL);
	column = gtk_tree_view_column_new_with_attributes(_("SMS"), renderer, "markup", SMSLIST_SMS, NULL);
	gtk_tree_view_append_column(GTK_TREE_VIEW(list), column);
	
	store = gtk_list_store_new(SMSLIST_COLUMNS, G_TYPE_STRING, G_TYPE_ULONG);
	gtk_tree_view_set_model(GTK_TREE_VIEW(list), GTK_TREE_MODEL(store));
	g_object_unref(store);
	
	g_signal_connect(G_OBJECT(list), "cursor-changed", G_CALLBACK(gmm_on_sms_list_cursor_changed), NULL);
}

static void gmm_on_sms_list_cursor_changed(GtkTreeView *tree_view, gpointer data)
{
	GtkTreeModel *model;
	GtkTreeSelection *selection;
	GtkTextBuffer *buffer;
	GtkTreeIter iter;
	GtkTextIter siter, eiter;
	gulong id;
	sms_full_message_t sms;
	
	if (dbus == NULL) return;
	
	model = gtk_tree_view_get_model(tree_view);
	selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(tree_view));
	buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(mainwindow.smstext));
	
	if (dbus->curdevice != NULL) {
		if ((model != NULL) && (buffer != NULL)) {
			if (gtk_tree_selection_get_selected(selection, &model, &iter)) {
				gtk_tree_model_get(model, &iter, SMSLIST_ID, &id, -1);
				sms = gmm_smsdb_read_sms_message(dbus->curdevice->smsdb, id);
				if (sms != NULL) {
					gtk_text_buffer_set_text(buffer, sms->message->str, -1);
				} else {
					gtk_text_buffer_get_bounds(buffer, &siter, &eiter);
					gtk_text_buffer_delete(buffer, &siter, &eiter);
				}
			}
		}
	}
}

static void gmm_add_sms(sms_full_message_t sms, GtkTreeModel *model)
{
	GtkTreeIter iter;
	gchar *markup;
	struct tm *tmptime;
	gchar timestr[200];
	
	if (sms == NULL) return;
	
	if (model == NULL) model = gtk_tree_view_get_model(GTK_TREE_VIEW(mainwindow.smslist));
	
	tmptime = localtime(&sms->timestamp);
	
	if (strftime(timestr, sizeof(timestr), "%T %D", tmptime) == 0) {
		g_snprintf(timestr, sizeof(timestr), _("Unknown"));
	}
		
	markup = g_strdup_printf(_("<b>%s</b>\n<small>%s</small>"), sms->number, timestr); 
	
	gtk_list_store_append(GTK_LIST_STORE(model), &iter);
	gtk_list_store_set(GTK_LIST_STORE(model), &iter, SMSLIST_SMS, markup, SMSLIST_ID, sms->dbid, -1);
	
	g_free(markup);
}

static void gmm_get_sms_foreach(gpointer data, gpointer user_data)
{
	sms_full_message_t sms;
	GtkTreeModel *model;
	
	sms = (sms_full_message_t)data;
	model = (GtkTreeModel *)user_data;
	
	gmm_add_sms(sms, model);
}

static void gmm_device_get_sms(void)
{
	GSList *smslist;
	GtkTreeModel *model;
	
	if (dbus->curdevice != NULL) {
		smslist = gmm_smsdb_read_sms_list(dbus->curdevice->smsdb);
		
		model = gtk_tree_view_get_model(GTK_TREE_VIEW(mainwindow.smslist));
		if (model != NULL) {
			g_object_ref(model);
			gtk_tree_view_set_model(GTK_TREE_VIEW(mainwindow.smslist), NULL);
			gtk_list_store_clear(GTK_LIST_STORE(model));
			if (smslist != NULL) g_slist_foreach(smslist, gmm_get_sms_foreach, model);
			gtk_tree_view_set_model(GTK_TREE_VIEW(mainwindow.smslist), model);
			g_object_unref(model);
		}
		
		if (smslist != NULL) gmm_smsdb_free_sms_list(smslist);
	}
}

static void gmm_device_update_info(device_t device)
{
	gchar strbuf[256];
	
	if (device != NULL) {
		g_snprintf(strbuf, sizeof(strbuf), "%s %s (%s)", device->manufacturer, device->model, device->port);
		gtk_label_set_label(GTK_LABEL(mainwindow.devicevlabel), strbuf);
		gtk_label_set_label(GTK_LABEL(mainwindow.operatorvlabel), dbus->curdevice->operatorname);
		gtk_label_set_label(GTK_LABEL(mainwindow.modevlabel), gmm_dbus_api_get_mode_string(dbus->curdevice->mode));
		gtk_label_set_label(GTK_LABEL(mainwindow.imeivlabel), dbus->curdevice->imei);
		gtk_label_set_label(GTK_LABEL(mainwindow.imsivlabel), dbus->curdevice->imsi);
		g_snprintf(strbuf, sizeof(strbuf), "%u%%", dbus->curdevice->siglevel);
		gtk_label_set_label(GTK_LABEL(mainwindow.signallevelvlabel), strbuf);
	}
	
}

static void gmm_device_open(guint id)
{
	if (gmm_dbus_api_device_open(dbus, id)) {
		gmm_clear_device_data();
		gmm_device_get_sms();
		gmm_device_update_info(dbus->curdevice);
		g_printf("Selected device ID: %i\n", id);
	} else {
		gmm_error_dialog(_("<b>Error opening device</b>"), gmm_dbus_api_get_error_message(dbus));
	}
}

static gboolean gmm_devlist_item_unselect_foreach(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
	gboolean enabled;
	
	gtk_tree_model_get(model, iter, DEVLIST_ENABLED, &enabled, -1);
	if (enabled) {
		gtk_list_store_set(GTK_LIST_STORE(model), iter, DEVLIST_ENABLED, FALSE, -1);
		return TRUE;
	}
	
	return FALSE;
}

static void on_devlist_item_selected(GtkCellRendererToggle *cell_renderer, gchar *path, gpointer data)
{
	GtkTreeIter iter;
	GtkTreeModel *model;
	gboolean enabled;
	guint id;
	
	model = gtk_tree_view_get_model(GTK_TREE_VIEW(mainwindow.devlist));
	if (gtk_tree_model_get_iter_from_string(model, &iter, path)) {
		gtk_tree_model_get(model, &iter, DEVLIST_ENABLED, &enabled, DEVLIST_ID, &id, -1);
		if (!enabled) {
			gtk_tree_model_foreach(model, gmm_devlist_item_unselect_foreach, NULL);
			gtk_list_store_set(GTK_LIST_STORE(model), &iter, DEVLIST_ENABLED, TRUE, -1);
			gmm_device_open(id);
		}
	}
}

static void gmm_init_device_list(GtkWidget *list)
{
	GtkCellRenderer *tbrenderer;
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column;
	GtkListStore *store;
	
	tbrenderer = gtk_cell_renderer_toggle_new();
	column = gtk_tree_view_column_new_with_attributes(_("Selected"), tbrenderer, "active", DEVLIST_ENABLED, NULL);
	gtk_tree_view_append_column(GTK_TREE_VIEW(list), column);
	gtk_cell_renderer_toggle_set_radio(GTK_CELL_RENDERER_TOGGLE(tbrenderer), TRUE);
		
	renderer = gtk_cell_renderer_text_new();
	column = gtk_tree_view_column_new_with_attributes(_("Device"), renderer, "markup", DEVLIST_DESCRIPTION, NULL);
	gtk_tree_view_append_column(GTK_TREE_VIEW(list), column);
	
	store = gtk_list_store_new(DEVLIST_COLUMNS, G_TYPE_BOOLEAN, G_TYPE_STRING, G_TYPE_INT, G_TYPE_STRING);
	gtk_tree_view_set_model(GTK_TREE_VIEW(list), GTK_TREE_MODEL(store));
	g_object_unref(store);
	
	g_signal_connect(G_OBJECT(tbrenderer), "toggled", G_CALLBACK(on_devlist_item_selected), NULL);
}

static void gmm_add_device(device_t device, GtkTreeModel *model)
{
	GtkTreeIter iter;
	gchar *markup;
	gchar *devtype;
	
	if (device == NULL) return;
	
	if (model == NULL) model = gtk_tree_view_get_model(GTK_TREE_VIEW(mainwindow.devlist));
	
	if (device->type == DBUS_API_DEVICE_TYPE_GSM) {
		devtype = "GSM";
	} else if (device->type == DBUS_API_DEVICE_TYPE_CDMA) {
		devtype = "CDMA";
	}
	
	markup = g_strdup_printf(_("<b>%s %s</b>\nVersion:%s Port:%s Type:%s"), device->manufacturer, device->model, device->version, device->port, devtype); 
	
	gtk_list_store_append(GTK_LIST_STORE(model), &iter);
	gtk_list_store_set(GTK_LIST_STORE(model), &iter, DEVLIST_ENABLED, FALSE,
													DEVLIST_DESCRIPTION, markup,
													DEVLIST_ID, device->id,
													DEVLIST_IDENTIFIER, device->identifier,
													-1);
	
	g_free(markup);
}

static void gmm_add_devices_foreach(gpointer data, gpointer user_data)
{
	device_t device;
	GtkTreeModel *model;
	
	device = (device_t)data;
	model = (GtkTreeModel *)user_data;
	
	gmm_add_device(device, model);
}

static gboolean gmm_remove_device_foreach(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
	guint id;
	guint delid;
	
	delid = GPOINTER_TO_UINT(data);
	
	gtk_tree_model_get(model, iter, DEVLIST_ID, &id, -1);
	if (id == delid) {
		gtk_list_store_remove(GTK_LIST_STORE(model), iter);
		return TRUE;
	}
	
	return FALSE;
}

static void gmm_remove_device(guint id)
{
	GtkTreeModel *model;
	
	model = gtk_tree_view_get_model(GTK_TREE_VIEW(mainwindow.devlist));
	
	if (model == NULL) return;
	
	gtk_tree_model_foreach(model, gmm_remove_device_foreach, GUINT_TO_POINTER(id));
}


static gboolean gmm_select_device_foreach(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
	guint id;
	gchar *identifier;
	selectdetails_t seldetails;
	
	seldetails = (selectdetails_t)data;
	
	gtk_tree_model_get(model, iter, DEVLIST_ID, &id, DEVLIST_IDENTIFIER, &identifier, -1);
	if (!seldetails->selectfirst) {
		if (identifier != NULL) {
			if (g_str_equal(seldetails->identifier, identifier)) {
				gtk_list_store_set(GTK_LIST_STORE(model), iter, DEVLIST_ENABLED, TRUE, -1);
				seldetails->id = id;
				seldetails->selected = TRUE;
				return TRUE;
			}
		}
	} else {
		gtk_list_store_set(GTK_LIST_STORE(model), iter, DEVLIST_ENABLED, TRUE, -1);
		seldetails->id = id;
		seldetails->identifier = identifier;
		seldetails->selected = TRUE;
		return TRUE;
	}
	
	return FALSE;
}

static void gmm_select_device(gchar *identifier)
{
	GtkTreeModel *model;
	selectdetails_t seldetails;
	
	model = gtk_tree_view_get_model(GTK_TREE_VIEW(mainwindow.devlist));
	
	if (model == NULL) return;
	
	seldetails = g_new(struct _selectdetails, 1);
	
	seldetails->id = 0;
	seldetails->identifier = identifier;
	seldetails->selected = FALSE;
	seldetails->selectfirst = FALSE;
	
	if (seldetails->identifier != NULL) {
		gtk_tree_model_foreach(model, gmm_select_device_foreach, seldetails);
	}
	
	if (!seldetails->selected) {
		seldetails->selectfirst = TRUE;
		gtk_tree_model_foreach(model, gmm_select_device_foreach, seldetails);
	}
	
	if (seldetails->selected) {
		gmm_device_open(seldetails->id);
		if (seldetails->identifier != NULL) {
			gmm_settings_set_string(settings, "device_identifier", seldetails->identifier);
		}
		gmm_buttons_lock(FALSE);
	} else {
		g_printf("No devices to select\n");
		gmm_buttons_lock(TRUE);
	}
	
	g_free(seldetails);
}

static void gmm_buttons_lock(gboolean lock)
{
	gtk_widget_set_sensitive(mainwindow.smsbutton, !lock);
	gtk_widget_set_sensitive(mainwindow.ussdbutton, !lock);
	gtk_widget_set_sensitive(mainwindow.infobutton, !lock);
	gtk_widget_set_sensitive(mainwindow.scanbutton, !lock);
	gtk_widget_set_sensitive(mainwindow.trafficbutton, !lock);
		
	if (lock) {
		gtk_toggle_tool_button_set_active(GTK_TOGGLE_TOOL_BUTTON(mainwindow.devbutton), TRUE);
		gtk_notebook_set_current_page(GTK_NOTEBOOK(mainwindow.notebook), MAINWINDOW_PAGE_DEVICES);
		gtk_widget_show(mainwindow.nodevbar);
	} else {
		gtk_widget_hide(mainwindow.nodevbar);
	}
}

static gboolean gmm_add_sms_from_thread(gpointer data)
{
	sms_notification_t smsnotify;
	sms_full_message_t sms;
	guint i, numsms;
	gulong ident;
	NotifyNotification *notify;
	gchar *nheader;
	GString *msgsenders;
	GdkPixbuf *icon;
	gchar smsnumstr[32];
	
	smsnotify = (sms_notification_t)data;
	
	if (smsnotify != NULL) {
		numsms = 0;
		msgsenders = g_string_new(_("Message senders: "));
		for (i=0; i<smsnotify->idents->len; i++) {
			ident = g_array_index(smsnotify->idents, gulong, i);
			sms = gmm_smsdb_read_sms_message(smsnotify->device->smsdb, ident);
			gmm_add_sms(sms, NULL);
			if (i == (smsnotify->idents->len-1)) {
				g_string_append_printf(msgsenders, "%s.", sms->number);
			} else {
				g_string_append_printf(msgsenders, "%s, ", sms->number);
			}
			numsms++;
		}
		gmm_dbus_api_smsnotify_free(smsnotify);
		
		if (numsms > 1) {
			nheader = g_strdup_printf(_("Received %u new SMS messages"), numsms);
		} else {
			nheader = g_strdup(_("Received new SMS message"));	
		}
		
		icon = gdk_pixbuf_new_from_file(RESOURCE_MAINWINDOW_ICON, NULL);
		
		notify = notify_notification_new(nheader, msgsenders->str, NULL);
		notify_notification_set_timeout(notify, 3000);
		notify_notification_set_image_from_pixbuf(notify, icon);
		if (!notify_notification_show(notify, NULL)) {
			g_warning("Failed to send notification");
		}
		g_object_unref(G_OBJECT(notify));
		g_object_unref(G_OBJECT(icon));
		
		g_free(nheader);
		g_string_free(msgsenders, TRUE);
		
		if (gtk_notebook_get_current_page(GTK_NOTEBOOK(mainwindow.notebook)) != MAINWINDOW_PAGE_SMS) {
			mainwindow.unreadsms += numsms;
			#ifdef _NO_LIBINDICATE
				memset(smsnumstr, 0, sizeof(smsnumstr));
				g_snprintf(smsnumstr, sizeof(smsnumstr), _("Unread SMS: %u"), mainwindow.unreadsms);
				gtk_status_icon_set_tooltip_text(mainwindow.statusicon, smsnumstr);
			#else
				memset(smsnumstr, 0, sizeof(smsnumstr));
				g_snprintf(smsnumstr, sizeof(smsnumstr), "%u", mainwindow.unreadsms);
				indicate_indicator_set_property(mainwindow.indunread, "draw_attention", "true");
				indicate_indicator_set_property(mainwindow.indunread, "count", smsnumstr);
			#endif
		} else {
			mainwindow.unreadsms = 0;
			#ifdef _NO_LIBINDICATE
				gtk_status_icon_set_tooltip_text(mainwindow.statusicon, _("No unread messages"));
			#else
				indicate_indicator_set_property(mainwindow.indunread, "draw_attention", "false");
				indicate_indicator_set_property(mainwindow.indunread, "count", "");
			#endif
		}
	}
		
	return FALSE;
} 

static gchar *gmm_format_speed(gfloat speed, gchar *buffer, gsize bufsize, gboolean small)
{
	gdouble fpvalue;
	
	if ((buffer == NULL) || (bufsize == 0)) return NULL;
	
	memset(buffer, 0, bufsize);
	
	if (speed < 1024.0) {
		if (small) {
			g_snprintf(buffer, bufsize, _("<small><b>%.3f kbps</b></small>"), speed);
		} else {
			g_snprintf(buffer, bufsize, _("%.3f kbps"), speed);
		}
	} else if ((speed >= 1024.0) && (speed < 1048576.0)) {
		fpvalue = speed / (gdouble)(1024.0);
		if (small) {
			g_snprintf(buffer, bufsize, _("<small><b>%.3g Mbps</b></small>"), fpvalue);
		} else {
			g_snprintf(buffer, bufsize, _("%.3g Mbps"), fpvalue);
		}
	} else {
		fpvalue = speed / (gdouble)(1048576.0);
		if (small) {
			g_snprintf(buffer, bufsize, _("<small><b>%.3g Gbps</b></small>"), fpvalue);
		} else {
			g_snprintf(buffer, bufsize, _("%.3g Gbps"), fpvalue);
		}
	}
	
	return buffer;	
}

static gchar *gmm_format_time_number(guchar number, gchar *buffer, gsize bufsize)
{
	if ((buffer == NULL) || (bufsize == 0)) return NULL;
	
	memset(buffer, 0, bufsize);
	
	if (number < 10) {
		g_snprintf(buffer, bufsize, "0%u", (guint)number);
	} else {
		g_snprintf(buffer, bufsize, "%u", (guint)number);
	}
	
	return buffer;
}

static gchar *gmm_format_time(guint64 seconds, gchar *buffer, gsize bufsize, gboolean small)
{
	gchar secbuffer[3], minbuffer[3], hourbuffer[3];
	
	if ((buffer == NULL) || (bufsize == 0)) return NULL;
	
	memset(buffer, 0, bufsize);
	
	if (seconds < 60) {
		if (small) {
			g_snprintf(buffer, bufsize, _("<small><b>%u sec</b></small>"), (guint)seconds);
		} else {
			g_snprintf(buffer, bufsize, _("%u sec"), (guint)seconds);
		}
	} else if ((seconds >= 60) && (seconds < 3600)) {
		if (small) {
			g_snprintf(buffer, bufsize, _("<small><b>%s:%s</b></small>"), gmm_format_time_number(seconds%3600/60, minbuffer, sizeof(minbuffer)), gmm_format_time_number(seconds%60, secbuffer, sizeof(secbuffer)));
		} else {
			g_snprintf(buffer, bufsize, _("%s:%s"), gmm_format_time_number(seconds%3600/60, minbuffer, sizeof(minbuffer)), gmm_format_time_number(seconds%60, secbuffer, sizeof(secbuffer)));
		}
	} else if ((seconds >= 3600) && (seconds < 86400)) {
		if (small) {
			g_snprintf(buffer, bufsize, _("<small><b>%s:%s:%s</b></small>"), gmm_format_time_number(seconds%86400/3600, hourbuffer, sizeof(hourbuffer)), gmm_format_time_number(seconds%3600/60, minbuffer, sizeof(minbuffer)), gmm_format_time_number(seconds%60, secbuffer, sizeof(secbuffer)));
		} else {
			g_snprintf(buffer, bufsize, _("%s:%s:%s"), gmm_format_time_number(seconds%86400/3600, hourbuffer, sizeof(hourbuffer)), gmm_format_time_number(seconds%3600/60, minbuffer, sizeof(minbuffer)), gmm_format_time_number(seconds%60, secbuffer, sizeof(secbuffer)));
		}
	} else {
		if (small) {
			g_snprintf(buffer, bufsize, _("<small><b>%u day(s) %s:%s:%s</b></small>"), seconds/86400, gmm_format_time_number(seconds%86400/3600, hourbuffer, sizeof(hourbuffer)), gmm_format_time_number(seconds%3600/60, minbuffer, sizeof(minbuffer)), gmm_format_time_number(seconds%60, secbuffer, sizeof(secbuffer)));
		} else {
			g_snprintf(buffer, bufsize, _("%u day(s) %s:%s:%s"), seconds/86400, gmm_format_time_number(seconds%86400/3600, hourbuffer, sizeof(hourbuffer)), gmm_format_time_number(seconds%3600/60, minbuffer, sizeof(minbuffer)), gmm_format_time_number(seconds%60, secbuffer, sizeof(secbuffer)));
		}
	}
	
	return buffer;
}

static gchar *gmm_format_bytes(guint64 bytes, gchar *buffer, gsize bufsize, gboolean small)
{
	gdouble fpvalue;
	
	if ((buffer == NULL) || (bufsize == 0)) return NULL;
	
	memset(buffer, 0, bufsize);
	
	if (bytes < 1024) {
		if (small) {
			g_snprintf(buffer, bufsize, _("<small><b>%u</b></small>"), (guint)bytes);
		} else {
			g_snprintf(buffer, bufsize, _("%u"), (guint)bytes);
		}
	} else if ((bytes >= 1024) && (bytes < 1048576ull)) {
		fpvalue = bytes / (gdouble)(1024.0);
		if (small) {
			g_snprintf(buffer, bufsize, _("<small><b>%.3g Kb</b></small>"), fpvalue);
		} else {
			g_snprintf(buffer, bufsize, _("%.3g Kb"), fpvalue);
		}
	} else if ((bytes >= 1048576ull) && (bytes < 1073741824ull)) {
		fpvalue = bytes / (gdouble)(1048576.0);
		if (small) {
			g_snprintf(buffer, bufsize, _("<small><b>%.3g Mb</b></small>"), fpvalue);
		} else {
			g_snprintf(buffer, bufsize, _("%.3g Mb"), fpvalue);
		}
	} else if ((bytes >= 1073741824ull) && (bytes < 109951162800ull)) {
		fpvalue = bytes / (gdouble)(1073741824.0);
		if (small) {
			g_snprintf(buffer, bufsize, _("<small><b>%.3g Gb</b></small>"), fpvalue);
		} else {
			g_snprintf(buffer, bufsize, _("%.3g Gb"), fpvalue);
		}
	} else {
		fpvalue = bytes / (gdouble)(109951162800.0);
		if (small) {
			g_snprintf(buffer, bufsize, _("<small><b>%.3g Tb</b></small>"), fpvalue);
		} else {
			g_snprintf(buffer, bufsize, _("%.3g Tb"), fpvalue);
		}
	}
	
	return buffer;	
}

static gboolean gmm_update_statusbar_from_thread(gpointer data)
{
	device_t device;
	gchar *statusmsg;
	gchar rxbuffer[32], txbuffer[32];
	
	device = (device_t)data;
	
	if (device != NULL) {
		if (!device->connected) {
			statusmsg = g_strdup_printf(_("%s disconnected"), device->operatorname);		
		} else {
			statusmsg = g_strdup_printf("%s ↓ %s ↑ %s", device->operatorname, gmm_format_bytes(device->rxbytes, rxbuffer, sizeof(rxbuffer), FALSE), gmm_format_bytes(device->txbytes, txbuffer, sizeof(txbuffer), FALSE));
		}
		
		gtk_statusbar_pop(GTK_STATUSBAR(mainwindow.statusbar), mainwindow.sbcontext);
		
		mainwindow.sbcontext = gtk_statusbar_get_context_id(GTK_STATUSBAR(mainwindow.statusbar), statusmsg);
		
		gtk_statusbar_push(GTK_STATUSBAR(mainwindow.statusbar), mainwindow.sbcontext, statusmsg);
		
		g_free(statusmsg);
	} else {
		gtk_statusbar_pop(GTK_STATUSBAR(mainwindow.statusbar), mainwindow.sbcontext);
	}
	
	return FALSE;
}

static gboolean gmm_update_traffic_page_from_thread_foreach(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
	device_t device;
	gint id;
	gchar buffer[64];
	gfloat speed;
	guint64 limitleft;
	
	device = (device_t)data;
	
	if ((device == NULL) || ((device != NULL) && (!device->connected))) {
		gtk_list_store_set(GTK_LIST_STORE(model), iter, TRAFFICLIST_VALUE, "", -1);
	} else {
		gtk_tree_model_get(model, iter, TRAFFICLIST_ID, &id, -1);
		switch (id) {
			case TRAFFICLIST_ID_RXDATA:
				gtk_list_store_set(GTK_LIST_STORE(model), iter, TRAFFICLIST_VALUE, gmm_format_bytes(device->rxbytes, buffer, sizeof(buffer), TRUE), -1);
				break;
			case TRAFFICLIST_ID_TXDATA:
				gtk_list_store_set(GTK_LIST_STORE(model), iter, TRAFFICLIST_VALUE, gmm_format_bytes(device->txbytes, buffer, sizeof(buffer), TRUE), -1);
				break;
			case TRAFFICLIST_ID_RXSPEED:
				if (device->speedindex < DBUS_API_SPEED_VALUES_NUMBER) {
					if (device->speedindex == 0) {
						speed = device->speedvalues[0][device->speedindex];
					} else {
						speed = device->speedvalues[0][device->speedindex-1];
					}
				} else {
					speed = device->speedvalues[0][DBUS_API_SPEED_VALUES_NUMBER-1];
				}
				gtk_list_store_set(GTK_LIST_STORE(model), iter, TRAFFICLIST_VALUE, gmm_format_speed(speed, buffer, sizeof(buffer), TRUE), -1);
				break;
			case TRAFFICLIST_ID_TXSPEED:
				if (device->speedindex < DBUS_API_SPEED_VALUES_NUMBER) {
					if (device->speedindex == 0) {
						speed = device->speedvalues[1][device->speedindex];
					} else {
						speed = device->speedvalues[1][device->speedindex-1];
					}
				} else {
					speed = device->speedvalues[1][DBUS_API_SPEED_VALUES_NUMBER-1];
				}
				gtk_list_store_set(GTK_LIST_STORE(model), iter, TRAFFICLIST_VALUE, gmm_format_speed(speed, buffer, sizeof(buffer), TRUE), -1);
				break;
			case TRAFFICLIST_ID_TIME:
				if (device->connected) {
					gtk_list_store_set(GTK_LIST_STORE(model), iter, TRAFFICLIST_VALUE, gmm_format_time(device->sessiontime, buffer, sizeof(buffer), TRUE), -1);
				} else {
					gtk_list_store_set(GTK_LIST_STORE(model), iter, TRAFFICLIST_VALUE, _("<small><b>Disconnected</b></small>"), -1);
				}
				break;
			case TRAFFICLIST_ID_DATALIMIT:
				if (device->connected) {
					if (!limits->trafficexecuted) {
						if (limits->trafficenabled) {
							limitleft = limits->trafficfull - (device->rxbytes + device->txbytes);
							if (limits->trafficfull > (device->rxbytes + device->txbytes)) {
								gtk_list_store_set(GTK_LIST_STORE(model), iter, TRAFFICLIST_VALUE, gmm_format_bytes(limitleft, buffer, sizeof(buffer), TRUE), -1);
							} else {
								gtk_list_store_set(GTK_LIST_STORE(model), iter, TRAFFICLIST_VALUE, _("<small><b>Limit</b></small>"), -1);
							}
						} else {
							gtk_list_store_set(GTK_LIST_STORE(model), iter, TRAFFICLIST_VALUE, _("<small><b>Disabled</b></small>"), -1);
						}
					} else {
						gtk_list_store_set(GTK_LIST_STORE(model), iter, TRAFFICLIST_VALUE, _("<small><b>Limit</b></small>"), -1);
					}
				} else {
					gtk_list_store_set(GTK_LIST_STORE(model), iter, TRAFFICLIST_VALUE, _("<small><b>Disconnected</b></small>"), -1);
				}
				break;
			case TRAFFICLIST_ID_TIMELIMIT:
				if (device->connected) {
					if (!limits->timeexecuted) {
						if (limits->timeenabled) {
							limitleft = limits->timefull - device->sessiontime;
							if (limits->timefull > device->sessiontime) {
								gtk_list_store_set(GTK_LIST_STORE(model), iter, TRAFFICLIST_VALUE, gmm_format_time(limitleft, buffer, sizeof(buffer), TRUE), -1);
							} else {
								gtk_list_store_set(GTK_LIST_STORE(model), iter, TRAFFICLIST_VALUE, _("<small><b>Limit</b></small>"), -1);
							}
						} else {
							gtk_list_store_set(GTK_LIST_STORE(model), iter, TRAFFICLIST_VALUE, _("<small><b>Disabled</b></small>"), -1);
						}
					} else {
						gtk_list_store_set(GTK_LIST_STORE(model), iter, TRAFFICLIST_VALUE, _("<small><b>Limit</b></small>"), -1);
					}
				} else {
					gtk_list_store_set(GTK_LIST_STORE(model), iter, TRAFFICLIST_VALUE, _("<small><b>Disconnected</b></small>"), -1);
				}
				break;
			default:
				break;
		}
	}
	
	return FALSE;
}

static gboolean gmm_update_traffic_page_from_thread(gpointer data)
{
	device_t device;
	GtkTreeModel *model;
	GdkWindow *window;
	
	device = (device_t)data;
	
	model = gtk_tree_view_get_model(GTK_TREE_VIEW(mainwindow.trafficparamslist));
	
	if (model != NULL) {
		gtk_tree_model_foreach(model, gmm_update_traffic_page_from_thread_foreach, data);
	}
	
	window = gtk_widget_get_window(mainwindow.trafficdrawingarea);
	
	if (window != NULL) {
		//TODO: Determine rectangle
		gdk_window_invalidate_rect(window, NULL, FALSE);
	}
	
	return FALSE;
}

static gboolean gmm_show_limit_message_from_thread(gpointer data)
{
	guint event;
	gchar *header, *message;
	NotifyNotification *notify;
	GdkPixbuf *icon;
	
	event = GPOINTER_TO_INT(data);
	
	if (limits != NULL) {
		icon = gdk_pixbuf_new_from_file(RESOURCE_MAINWINDOW_ICON, NULL);
		
		if (event == DBUS_API_EVENT_TRAFFIC_LIMIT) {
			header = g_strdup(_("Traffic limit exceeded"));
			message = limits->trafficmessage;
		} else if (event == DBUS_API_EVENT_TIME_LIMIT) {
			header = g_strdup(_("Time limit exceeded"));
			message = limits->timemessage;
		} else {
			g_warning("Unknown limit");
			return FALSE;
		}
		
		notify = notify_notification_new(header, message, NULL);
		notify_notification_set_timeout(notify, 3000);
		notify_notification_set_image_from_pixbuf(notify, icon);
		if (!notify_notification_show(notify, NULL)) {
			g_warning("Failed to send notification");
		}
		
		g_free(header);
		g_object_unref(G_OBJECT(notify));
		g_object_unref(G_OBJECT(icon));
	}
		
	return FALSE;
}

void gmm_event_callback(gint event, gpointer data)
{
	guint devid;
	device_t device;
	gchar strbuf[256];
	device_info_t devinfo;
	
	switch (event) {
		case DBUS_API_EVENT_DEVICE_ADDED:
			gmm_buttons_lock(FALSE);
			device = (device_t)data;
			gmm_add_device(device, NULL);
			if (!gmm_dbus_api_get_device_opened(dbus)) {
				gmm_select_device(device->identifier);
			}
			break;
		case DBUS_API_EVENT_DEVICE_REMOVED:
			devid = GPOINTER_TO_UINT(data);
			gmm_remove_device(devid);
			if (gmm_dbus_api_get_devices(dbus) == NULL) {
				gmm_buttons_lock(TRUE);
			} else if (!gmm_dbus_api_get_device_opened(dbus)) {
				gmm_select_device(DBUS_API_DEFAULT_DEVICE_IDENTIFIER);
			}
			break;
		case DBUS_API_EVENT_SIGNAL_LEVEL_CHANGE:
			g_snprintf(strbuf, sizeof(strbuf), "%u%%", GPOINTER_TO_UINT(data));
			gtk_label_set_label(GTK_LABEL(mainwindow.signallevelvlabel), strbuf);
			break;
		case DBUS_API_EVENT_NETWORK_MODE_CHANGE:
			gtk_label_set_label(GTK_LABEL(mainwindow.modevlabel), gmm_dbus_api_get_mode_string(gmm_dbus_api_get_device_mode(dbus)));
			break;
		case DBUS_API_EVENT_NETWORK_REGISTRATION_CHANGE:
			device = (device_t)data;
			gmm_device_update_info(device);
			break;
		case DBUS_API_EVENT_CARD_ACCESS_ENABLED:
			//gmm_device_get_sms();
			break;
		case DBUS_API_EVENT_MODEM_ENABLE_RESULT:
			gmm_progress_dialog_stop();
			device = (device_t)data;
			if (device->enabled) {
				gmm_lock_sms_controls(FALSE);
				gmm_lock_ussd_controls(FALSE);
				gmm_lock_scan_controls(FALSE);
				gmm_device_get_sms();
			} else {
				gmm_lock_sms_controls(TRUE);
				gmm_lock_ussd_controls(TRUE);
				gmm_lock_scan_controls(TRUE);
				gmm_error_dialog(_("<b>Error enabling device</b>"), gmm_dbus_api_get_error_message(dbus));
			}
			break;
		case DBUS_API_EVENT_SCAN_RESULT:
			gmm_progress_dialog_stop();
			device = (device_t)data;
			gmm_device_get_scan_result(device);
			break;
		case DBUS_API_EVENT_USSD_RESULT:
			gmm_progress_dialog_stop();
			device = (device_t)data;
			gmm_ussd_send_end(device);
			break;
		case DBUS_API_EVENT_SMS_ADDED:
			g_idle_add(gmm_add_sms_from_thread, data);
			break;
		case DBUS_API_EVENT_SMS_SENT:
			gmm_progress_dialog_stop();
			if (!GPOINTER_TO_UINT(data)) {
				gmm_error_dialog(_("<b>Error sending message</b>"), gmm_dbus_api_get_error_message(dbus));
			}
			break;
		case DBUS_API_EVENT_NET_STATUS:
			g_idle_add(gmm_update_statusbar_from_thread, data);
			g_idle_add(gmm_update_traffic_page_from_thread, data);
			break;
		case DBUS_API_EVENT_TRAFFIC_LIMIT:
		case DBUS_API_EVENT_TIME_LIMIT:
			g_idle_add(gmm_show_limit_message_from_thread, GINT_TO_POINTER(event));
			break;
		default:
			
			break;
	}
	
}

#ifdef _NO_LIBINDICATE
static void gmm_tray_action_signal(GtkStatusIcon *status_icon, gpointer data)
{
	if (gtk_widget_get_visible(mainwindow.window)) {
		gtk_widget_hide(mainwindow.window);
		g_signal_handler_block(G_OBJECT(mainwindow.showwin_tm), mainwindow.traysigid);
		gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(mainwindow.showwin_tm), FALSE);
		g_signal_handler_unblock(G_OBJECT(mainwindow.showwin_tm), mainwindow.traysigid);
	} else {
		gtk_widget_show(mainwindow.window);
		if ((mainwindow.unreadsms > 0) && (gmm_dbus_api_device_enabled(dbus))) {
			gtk_toggle_tool_button_set_active(GTK_TOGGLE_TOOL_BUTTON(mainwindow.smsbutton), TRUE);
			gtk_notebook_set_current_page(GTK_NOTEBOOK(mainwindow.notebook), MAINWINDOW_PAGE_SMS);
			mainwindow.unreadsms = 0;
			gtk_status_icon_set_tooltip_text(mainwindow.statusicon, _("No unread messages"));
		}
		g_signal_handler_block(G_OBJECT(mainwindow.showwin_tm), mainwindow.traysigid);
		gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(mainwindow.showwin_tm), TRUE);
		g_signal_handler_unblock(G_OBJECT(mainwindow.showwin_tm), mainwindow.traysigid);
	}
}

static gint gmm_tray_window_show_signal(GtkWidget *widget, GdkEvent *event, gpointer data)
{
	gmm_tray_action_signal(GTK_STATUS_ICON(mainwindow.statusicon), NULL);
	
	return 0;
}

static void gmm_tray_new_sms_signal(GtkMenuItem *menuitem, gpointer data)
{
	if (!gtk_widget_get_visible(mainwindow.window)) {
		gtk_widget_show(mainwindow.window);
		g_signal_handler_block(G_OBJECT(mainwindow.showwin_tm), mainwindow.traysigid);
		gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(mainwindow.showwin_tm), TRUE);
		g_signal_handler_unblock(G_OBJECT(mainwindow.showwin_tm), mainwindow.traysigid);
	} else {
		gtk_window_present(GTK_WINDOW(mainwindow.window));
	}
	
	if (gmm_dbus_api_device_enabled(dbus)) {
		gtk_toggle_tool_button_set_active(GTK_TOGGLE_TOOL_BUTTON(mainwindow.smsbutton), TRUE);
		gtk_notebook_set_current_page(GTK_NOTEBOOK(mainwindow.notebook), MAINWINDOW_PAGE_SMS);
		
		if (mainwindow.unreadsms > 0) {
			mainwindow.unreadsms = 0;
			gtk_status_icon_set_tooltip_text(mainwindow.statusicon, _("No unread messages"));
		}
		
		if (!gtk_widget_get_visible(mainwindow.newsmsdialog)) {
			g_signal_emit_by_name(G_OBJECT(mainwindow.newsmsbutton), "clicked", NULL);
		}
	}
}

static void gmm_tray_popup_signal(GtkStatusIcon *status_icon, guint button, guint activate_time, gpointer data)
{
	if (gmm_dbus_api_device_enabled(dbus)) {
		gtk_widget_set_sensitive(mainwindow.newsms_tm, TRUE);
	} else {
		gtk_widget_set_sensitive(mainwindow.newsms_tm, FALSE);
	}
	
	gtk_menu_popup(GTK_MENU(mainwindow.traymenu), NULL, NULL, gtk_status_icon_position_menu, status_icon, button, activate_time);
}
#else
static void gmm_indicator_server_show_signal(IndicateServer *arg0, guint arg1, gpointer user_data)
{
	guint indicator;
	
	indicator = GPOINTER_TO_UINT(user_data);
	
	gtk_window_present(GTK_WINDOW(mainwindow.window));
	
	if (gmm_dbus_api_device_enabled(dbus)) {
		gtk_toggle_tool_button_set_active(GTK_TOGGLE_TOOL_BUTTON(mainwindow.smsbutton), TRUE);
		gtk_notebook_set_current_page(GTK_NOTEBOOK(mainwindow.notebook), MAINWINDOW_PAGE_SMS);
		
		if (mainwindow.unreadsms > 0) {
			mainwindow.unreadsms = 0;
			indicate_indicator_set_property(mainwindow.indunread, "draw_attention", "false");
			indicate_indicator_set_property(mainwindow.indunread, "count", "");
		}
		
		if (indicator == INDICATOR_ID_NEW) {
			if (!gtk_widget_get_visible(mainwindow.newsmsdialog)) {
				g_signal_emit_by_name(G_OBJECT(mainwindow.newsmsbutton), "clicked", NULL);
			}
		}
	}
}
#endif

static void gmm_load_settings(settings_t settings, traffic_limits_t limits)
{
	gboolean boolean;
	gchar *text;
	
	boolean = gmm_settings_get_boolean(settings, "ussd_ucs2_encode", TRUE);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(mainwindow.ussdencoding), boolean);
	text = gmm_settings_get_string(settings, "ussd_request", "*100#");
	gtk_entry_set_text(GTK_ENTRY(mainwindow.ussdentry), text);
	g_free(text);
	text = gmm_settings_get_string(settings, "device_identifier", DBUS_API_DEFAULT_DEVICE_IDENTIFIER);
	gmm_select_device(text);
	g_free(text);
	
	if (limits != NULL) {
		limits->trafficenabled = gmm_settings_get_boolean(settings, "limits_traffic_enabled", FALSE);
		limits->trafficamount = (guint)gmm_settings_get_int(settings, "limits_traffic_amount", 150);
		limits->trafficunits = (guint)gmm_settings_get_int(settings, "limits_traffic_units", 0);
		limits->trafficmessage = gmm_settings_get_string(settings, "limits_traffic_message", _("Traffic limit exceeded... It's time to take rest \\(^_^)/"));
		limits->trafficaction = (guint)gmm_settings_get_int(settings, "limits_traffic_action", 0);
		
		switch (limits->trafficunits) {
			case 0:
				limits->trafficfull = limits->trafficamount*1024*1024;
				break;
			case 1:
				limits->trafficfull = limits->trafficamount*1024*1024*1024;
				break;
			case 2:
				limits->trafficfull = limits->trafficamount*1024*1024*1024*1024;
				break;
			default:
				limits->trafficfull = limits->trafficamount*1024*1024;
				break;
		}
		
		limits->trafficexecuted = FALSE;
		
		limits->timeenabled = gmm_settings_get_boolean(settings, "limits_time_enabled", FALSE);
		limits->timeamount = (guint)gmm_settings_get_int(settings, "limits_time_amount", 60);
		limits->timeunits = (guint)gmm_settings_get_int(settings, "limits_time_units", 0);
		limits->timemessage = gmm_settings_get_string(settings, "limits_time_message", _("Time limit exceeded... Go sleep and have nice dreams -_-"));
		limits->timeaction = (guint)gmm_settings_get_int(settings, "limits_time_action", 0);
		
		switch (limits->timeunits) {
			case 0:
				limits->timefull = limits->timeamount*60;
				break;
			case 1:
				limits->timefull = limits->timeamount*60*60;
				break;
			default:
				limits->timefull = limits->timeamount*60;
				break;
		}
		
		limits->timeexecuted = FALSE;
	}
}

static void gmm_application_activate_signal(GtkApplication *application)
{
	GList *list;
	GtkWidget *window;
	
	GtkBuilder *builder;
	GtkTreeModel *model;
	GtkStyleContext *context;
	GdkPixbuf *logo;
	GtkWidget *tbimage;
	
	list = gtk_application_get_windows(GTK_APPLICATION(application));
	
	if (list) {
		gtk_window_present(GTK_WINDOW(g_list_nth(list, 0)));
	} else {
		builder = gtk_builder_new();	
		gtk_builder_add_from_file(builder, RESOURCE_MAINWINDOW_UI, NULL);
		gtk_builder_set_translation_domain(builder, RESOURCE_LOCALE_DOMAIN);
		
		mainwindow.window = GTK_WIDGET(gtk_builder_get_object(builder, "window"));
		mainwindow.toolbar = GTK_WIDGET(gtk_builder_get_object(builder, "toolbar"));
		mainwindow.statusbar = GTK_WIDGET(gtk_builder_get_object(builder, "statusbar"));
		mainwindow.notebook = GTK_WIDGET(gtk_builder_get_object(builder, "notebook"));
				
		mainwindow.devbutton = GTK_WIDGET(gtk_builder_get_object(builder, "devbutton"));
		mainwindow.smsbutton = GTK_WIDGET(gtk_builder_get_object(builder, "smsbutton"));
		mainwindow.ussdbutton = GTK_WIDGET(gtk_builder_get_object(builder, "ussdbutton"));
		mainwindow.infobutton = GTK_WIDGET(gtk_builder_get_object(builder, "infobutton"));
		mainwindow.scanbutton = GTK_WIDGET(gtk_builder_get_object(builder, "scanbutton"));
		mainwindow.trafficbutton = GTK_WIDGET(gtk_builder_get_object(builder, "trafficbutton"));
		
		mainwindow.aboutdialog = GTK_WIDGET(gtk_builder_get_object(builder, "aboutdialog"));
		mainwindow.questiondialog = GTK_WIDGET(gtk_builder_get_object(builder, "questiondialog"));
		mainwindow.errordialog = GTK_WIDGET(gtk_builder_get_object(builder, "errordialog"));
		mainwindow.progressdialog = GTK_WIDGET(gtk_builder_get_object(builder, "progressdialog"));
		mainwindow.progressbar = GTK_WIDGET(gtk_builder_get_object(builder, "progressbar"));
		mainwindow.nodevbar = GTK_WIDGET(gtk_builder_get_object(builder, "nodevbar"));
		mainwindow.newsmsdialog = GTK_WIDGET(gtk_builder_get_object(builder, "newsmsdialog"));
		mainwindow.smsnumberentry = GTK_WIDGET(gtk_builder_get_object(builder, "smsnumberentry"));
		mainwindow.smstextview = GTK_WIDGET(gtk_builder_get_object(builder, "smstextview"));
		mainwindow.sendsmsbutton = GTK_WIDGET(gtk_builder_get_object(builder, "sendsmsbutton"));
		
		mainwindow.devlist = GTK_WIDGET(gtk_builder_get_object(builder, "devlist"));
		
		mainwindow.smslist = GTK_WIDGET(gtk_builder_get_object(builder, "smslist"));
		mainwindow.smstext = GTK_WIDGET(gtk_builder_get_object(builder, "smstext"));
		mainwindow.newsmsbutton = GTK_WIDGET(gtk_builder_get_object(builder, "newsmsbutton"));
		mainwindow.removesmsbutton = GTK_WIDGET(gtk_builder_get_object(builder, "removesmsbutton"));
		
		mainwindow.devicevlabel = GTK_WIDGET(gtk_builder_get_object(builder, "devicevlabel"));
		mainwindow.operatorvlabel = GTK_WIDGET(gtk_builder_get_object(builder, "operatorvlabel"));
		mainwindow.modevlabel = GTK_WIDGET(gtk_builder_get_object(builder, "modevlabel"));
		mainwindow.imeivlabel = GTK_WIDGET(gtk_builder_get_object(builder, "imeivlabel"));
		mainwindow.imsivlabel = GTK_WIDGET(gtk_builder_get_object(builder, "imsivlabel"));
		mainwindow.signallevelvlabel = GTK_WIDGET(gtk_builder_get_object(builder, "signallevelvlabel"));
		
		mainwindow.ussdentry = GTK_WIDGET(gtk_builder_get_object(builder, "ussdentry"));
		mainwindow.ussdencoding = GTK_WIDGET(gtk_builder_get_object(builder, "ussdencoding"));
		mainwindow.ussdsend = GTK_WIDGET(gtk_builder_get_object(builder, "ussdsend"));
		mainwindow.ussdtext = GTK_WIDGET(gtk_builder_get_object(builder, "ussdtext"));
		
		mainwindow.scanlist = GTK_WIDGET(gtk_builder_get_object(builder, "scanlist"));
		mainwindow.startscanbutton = GTK_WIDGET(gtk_builder_get_object(builder, "startscanbutton"));
		
		mainwindow.trafficparamslist = GTK_WIDGET(gtk_builder_get_object(builder, "trafficparamslist"));
		mainwindow.trafficdrawingarea = GTK_WIDGET(gtk_builder_get_object(builder, "trafficdrawingarea"));
		
		mainwindow.trafficlimitsdialog = GTK_WIDGET(gtk_builder_get_object(builder, "trafficlimitsdialog"));
		
		mainwindow.trafficlimitcheckbutton = GTK_WIDGET(gtk_builder_get_object(builder, "trafficlimitcheckbutton"));
		mainwindow.trafficamount = GTK_WIDGET(gtk_builder_get_object(builder, "trafficamount"));
		mainwindow.trafficunits = GTK_WIDGET(gtk_builder_get_object(builder, "trafficunits"));
		mainwindow.trafficmessage = GTK_WIDGET(gtk_builder_get_object(builder, "trafficmessage"));
		mainwindow.trafficaction = GTK_WIDGET(gtk_builder_get_object(builder, "trafficaction"));
		
		mainwindow.timelimitcheckbutton = GTK_WIDGET(gtk_builder_get_object(builder, "timelimitcheckbutton"));
		mainwindow.timeamount = GTK_WIDGET(gtk_builder_get_object(builder, "timeamount"));
		mainwindow.timeunits = GTK_WIDGET(gtk_builder_get_object(builder, "timeunits"));
		mainwindow.timemessage = GTK_WIDGET(gtk_builder_get_object(builder, "timemessage"));
		mainwindow.timeaction = GTK_WIDGET(gtk_builder_get_object(builder, "timeaction"));
		
		context = gtk_widget_get_style_context(GTK_WIDGET(mainwindow.toolbar));
		gtk_style_context_add_class(context, GTK_STYLE_CLASS_PRIMARY_TOOLBAR);
		
		gtk_builder_connect_signals(builder, NULL);
		
		g_object_unref(G_OBJECT(builder));
		
		logo = gdk_pixbuf_new_from_file(RESOURCE_MAINWINDOW_ICON, NULL);
		
		if (logo != NULL) {
			gtk_about_dialog_set_logo(GTK_ABOUT_DIALOG(mainwindow.aboutdialog), logo);
			gtk_window_set_icon(GTK_WINDOW(mainwindow.window), logo);
			gtk_window_set_icon(GTK_WINDOW(mainwindow.aboutdialog), logo);
			gtk_window_set_icon(GTK_WINDOW(mainwindow.questiondialog), logo);
			gtk_window_set_icon(GTK_WINDOW(mainwindow.errordialog), logo);
			gtk_window_set_icon(GTK_WINDOW(mainwindow.progressdialog), logo);
			gtk_window_set_icon(GTK_WINDOW(mainwindow.newsmsdialog), logo);
			g_object_unref(G_OBJECT(logo));
		}
		
		tbimage = gtk_image_new_from_file(RESOURCE_TOOLBAR_DEV);
		gtk_tool_button_set_icon_widget(GTK_TOOL_BUTTON(mainwindow.devbutton), GTK_WIDGET(tbimage));
		gtk_widget_show(tbimage);
		
		tbimage = gtk_image_new_from_file(RESOURCE_TOOLBAR_SMS);
		gtk_tool_button_set_icon_widget(GTK_TOOL_BUTTON(mainwindow.smsbutton), GTK_WIDGET(tbimage));
		gtk_widget_show(tbimage);
		
		tbimage = gtk_image_new_from_file(RESOURCE_TOOLBAR_USSD);
		gtk_tool_button_set_icon_widget(GTK_TOOL_BUTTON(mainwindow.ussdbutton), GTK_WIDGET(tbimage));
		gtk_widget_show(tbimage);
		
		tbimage = gtk_image_new_from_file(RESOURCE_TOOLBAR_INFO);
		gtk_tool_button_set_icon_widget(GTK_TOOL_BUTTON(mainwindow.infobutton), GTK_WIDGET(tbimage));
		gtk_widget_show(tbimage);
		
		tbimage = gtk_image_new_from_file(RESOURCE_TOOLBAR_SCAN);
		gtk_tool_button_set_icon_widget(GTK_TOOL_BUTTON(mainwindow.scanbutton), GTK_WIDGET(tbimage));
		gtk_widget_show(tbimage);
		
		tbimage = gtk_image_new_from_file(RESOURCE_TOOLBAR_TRAFFIC);
		gtk_tool_button_set_icon_widget(GTK_TOOL_BUTTON(mainwindow.trafficbutton), GTK_WIDGET(tbimage));
		gtk_widget_show(tbimage);
				
		gtk_window_set_application(GTK_WINDOW(mainwindow.window), GTK_APPLICATION(application));
		gtk_window_set_application(GTK_WINDOW(mainwindow.aboutdialog), GTK_APPLICATION(application));
		gtk_window_set_application(GTK_WINDOW(mainwindow.questiondialog), GTK_APPLICATION(application));
		gtk_window_set_application(GTK_WINDOW(mainwindow.errordialog), GTK_APPLICATION(application));
		gtk_window_set_application(GTK_WINDOW(mainwindow.progressdialog), GTK_APPLICATION(application));
		gtk_window_set_application(GTK_WINDOW(mainwindow.newsmsdialog), GTK_APPLICATION(application));
		
		if (!clioptions->invisible) gtk_widget_show(mainwindow.window);
		
		gmm_init_device_list(mainwindow.devlist);
		gmm_init_sms_list(mainwindow.smslist);
		gmm_init_scan_list(mainwindow.scanlist);
		gmm_init_traffic_list(mainwindow.trafficparamslist);
		
		settings = gmm_settings_open(RESOURCE_LOCALE_DOMAIN, "settings.conf");
		
		
		
		dbus = gmm_dbus_api_open(gmm_event_callback, clioptions, limits);
		
		if (dbus != NULL) {
			if (gmm_dbus_api_enum_devices(dbus)) {
				model = gtk_tree_view_get_model(GTK_TREE_VIEW(mainwindow.devlist));
				if (model != NULL) {
					g_object_ref(model);
					gtk_tree_view_set_model(GTK_TREE_VIEW(mainwindow.devlist), NULL);
					gtk_list_store_clear(GTK_LIST_STORE(model));
					g_slist_foreach(gmm_dbus_api_get_devices(dbus), gmm_add_devices_foreach, model);
					gtk_tree_view_set_model(GTK_TREE_VIEW(mainwindow.devlist), model);
					g_object_unref(model);
				}
			}
			
			g_signal_connect(G_OBJECT(mainwindow.trafficdrawingarea), "draw", G_CALLBACK(gmm_draw_traffic_speed_plot), dbus);
			
			notify_init("Modem manager GUI");
			
			mainwindow.unreadsms = 0;
			
			#ifdef _NO_LIBINDICATE
				mainwindow.statusicon = gtk_status_icon_new_from_file(RESOURCE_MAINWINDOW_ICON);
				g_signal_connect(G_OBJECT(mainwindow.statusicon), "activate", G_CALLBACK(gmm_tray_action_signal), NULL);
				gtk_status_icon_set_tooltip_text(mainwindow.statusicon, _("No unread messages"));
				
				mainwindow.traymenu = gtk_menu_new();
				
				mainwindow.showwin_tm =	gtk_check_menu_item_new_with_label(_("Show window"));
				gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(mainwindow.showwin_tm), gtk_widget_get_visible(mainwindow.window));
				mainwindow.traysigid = g_signal_connect(G_OBJECT(mainwindow.showwin_tm), "activate", G_CALLBACK(gmm_tray_window_show_signal), NULL);
				
				mainwindow.sep1_tm = gtk_separator_menu_item_new();
				
				mainwindow.newsms_tm = gtk_menu_item_new_with_label(_("New SMS"));
				gtk_widget_set_sensitive(mainwindow.newsms_tm, FALSE);
				g_signal_connect(G_OBJECT(mainwindow.newsms_tm), "activate", G_CALLBACK(gmm_tray_new_sms_signal), NULL);
				
				mainwindow.sep2_tm = gtk_separator_menu_item_new();
				
				mainwindow.quit_tm = gtk_image_menu_item_new_from_stock(GTK_STOCK_QUIT, NULL);
				g_signal_connect(G_OBJECT(mainwindow.quit_tm), "activate", G_CALLBACK(on_exitmenuitem_activate), NULL);
				
				gtk_menu_shell_append(GTK_MENU_SHELL(mainwindow.traymenu), mainwindow.showwin_tm);
				gtk_menu_shell_append(GTK_MENU_SHELL(mainwindow.traymenu), mainwindow.sep1_tm);
				gtk_menu_shell_append(GTK_MENU_SHELL(mainwindow.traymenu), mainwindow.newsms_tm);
				gtk_menu_shell_append(GTK_MENU_SHELL(mainwindow.traymenu), mainwindow.sep2_tm);
				gtk_menu_shell_append(GTK_MENU_SHELL(mainwindow.traymenu), mainwindow.quit_tm);
				
				gtk_widget_show_all(mainwindow.traymenu);
				
				g_signal_connect(G_OBJECT(mainwindow.statusicon), "popup-menu", G_CALLBACK(gmm_tray_popup_signal), NULL);
			#else
				mainwindow.indserver = indicate_server_ref_default();
				if (mainwindow.indserver != NULL) {
					g_signal_connect(G_OBJECT(mainwindow.indserver), "server-display", G_CALLBACK(gmm_indicator_server_show_signal), GUINT_TO_POINTER(INDICATOR_ID_SERVER));
					indicate_server_set_type(mainwindow.indserver, "message.im");
					indicate_server_set_desktop_file(mainwindow.indserver, RESOURCE_DESKTOP_FILE);
					indicate_server_show(mainwindow.indserver);
					
					mainwindow.indnew = indicate_indicator_new();
					if (mainwindow.indnew != NULL) {
						g_signal_connect(G_OBJECT(mainwindow.indnew), "user-display", G_CALLBACK(gmm_indicator_server_show_signal), GUINT_TO_POINTER(INDICATOR_ID_NEW));
						indicate_indicator_set_property(mainwindow.indnew, "subtype", "im");
						indicate_indicator_set_property(mainwindow.indnew, "sender", _("New SMS"));
						indicate_indicator_set_property(mainwindow.indnew, "draw_attention", "false");
						indicate_indicator_set_property(mainwindow.indnew, "count", "");
						indicate_indicator_show(mainwindow.indnew);
					}
					
					mainwindow.indunread = indicate_indicator_new();
					if (mainwindow.indunread != NULL) {
						g_signal_connect(G_OBJECT(mainwindow.indunread), "user-display", G_CALLBACK(gmm_indicator_server_show_signal), GUINT_TO_POINTER(INDICATOR_ID_UNREAD));
						indicate_indicator_set_property(mainwindow.indunread, "subtype", "im");
						indicate_indicator_set_property(mainwindow.indunread, "sender", _("Unread SMS"));
						indicate_indicator_set_property(mainwindow.indunread, "draw_attention", "false");
						indicate_indicator_set_property(mainwindow.indunread, "count", "");
						indicate_indicator_show(mainwindow.indunread);
					}
				}
			#endif
			gmm_load_settings(settings, limits);
		} else {
			gmm_error_dialog(_("<b>Error while initialization</b>"), _("ModemManager API not available"));
			g_application_quit(G_APPLICATION(mainwindow.application));
		}
	}
}

static void gmm_application_shutdown_signal(GtkApplication *application, gpointer user_data)
{
	gmm_dbus_api_close(dbus);
	
	gmm_settings_close(settings);
	
	#ifndef _NO_LIBINDICATE
		if (mainwindow.indserver != NULL) {
			if (mainwindow.indnew != NULL) {
				indicate_indicator_hide(mainwindow.indnew);
				g_object_unref(G_OBJECT(mainwindow.indnew));
			}
			if (mainwindow.indunread != NULL) {
				indicate_indicator_hide(mainwindow.indunread);
				g_object_unref(G_OBJECT(mainwindow.indunread));
			}
			indicate_server_hide(mainwindow.indserver);
			g_object_unref(G_OBJECT(mainwindow.indserver));
		}
	#endif
}

int main(int argc, char *argv[])
{
	gint status;
	GOptionContext *optcontext;
	GError *error;
	
	clioptions = g_new0(struct _cli_options, 1);
	limits = g_new0(struct _traffic_limits, 1);
	
	GOptionEntry entries[] = {
		{ "invisible",    'i', 0, G_OPTION_ARG_NONE, &clioptions->invisible,    "Do not show window on start", NULL },
		{ "nostatistics", 'n', 0, G_OPTION_ARG_NONE, &clioptions->nostatistics, "Disable traffic statistics", NULL },
		{ "nosmsupdate",  's', 0, G_OPTION_ARG_NONE, &clioptions->nosmsupdate,  "Disable sms update", NULL },
		{ "smsdebug",     'd', 0, G_OPTION_ARG_NONE, &clioptions->smsdebug,     "Write SMS debugging messages to standard output", NULL },
		{ NULL }
	};
			
	#ifndef LC_ALL
		#define LC_ALL 0
	#endif
	
	setlocale(LC_ALL, "");
	bindtextdomain(RESOURCE_LOCALE_DOMAIN, RESOURCE_LOCALE_DIR);
	bind_textdomain_codeset(RESOURCE_LOCALE_DOMAIN, "UTF-8");
	textdomain(RESOURCE_LOCALE_DOMAIN);
	
	g_set_application_name("Modem Manager GUI");
	
	optcontext = g_option_context_new("- simple graphical interface for Modem Manager daemon");
	g_option_context_add_main_entries(optcontext, entries, RESOURCE_LOCALE_DOMAIN);
	g_option_context_add_group(optcontext, gtk_get_option_group(TRUE));
	
	clioptions->invisible = FALSE;
	error = NULL;
	
	if (!g_option_context_parse(optcontext, &argc, &argv, &error)) {
		g_print("Command line option parsing failed: %s\n", (error->message != NULL) ? error->message : "Unknown");
		g_option_context_free(optcontext);
		g_error_free(error);
		return EXIT_FAILURE;
	}
	
	g_option_context_free(optcontext);
	
	mainwindow.application = gtk_application_new("org.gtk.ModemManagerGUI", 0);
		
	g_signal_connect(mainwindow.application, "activate", G_CALLBACK(gmm_application_activate_signal), NULL);
	
	g_signal_connect(mainwindow.application, "shutdown", G_CALLBACK(gmm_application_shutdown_signal), NULL);
	
	status = g_application_run(G_APPLICATION(mainwindow.application), argc, argv);
	
	g_object_unref(G_OBJECT(mainwindow.application));
	
	g_free(clioptions);
	g_free(limits);
	
	return status;
}
