/*
 *      dbus-api.h
 *      
 *      Copyright 2012 Alex <alex@linuxonly.ru>
 *      
 *      This program is free software: you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation; either version 3 of the License, or
 *      (at your option) any later version.
 *      
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *      
 *      You should have received a copy of the GNU General Public License
 *      along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef __DBUS_API_H__
#define __DBUS_API_H__

#include <glib.h>
#include <gio/gio.h>
#include <gdbm.h>

#define DBUS_API_DEFAULT_DEVICE_IDENTIFIER  "0000000000000000000000000000000000000000"

#define DBUS_API_SMS_TIME_INTERVAL          5.0

#define DBUS_API_THREAD_STOP_CMD            0xffffffff

#define DBUS_API_THREAD_SLEEP_INTERVAL      2

#define DBUS_API_THREAD_SMS_CHECK_INTERVAL  5

#define DBUS_API_ASYNC_ENABLE_TIMEOUT       20000
#define DBUS_API_ASYNC_SEND_SMS_TIMEOUT     35000
#define DBUS_API_ASYNC_USSD_TIMEOUT         25000
#define DBUS_API_ASYNC_SCAN_TIMEOUT         60000

#define DBUS_API_MIN_SMS_NUMBER_LENGTH      3
#define DBUS_API_MAX_SMS_NUMBER_LENGTH      20

#define DBUS_API_MIN_USSD_REQUEST_LENGTH    1
#define DBUS_API_MAX_USSD_REQUEST_LENGTH    160

#define DBUS_API_SPEED_VALUES_NUMBER        20

enum _dbus_api_events {
	DBUS_API_EVENT_DEVICE_ADDED = 0,
	DBUS_API_EVENT_DEVICE_REMOVED,
	DBUS_API_EVENT_SMS_ADDED,
	DBUS_API_EVENT_SMS_SENT,
	DBUS_API_EVENT_SMS_REMOVED,
	DBUS_API_EVENT_SIGNAL_LEVEL_CHANGE,
	DBUS_API_EVENT_NETWORK_MODE_CHANGE,
	DBUS_API_EVENT_NETWORK_REGISTRATION_CHANGE,
	DBUS_API_EVENT_CARD_ACCESS_ENABLED,
	DBUS_API_EVENT_MODEM_ENABLE_RESULT,
	DBUS_API_EVENT_SCAN_RESULT,
	DBUS_API_EVENT_USSD_RESULT,
	DBUS_API_EVENT_NET_STATUS,
	DBUS_API_EVENT_TRAFFIC_LIMIT,
	DBUS_API_EVENT_TIME_LIMIT
};

enum _dbus_api_devtypes {
	DBUS_API_DEVICE_TYPE_GSM = 1,
	DBUS_API_DEVICE_TYPE_CDMA
};

enum _dbus_api_modes {
	DBUS_API_MODE_UNKNOWN = 0,
	DBUS_API_MODE_GSM,
	DBUS_API_MODE_GSM_COMPACT,
	DBUS_API_MODE_GPRS,
	DBUS_API_MODE_EDGE,
	DBUS_API_MODE_UMTS,
	DBUS_API_MODE_HSDPA,
	DBUS_API_MODE_HSUPA,
	DBUS_API_MODE_HSPA
};

enum _dbus_api_network_availability {
	DBUS_API_NA_UNKNOWN = 0,
	DBUS_API_NA_AVAILABLE,
	DBUS_API_NA_CURRENT,
	DBUS_API_NA_FORBIDDEN
};

enum _dbus_api_access_tech {
	DBUS_API_ACCESS_TECH_GSM = 0,
	DBUS_API_ACCESS_TECH_GSM_COMPACT,
	DBUS_API_ACCESS_TECH_UMTS,
	DBUS_API_ACCESS_TECH_EDGE,
	DBUS_API_ACCESS_TECH_HSDPA,
	DBUS_API_ACCESS_TECH_HSUPA,
	DBUS_API_ACCESS_TECH_HSPA
};

enum _dbus_api_encoding {
	DBUS_API_ENCODING_GSM7 = 0,
	DBUS_API_ENCODING_UCS2
};

enum _dbus_api_reg_status {
	DBUS_API_REG_STATUS_IDLE = 0,
	DBUS_API_REG_STATUS_HOME,
	DBUS_API_REG_STATUS_SEARCHING,
	DBUS_API_REG_STATUS_DENIED,
	DBUS_API_REG_STATUS_UNKNOWN,
	DBUS_API_REG_STATUS_ROAMING
};

enum _dbus_api_state {
	DBUS_API_STATE_UNKNOWN = 0,
    DBUS_API_STATE_DISABLED = 10,
    DBUS_API_STATE_DISABLING = 20,
    DBUS_API_STATE_ENABLING = 30,
    DBUS_API_STATE_ENABLED = 40,
    DBUS_API_STATE_SEARCHING = 50,
    DBUS_API_STATE_REGISTERED = 60,
    DBUS_API_STATE_DISCONNECTING = 70,
    DBUS_API_STATE_CONNECTING = 80,
    DBUS_API_STATE_CONNECTED = 90
};

enum _dbus_ussd_state {
	DBUS_API_USSD_STATE_UNKNOWN = 0,
	DBUS_API_USSD_STATE_IDLE,
	DBUS_API_USSD_STATE_ACTIVE,
	DBUS_API_USSD_STATE_USER_RESPONSE
};

enum _dbus_ussd_validation {
	DBUS_API_USSD_VALIDATION_INVALID = 0,
	DBUS_API_USSD_VALIDATION_REQUEST,
	DBUS_API_USSD_VALIDATION_RESPONSE
};

enum _dbus_api_event_action {
	DBUS_API_EVENT_ACTION_NOTHING = 0,
	DBUS_API_EVENT_ACTION_DISCONNECT
};

typedef void (*dbus_api_event_callback)(gint event, gpointer data);

struct _network {
	guint status;
	guint access_tech;
	guint operator_num;
	gchar *operator_long;
	gchar *operator_short;
};

typedef struct _network *network_t;

struct _sms_message {
	guint id;
	gchar *snumber;
	gchar *text;
	gchar *number;
	time_t timestamp;
};

typedef struct _sms_message *sms_message_t;

struct _sms_full_message {
	gulong dbid;
	time_t timestamp;
	gchar *number;
	GArray *idents;
	GString *message;
};

typedef struct _sms_full_message *sms_full_message_t;

struct _device_info {
	gchar *device;
	gchar *imei;
	gchar *imsi;
	gchar *operatorcode;
	gchar *operatorname;
	guint regstatus;
	guint allmode;
	guint mode;
	guint siglevel;
};

typedef struct _device_info *device_info_t;

struct _device {
	guint id;
	gboolean enabled;
	gchar *manufacturer;
	gchar *model;
	gchar *version;
	gchar *port;
	gchar *identifier;
	guint type;
	
	gchar *imei;
	gchar *imsi;
	gchar *operatorcode;
	gchar *operatorname;
	guint regstatus;
	guint allmode;
	
	guint mode;
	guint siglevel;
	
	guint64 rxbytes;
	guint64 txbytes;
	guint64 sessiontime;
	time_t speedchecktime;
	time_t smschecktime;
	gfloat speedvalues[2][DBUS_API_SPEED_VALUES_NUMBER];
	guint speedindex;
	gboolean connected;
	gchar *netinterface;
	
	GSList *netlist;
	gchar *ussdanswer;
	guint ussdencoding;
	gpointer smsdb;
};

typedef struct _device *device_t;

struct _cli_options {
	gboolean invisible;
	gboolean nostatistics;
	gboolean nosmsupdate;
	gboolean smsdebug;
};

typedef struct _cli_options *cli_options_t;

struct _traffic_limits {
	gboolean trafficenabled;
	gboolean trafficexecuted;
	guint trafficamount;
	guint trafficunits;
	guint64 trafficfull;
	gchar *trafficmessage;
	guint trafficaction;
	gboolean timeenabled;
	gboolean timeexecuted;
	guint timeamount;
	guint timeunits;
	guint64 timefull;
	gchar *timemessage;
	guint timeaction;
};

typedef struct _traffic_limits *traffic_limits_t;

struct _dbus_api {
	GDBusConnection *connection;
	GDBusProxy *managerproxy;
	GDBusProxy *modemproxy;
	GDBusProxy *cardproxy;
	GDBusProxy *netproxy;
	GDBusProxy *smsproxy;
	GDBusProxy *ussdproxy;
	gulong netsignal;
	gulong smssignal;
	gulong statesignal;
	GSList *devices;
	GError *error;
	GThread *updsmsthread;
	gint threadctl[2];
	device_t curdevice;
	gpointer nm;
	gboolean nmdisabled;
	gboolean smsdisabled;
	gboolean smsdebug;
	traffic_limits_t limits;
	dbus_api_event_callback callback;
};

typedef struct _dbus_api *dbus_api_t;

struct _sms_notification {
	GArray *idents;
	device_t device;
	dbus_api_t instance;
};

typedef struct _sms_notification *sms_notification_t;

gchar *gmm_dbus_api_get_error_message(dbus_api_t instance);
gboolean gmm_dbus_api_device_enabled(dbus_api_t instance);
gboolean gmm_dbus_api_device_blocked(dbus_api_t instance);
gboolean gmm_dbus_api_device_connected(dbus_api_t instance);
gboolean gmm_dbus_api_set_device_enabled_async(dbus_api_t instance, gboolean enabled);
dbus_api_t gmm_dbus_api_open(dbus_api_event_callback callback, cli_options_t options, traffic_limits_t limits);
gboolean gmm_dbus_api_close(dbus_api_t instance);
gboolean gmm_dbus_api_remove_device(dbus_api_t instance, guint id);
GSList *gmm_dbus_api_get_devices(dbus_api_t instance);
guint gmm_dbus_api_enum_devices(dbus_api_t instance);
gboolean gmm_dbus_api_device_open(dbus_api_t instance, guint id);
gboolean gmm_dbus_api_get_device_opened(dbus_api_t instance);
gboolean gmm_dbus_api_update_device_info(dbus_api_t instance);
guint gmm_dbus_api_get_device_mode(dbus_api_t instance);
gchar *gmm_dbus_api_get_mode_string(enum _dbus_api_modes mode);
gchar *gmm_dbus_api_get_na_status_string(enum _dbus_api_network_availability status);
gchar *gmm_dbus_api_get_access_tech_string(enum _dbus_api_access_tech status);
gchar *gmm_dbus_api_get_reg_status(enum _dbus_api_reg_status status);
void gmm_dbus_api_smsnotify_free(sms_notification_t smsnotify);
gboolean gmm_dbus_api_send_sms(dbus_api_t instance, gchar *number, gchar *text);
gboolean gmm_dbus_api_validate_sms_number(gchar *number);
guint gmm_dbus_api_validate_ussd_request(gchar *request);
guint gmm_dbus_api_get_ussd_state(dbus_api_t instance);
gboolean gmm_dbus_api_send_ussd_async(dbus_api_t instance, gchar *request, enum _dbus_api_encoding encoding);
GSList *gmm_dbus_api_get_networks(dbus_api_t instance);
gboolean ggmm_dbus_api_scan_networks_async(dbus_api_t instance);

#endif /* __DBUS_API_H__ */
