/*
 *      dbus-api.c
 *      
 *      Copyright 2012 Alex <alex@linuxonly.ru>
 *      
 *      This program is free software: you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation; either version 3 of the License, or
 *      (at your option) any later version.
 *      
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *      
 *      You should have received a copy of the GNU General Public License
 *      along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdlib.h>
#include <glib.h>
#include <glib/gi18n.h>
#include <gio/gio.h>
#include <glib/gprintf.h>
#include <string.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/select.h>

#include "nm.h"
#include "encoding.h"
#include "smsdb.h"
#include "dbus-api.h"
#include "resources.h"

static void gmm_dbus_api_signal_handler(GDBusProxy *proxy, const gchar *sender_name, const gchar *signal_name, GVariant *parameters, gpointer user_data);
static void gmm_dbus_api_free_error(dbus_api_t instance);
static void gmm_dbus_api_async_enable_handler(GDBusProxy *proxy, GAsyncResult *res, gpointer user_data);
static guint gmm_dbus_api_device_id(const gchar *objectpath);
static gboolean gmm_dbus_api_device_disconnect(dbus_api_t instance);
static device_t gmm_dbus_api_device_info(const gchar *objectpath, dbus_api_t instance);
static gint gmm_dbus_api_remove_device_compare(gconstpointer a, gconstpointer b);
static void gmm_dbus_api_free_devices_foreach(gpointer data, gpointer user_data);
static gboolean gmm_dbus_api_free_devices(dbus_api_t instance);
static gint gmm_dbus_api_device_open_compare(gconstpointer a, gconstpointer b);
static gboolean gmm_dbus_api_device_close(dbus_api_t instance);
static void gmm_dbus_api_free_part_sms_messages_foreach(gpointer data, gpointer user_data);
static void gmm_dbus_api_free_full_sms_messages_foreach(gpointer data, gpointer user_data);
static gint gmm_dbus_api_sms_message_sort_compare(gconstpointer a, gconstpointer b);
static gint gmm_dbus_api_sms_message_sort_index_compare(gconstpointer a, gconstpointer b);
static gint gmm_dbus_api_sms_message_part_compare(gconstpointer a, gconstpointer b);
static void gmm_dbus_api_sms_message_merge_foreach(gpointer data, gpointer user_data);
static void gmm_dbus_api_sms_add_foreach(gpointer data, gpointer user_data);
static time_t gmm_dbus_api_str_to_time(const gchar *str);
static gpointer gmm_update_thread(gpointer data);
static void gmm_dbus_api_async_send_sms_handler(GDBusProxy *proxy, GAsyncResult *res, gpointer user_data);
static gchar *gmm_dbus_api_gsm7_to_ucs2(gchar *srcstr);
static void gmm_dbus_api_async_ussd_handler(GDBusProxy *proxy, GAsyncResult *res, gpointer user_data);
static void gmm_dbus_api_free_networks_foreach(gpointer data, gpointer user_data);
static gboolean gmm_dbus_api_free_networks(dbus_api_t instance);
static void gmm_dbus_api_async_scan_handler(GDBusProxy *proxy, GAsyncResult *res, gpointer user_data);



static void gmm_dbus_api_signal_handler(GDBusProxy *proxy, const gchar *sender_name, const gchar *signal_name, GVariant *parameters, gpointer user_data)
{
	dbus_api_t instance;
	gchar *devpath;
	device_t device;
	guint deviceid;
	guint intvalue;
	gboolean enabled;
	
	instance = (dbus_api_t)user_data;
	
	if (instance->callback != NULL) {
		if (g_str_equal(signal_name, "DeviceAdded")) {
			g_variant_get(parameters, "(o)", &devpath);
			if (devpath != NULL) {
				device = gmm_dbus_api_device_info(devpath, instance);
				instance->devices = g_slist_append(instance->devices, device);
				(instance->callback)(DBUS_API_EVENT_DEVICE_ADDED, device);
			}
		} else if (g_str_equal(signal_name, "DeviceRemoved")) {
			g_variant_get(parameters, "(o)", &devpath);
			if (devpath != NULL) {
				deviceid = gmm_dbus_api_device_id(devpath);
				gmm_dbus_api_remove_device(instance, deviceid);
				(instance->callback)(DBUS_API_EVENT_DEVICE_REMOVED, GUINT_TO_POINTER(deviceid));
			}
		} else if (g_str_equal(signal_name, "SignalQuality")) {
			g_variant_get(parameters, "(u)", &intvalue);
			(instance->callback)(DBUS_API_EVENT_SIGNAL_LEVEL_CHANGE, GUINT_TO_POINTER(intvalue));
		} else if (g_str_equal(signal_name, "NetworkMode")) {
			g_variant_get(parameters, "(u)", &intvalue);
			(instance->callback)(DBUS_API_EVENT_NETWORK_MODE_CHANGE, GUINT_TO_POINTER(intvalue));
		} else if (g_str_equal(signal_name, "RegistrationInfo")) {
			if (instance->curdevice != NULL) {
				if (instance->curdevice->operatorcode != NULL) g_free(instance->curdevice->operatorcode);
				if (instance->curdevice->operatorname != NULL) g_free(instance->curdevice->operatorname);
				g_variant_get(parameters, "(uss)", &instance->curdevice->regstatus, &instance->curdevice->operatorcode, &instance->curdevice->operatorname);
				instance->curdevice->operatorcode = g_strdup(instance->curdevice->operatorcode);
				instance->curdevice->operatorname = g_strdup(instance->curdevice->operatorname);
				
				(instance->callback)(DBUS_API_EVENT_NETWORK_REGISTRATION_CHANGE, instance->curdevice);
			}
		} else if (g_str_equal(signal_name, "StateChanged")) {
			
		}
	}
	
	printf("SIGNAL: %s (%s) argtype: %s\n", signal_name, sender_name, g_variant_get_type_string(parameters));
}

static void gmm_dbus_api_free_error(dbus_api_t instance)
{
	if (instance == NULL) return;
	
	if (instance->error != NULL) {
		g_error_free(instance->error);
		instance->error = NULL;
	}
}

gchar *gmm_dbus_api_get_error_message(dbus_api_t instance)
{
	if (instance == NULL) return _("Unknown");
	
	if (instance->error == NULL) return _("Unknown");
		
	if (instance->error->message != NULL) {
		return instance->error->message;
	} else {
		return _("Unknown");
	}
}

static void gmm_dbus_api_async_enable_handler(GDBusProxy *proxy, GAsyncResult *res, gpointer user_data)
{
	dbus_api_t instance;
	GVariant *result;
		
	instance = (dbus_api_t)user_data;
	
	if (instance == NULL) return;
	
	gmm_dbus_api_free_error(instance);
	
	result = g_dbus_proxy_call_finish(proxy, res, &instance->error);
	
	if (result != NULL) g_variant_unref(result);
	
	instance->curdevice->enabled = gmm_dbus_api_device_enabled(instance);
	
	(instance->callback)(DBUS_API_EVENT_MODEM_ENABLE_RESULT, instance->curdevice);
}

static guint gmm_dbus_api_device_id(const gchar *objectpath)
{
	guint id;
	gchar *devidstr;
	
	devidstr = strrchr(objectpath, '/') + 1;
	if ((devidstr != NULL) && (devidstr[0] != '\0')) {
		id = atoi(devidstr);
	} else {
		id = 0;
	}
	
	return id;
}

gboolean gmm_dbus_api_device_enabled(dbus_api_t instance)
{
	GVariant *info;
	gboolean enabled;
		
	if (instance == NULL) return FALSE;
	if (instance->connection == NULL) return FALSE;
	if (instance->curdevice == NULL) return FALSE;
	if (instance->modemproxy == NULL) return FALSE;
	
	info = g_dbus_proxy_get_cached_property(instance->modemproxy, "Enabled");
	
	enabled = g_variant_get_boolean(info);
	
	g_variant_unref(info);
	
	return enabled;
}

gboolean gmm_dbus_api_device_blocked(dbus_api_t instance)
{
	GVariant *info;
	gchar *blockstr;
	gsize strsize = 256;
	gboolean enabled;
		
	if (instance == NULL) return FALSE;
	if (instance->connection == NULL) return FALSE;
	if (instance->curdevice == NULL) return FALSE;
	if (instance->modemproxy == NULL) return FALSE;
	
	info = g_dbus_proxy_get_cached_property(instance->modemproxy, "UnlockRequired");
	
	blockstr = (gchar *)g_variant_get_string(info, &strsize);
	
	g_variant_unref(info);
	
	if (blockstr[0] == '\0') {
		enabled = FALSE;
	} else {
		enabled = TRUE;
	}
	
	return enabled;
}

gboolean gmm_dbus_api_device_connected(dbus_api_t instance)
{
	GVariant *info;
	guint value;
		
	if (instance == NULL) return FALSE;
	if (instance->connection == NULL) return FALSE;
	if (instance->curdevice == NULL) return FALSE;
	if (instance->modemproxy == NULL) return FALSE;
	
	if ((!instance->nmdisabled) && (instance->nm != NULL)) {
		return nm_get_device_enabled(instance->nm);
	} else {
		info = g_dbus_proxy_get_cached_property(instance->modemproxy, "State");
		
		value = g_variant_get_uint32(info);
		
		g_variant_unref(info);
		
		switch (value) {
			case DBUS_API_STATE_UNKNOWN:
				return FALSE;
			case DBUS_API_STATE_DISABLED:
				return FALSE;
			case DBUS_API_STATE_DISABLING:
				return FALSE;
			case DBUS_API_STATE_ENABLING:
				return FALSE;
			case DBUS_API_STATE_ENABLED:
				return FALSE;
			case DBUS_API_STATE_SEARCHING:
				return FALSE;
			case DBUS_API_STATE_REGISTERED:
				return FALSE;
			case DBUS_API_STATE_DISCONNECTING:
				return TRUE;
			case DBUS_API_STATE_CONNECTING:
				return TRUE;
			case DBUS_API_STATE_CONNECTED:
				return TRUE;
			default:
				return FALSE;
		}
	}
}

gboolean gmm_dbus_api_set_device_enabled_async(dbus_api_t instance, gboolean enabled)
{
	if (instance == NULL) return FALSE;
	if (instance->connection == NULL) return FALSE;
	if (instance->curdevice == NULL) return FALSE;
	if (instance->modemproxy == NULL) return FALSE;
	
	g_dbus_proxy_call(instance->modemproxy,
                     "Enable",
                     g_variant_new("(b)", enabled),
                     G_DBUS_CALL_FLAGS_NONE,
                     DBUS_API_ASYNC_ENABLE_TIMEOUT,
                     NULL,
                     (GAsyncReadyCallback)gmm_dbus_api_async_enable_handler,
                     instance);
	
	return TRUE;
}

static gboolean gmm_dbus_api_device_disconnect(dbus_api_t instance)
{
	if (instance == NULL) return FALSE;
	if (instance->connection == NULL) return FALSE;
	if (instance->curdevice == NULL) return FALSE;
	if (instance->modemproxy == NULL) return FALSE;
	
	if (!instance->curdevice->connected) return TRUE;
	
	g_dbus_proxy_call_sync(instance->modemproxy, "Disconnect", NULL, 0, -1, NULL, &instance->error);
	
	if (instance->error != NULL) {
		g_warning("DBus proxy call error: %s", instance->error->message);
		return FALSE;
	}
	
	return TRUE;
}

static device_t gmm_dbus_api_device_info(const gchar *objectpath, dbus_api_t instance)
{
	device_t device;
	//GError *error;
	GDBusProxy *modemproxy;
	GVariant *info;
	gchar *manufacturer, *model, *version;
	gsize size = 256;
		
	if ((objectpath == NULL) || (instance == NULL) || (instance->connection == NULL)) return NULL;
	
	device = g_new(struct _device, 1);
	
	device->id = gmm_dbus_api_device_id(objectpath);
	
	//device->smslist = NULL;
	device->smsdb = NULL;
	device->netlist = NULL;
	device->operatorname = NULL;
	device->operatorcode = NULL;
	device->imei = NULL;
	device->imsi = NULL;
	
	gmm_dbus_api_free_error(instance);
	
	modemproxy = g_dbus_proxy_new_sync(instance->connection,
										G_DBUS_PROXY_FLAGS_NONE,
										NULL,
										"org.freedesktop.ModemManager",
										objectpath,
										"org.freedesktop.ModemManager.Modem",
										NULL,
										&instance->error);
										
	if ((modemproxy == NULL) && (instance->error != NULL)) {
		g_warning("DBus proxy error: %s", instance->error->message);
		//g_error_free(error);
		device->manufacturer = strdup(_("Unknown"));
		device->model = strdup(_("Unknown"));
		device->version = strdup(_("Unknown"));
		device->port = strdup(_("Unknown"));
		device->type = 0;
		return device;
	}
	
	gmm_dbus_api_free_error(instance);
	
	info = g_dbus_proxy_call_sync(modemproxy, "GetInfo", NULL, 0, -1, NULL, &instance->error);
	
	if ((info == NULL) && (instance->error != NULL)) {
		g_warning("DBus proxy call error: %s", instance->error->message);
		//g_error_free(error);
		device->manufacturer = strdup(_("Unknown"));
		device->model = strdup(_("Unknown"));
		device->version = strdup(_("Unknown"));
		device->port = strdup(_("Unknown"));
		device->type = 0;
		g_object_unref(modemproxy);
		return device;
	}
	
	g_variant_get(info, "((sss))", &manufacturer, &model, &version);
	
	if (manufacturer != NULL) {
		device->manufacturer = strdup(manufacturer);
	} else {
		device->manufacturer = strdup(_("Unknown"));
	}
	
	if (model != NULL) {
		device->model = strdup(model);
	} else {
		device->model = strdup(_("Unknown"));
	}
	
	if (version != NULL) {
		device->version = strdup(version);
	} else {
		device->version = strdup(_("Unknown"));
	}
	
	g_variant_unref(info);
	
	info = g_dbus_proxy_get_cached_property(modemproxy, "Device");
	
	device->port = g_strdup(g_variant_get_string(info, &size));
	
	g_variant_unref(info);
	
	info = g_dbus_proxy_get_cached_property(modemproxy, "Type");
	
	device->type = g_variant_get_uint32(info);
		
	g_variant_unref(info);
	
	info = g_dbus_proxy_get_cached_property(modemproxy, "Enabled");
	
	device->enabled = g_variant_get_boolean(info);
		
	g_variant_unref(info);
	
	size = 256;
		
	info = g_dbus_proxy_get_cached_property(modemproxy, "DeviceIdentifier");
	
	device->identifier = g_strdup(g_variant_get_string(info, &size));
	
	g_variant_unref(info);
		
	g_object_unref(modemproxy);
	
	return device;
}

dbus_api_t gmm_dbus_api_open(dbus_api_event_callback callback, cli_options_t options, traffic_limits_t limits)
{
	dbus_api_t instance;
	
	instance = g_new0(struct _dbus_api, 1);
	
	if (instance == NULL) return NULL;
	
	gmm_dbus_api_free_error(instance);
	
	instance->callback = callback;
	instance->curdevice = NULL;
	instance->limits = limits;
	
	if (options != NULL) {
		instance->nmdisabled = options->nostatistics;
		instance->smsdisabled = options->nosmsupdate;
		instance->smsdebug = options->smsdebug;
	} else {
		instance->nmdisabled = FALSE;
		instance->smsdisabled = FALSE;
		instance->smsdebug = FALSE;
	}
	
	instance->connection = g_bus_get_sync(G_BUS_TYPE_SYSTEM, NULL, &instance->error);
	
	if ((instance->connection == NULL) && (instance->error != NULL)) {
		g_warning("DBus connection error: %s\n", (instance->error->message != NULL) ? instance->error->message : "Unknown");
		gmm_dbus_api_free_error(instance);
		g_free(instance);
		return NULL;
	}
	
	instance->managerproxy = g_dbus_proxy_new_sync(instance->connection,
													G_DBUS_PROXY_FLAGS_NONE,
													NULL,
													"org.freedesktop.ModemManager",
													"/org/freedesktop/ModemManager",
													"org.freedesktop.ModemManager",
													NULL,
													&instance->error);
	
	
	if ((instance->managerproxy != NULL) && (instance->error != NULL)) {
		g_warning("DBus manager proxy error: %s\n", (instance->error->message != NULL) ? instance->error->message : "Unknown");
		gmm_dbus_api_free_error(instance);
		g_free(instance);
		return NULL;
	}
	
	if (pipe(instance->threadctl) == 0) {
		#if GLIB_CHECK_VERSION(2,32,0)
			instance->updsmsthread = g_thread_new("Modem Manager GUI update thread", gmm_update_thread, instance);
		#else 
			instance->updsmsthread = g_thread_create(gmm_update_thread, instance, TRUE, &instance->error);
		#endif
	}
	
	if (!instance->nmdisabled) {
		instance->nm = nm_open();
	} else {
		instance->nm = NULL;
	}
	
	g_signal_connect(instance->managerproxy, "g-signal", G_CALLBACK(gmm_dbus_api_signal_handler), instance);
	
	return instance;
}

gboolean gmm_dbus_api_close(dbus_api_t instance)
{
	gint command;
	
	if (instance == NULL) return FALSE;
	
	command = DBUS_API_THREAD_STOP_CMD;
	
	if (write(instance->threadctl[0], &command, sizeof(command)) > 0) {
		g_thread_join(instance->updsmsthread);
		close(instance->threadctl[0]);
	}
	
	if (!instance->nmdisabled) {
		nm_close(instance->nm);
		instance->nm = NULL;
	}
	
	if (instance->managerproxy != NULL) {
		g_object_unref(instance->managerproxy);
		instance->managerproxy = NULL;
	}
	
	gmm_dbus_api_device_close(instance);
	
	if (instance->connection != NULL) {
		g_object_unref(instance->connection);
		instance->connection = NULL;
	}
	
	g_free(instance);
	
	return TRUE;
}

static gint gmm_dbus_api_remove_device_compare(gconstpointer a, gconstpointer b)
{
	device_t device;
	guint id;
	
	device = (device_t)a;
	id = GPOINTER_TO_UINT(b);
	
	if (device->id < id) {
		return 1;
	} else if (device->id > id) {
		return -1;
	} else {
		return 0;
	}
}

gboolean gmm_dbus_api_remove_device(dbus_api_t instance, guint id)
{
	GSList *devicelist;
	
	if ((instance == NULL) || ((instance != NULL) && (instance->devices == NULL))) return FALSE;
	
	devicelist = g_slist_find_custom(instance->devices, GUINT_TO_POINTER(id), gmm_dbus_api_remove_device_compare);
	
	if (devicelist == NULL) {
		return FALSE;
	} else {
		if (id == instance->curdevice->id) {
			gmm_dbus_api_device_close(instance);
		}
		instance->devices = g_slist_remove(instance->devices, devicelist->data);
		return TRUE;
	}
}

static void gmm_dbus_api_free_devices_foreach(gpointer data, gpointer user_data)
{
	device_t device;
	
	if (data != NULL) return;
	
	device = (device_t)data;
	
	if (device->manufacturer != NULL) {
		g_free(device->manufacturer);
		device->manufacturer = NULL;
	}
	
	if (device->model != NULL) {
		g_free(device->model);
		device->model = NULL;
	}
	
	if (device->version != NULL) {
		g_free(device->version);
		device->version = NULL;
	}
	
	if (device->port != NULL) {
		g_free(device->port);
		device->port = NULL;
	}
	
	if (device->identifier != NULL) {
		g_free(device->identifier);
		device->identifier = NULL;
	}
}

static gboolean gmm_dbus_api_free_devices(dbus_api_t instance)
{
	if (instance == NULL) return FALSE;
	
	if (instance->devices == NULL) return TRUE;
	
	g_slist_foreach(instance->devices, gmm_dbus_api_free_devices_foreach, NULL);
	
	g_slist_free(instance->devices);
	
	instance->devices = NULL;
	
	return TRUE;
}

GSList *gmm_dbus_api_get_devices(dbus_api_t instance)
{
	return instance->devices;
}

guint gmm_dbus_api_enum_devices(dbus_api_t instance)
{
	GVariant *devicesvar;
	GVariantIter deviter, deviter2;
	GVariant *devnode, *devnode2;
	GVariant *devvalue;
	gsize strsize = 256;
	const gchar *devpath;
	guint devnum;
	device_t device;
	
	if ((instance == NULL) || (instance->managerproxy == NULL)) return 0;
	
	gmm_dbus_api_free_error(instance);
	
	devicesvar = g_dbus_proxy_call_sync(instance->managerproxy,
										"EnumerateDevices",
										NULL,
										0,
										-1,
										NULL,
										&instance->error);
	
	if ((devicesvar != NULL) && (instance->error != NULL)) {
		g_warning("DBus manager proxy error: %s\n", (instance->error->message != NULL) ? instance->error->message : "Unknown");
		//g_error_free(instance->error);
		return 0;
	}
	
	gmm_dbus_api_free_devices(instance);
	
	devnum = 0;
	
	g_variant_iter_init(&deviter, devicesvar);
	
	while ((devnode = g_variant_iter_next_value(&deviter)) != NULL) {
		g_variant_iter_init(&deviter2, devnode);
		while ((devnode2 = g_variant_iter_next_value(&deviter2)) != NULL) {
			devpath = g_variant_get_string(devnode2, &strsize);
			device = gmm_dbus_api_device_info(devpath, instance);
			instance->devices = g_slist_prepend(instance->devices, device);
			devnum++;
			g_variant_unref(devnode2);
		}
		g_variant_unref(devnode);
	}
	
	if (instance->devices != NULL) instance->devices = g_slist_reverse(instance->devices);
	
	//printf("DEVNUM: %i\n", devnum);
	
	return devnum;
}

static gint gmm_dbus_api_device_open_compare(gconstpointer a, gconstpointer b)
{
	device_t device;
	guint id;
	
	device = (device_t)a;
	id = GPOINTER_TO_UINT(b);
	
	if (device->id < id) {
		return 1;
	} else if (device->id > id) {
		return -1;
	} else {
		return 0;
	}
}

gboolean gmm_dbus_api_device_open(dbus_api_t instance, guint id)
{
	GSList *devicelist;
	gchar *objectpath;
	//device_t device;
	
	if (instance == NULL) return FALSE;
	if (instance->devices == NULL) return FALSE;
	
	gmm_dbus_api_free_error(instance);
	
	if (instance->curdevice != NULL) {
		gmm_dbus_api_device_close(instance);
	}
	
	devicelist = g_slist_find_custom(instance->devices, GUINT_TO_POINTER(id), gmm_dbus_api_device_open_compare);
	
	if (devicelist == NULL) return FALSE;
		
	instance->curdevice = (device_t)devicelist->data;
	objectpath = g_strdup_printf("/org/freedesktop/ModemManager/Modems/%u", id);
	
	if (instance->curdevice->type == DBUS_API_DEVICE_TYPE_GSM) {
		instance->cardproxy = g_dbus_proxy_new_sync(instance->connection,
													G_DBUS_PROXY_FLAGS_NONE,
													NULL,
													"org.freedesktop.ModemManager",
													objectpath,
													"org.freedesktop.ModemManager.Modem.Gsm.Card",
													NULL,
													&instance->error);
		
		
		if ((instance->cardproxy == NULL) && (instance->error != NULL)) {
			g_warning("DBus manager proxy error: %s\n", (instance->error->message != NULL) ? instance->error->message : "Unknown");
			gmm_dbus_api_free_error(instance);
			//g_error_free(instance->error);
			//instance->error = NULL;
		}
	} else if (instance->curdevice->type == DBUS_API_DEVICE_TYPE_CDMA) {
		instance->cardproxy = g_dbus_proxy_new_sync(instance->connection,
													G_DBUS_PROXY_FLAGS_NONE,
													NULL,
													"org.freedesktop.ModemManager",
													objectpath,
													"org.freedesktop.ModemManager.Modem.Cdma",
													NULL,
													&instance->error);
		
		
		if ((instance->cardproxy == NULL) && (instance->error != NULL)) {
			g_warning("DBus manager proxy error: %s\n", (instance->error->message != NULL) ? instance->error->message : "Unknown");
			gmm_dbus_api_free_error(instance);
			//g_error_free(instance->error);
			//instance->error = NULL;
		}
	}
	
	instance->netproxy = g_dbus_proxy_new_sync(instance->connection,
													G_DBUS_PROXY_FLAGS_NONE,
													NULL,
													"org.freedesktop.ModemManager",
													objectpath,
													"org.freedesktop.ModemManager.Modem.Gsm.Network",
													NULL,
													&instance->error);
	
	
	if ((instance->netproxy == NULL) && (instance->error != NULL)) {
		g_warning("DBus manager proxy error: %s\n", (instance->error->message != NULL) ? instance->error->message : "Unknown");
		gmm_dbus_api_free_error(instance);
		//g_error_free(instance->error);
		//instance->error = NULL;
	} else {
		instance->netsignal = g_signal_connect(instance->netproxy, "g-signal", G_CALLBACK(gmm_dbus_api_signal_handler), instance);
	}
	
	instance->modemproxy = g_dbus_proxy_new_sync(instance->connection,
												G_DBUS_PROXY_FLAGS_NONE,
												NULL,
												"org.freedesktop.ModemManager",
												objectpath,
												"org.freedesktop.ModemManager.Modem",
												NULL,
												&instance->error);
	
	if ((instance->modemproxy == NULL) && (instance->error != NULL)) {
		g_warning("DBus modem proxy error: %s", (instance->error->message != NULL) ? instance->error->message : "Unknown");
		gmm_dbus_api_free_error(instance);
		//g_error_free(instance->error);
		//instance->error = NULL;
	} else {
		instance->statesignal = g_signal_connect(instance->netproxy, "g-signal", G_CALLBACK(gmm_dbus_api_signal_handler), instance);
	}
	
	instance->smsproxy = g_dbus_proxy_new_sync(instance->connection,
													G_DBUS_PROXY_FLAGS_NONE,
													NULL,
													"org.freedesktop.ModemManager",
													objectpath,
													"org.freedesktop.ModemManager.Modem.Gsm.SMS",
													NULL,
													&instance->error);
	
	
	if ((instance->smsproxy == NULL) && (instance->error != NULL)) {
		g_warning("DBus manager proxy error: %s\n", (instance->error->message != NULL) ? instance->error->message : "Unknown");
		gmm_dbus_api_free_error(instance);
		//g_error_free(instance->error);
		//instance->error = NULL;
	} else {
		instance->smssignal = g_signal_connect(instance->smsproxy, "g-signal", G_CALLBACK(gmm_dbus_api_signal_handler), instance);
	}
	
	instance->ussdproxy = g_dbus_proxy_new_sync(instance->connection,
													G_DBUS_PROXY_FLAGS_NONE,
													NULL,
													"org.freedesktop.ModemManager",
													objectpath,
													"org.freedesktop.ModemManager.Modem.Gsm.Ussd",
													NULL,
													&instance->error);
	
	
	if ((instance->ussdproxy == NULL) && (instance->error != NULL)) {
		g_warning("DBus manager proxy error: %s\n", (instance->error->message != NULL) ? instance->error->message : "Unknown");
		gmm_dbus_api_free_error(instance);
		//g_error_free(instance->error);
		//instance->error = NULL;
	}
	
	g_free(objectpath);
	
	instance->curdevice->smsdb = gmm_smsdb_open(RESOURCE_LOCALE_DOMAIN, instance->curdevice->identifier);
	
	instance->curdevice->rxbytes = 0;
	instance->curdevice->txbytes = 0;
	instance->curdevice->sessiontime = 0;
	instance->curdevice->speedchecktime = 0;
	instance->curdevice->smschecktime = 0;
	instance->curdevice->speedindex = 0;
	instance->curdevice->connected = FALSE;
	instance->curdevice->netinterface = NULL;
	memset(instance->curdevice->speedvalues, 0, sizeof(instance->curdevice->speedvalues));
	
	gmm_dbus_api_update_device_info(instance);
	
	return TRUE;
}

static gboolean gmm_dbus_api_device_close(dbus_api_t instance)
{
	if (instance == NULL) return FALSE;
	
	//gmm_dbus_api_free_full_sms_messages(instance);
	
	if (instance->curdevice != NULL) {
		gmm_smsdb_close(instance->curdevice->smsdb);
		if (instance->curdevice->operatorcode != NULL) g_free(instance->curdevice->operatorcode);
		if (instance->curdevice->operatorname != NULL) g_free(instance->curdevice->operatorname);
		if (instance->curdevice->imei != NULL) g_free(instance->curdevice->imei);
		if (instance->curdevice->imsi != NULL) g_free(instance->curdevice->imsi);
		//gmm_dbus_api_free_device_info(instance);
		gmm_dbus_api_free_networks(instance);
		instance->curdevice = NULL;
	}
	
	if (instance->cardproxy != NULL) {
		g_object_unref(instance->cardproxy);
		instance->cardproxy = NULL;
	}
	
	if (instance->netproxy != NULL) {
		if (g_signal_handler_is_connected(instance->netproxy, instance->netsignal)) {
			g_signal_handler_disconnect(instance->netproxy, instance->netsignal);
		}
		g_object_unref(instance->netproxy);
		instance->netproxy = NULL;
	}
	
	if (instance->modemproxy != NULL) {
		if (g_signal_handler_is_connected(instance->modemproxy, instance->statesignal)) {
			g_signal_handler_disconnect(instance->modemproxy, instance->statesignal);
		}
		g_object_unref(instance->modemproxy);
		instance->modemproxy = NULL;
	}
	
	if (instance->smsproxy != NULL) {
		if (g_signal_handler_is_connected(instance->smsproxy, instance->smssignal)) {
			g_signal_handler_disconnect(instance->smsproxy, instance->smssignal);
		}
		g_object_unref(instance->smsproxy);
		instance->smsproxy = NULL;
	}
	
	if (instance->ussdproxy != NULL) {
		g_object_unref(instance->ussdproxy);
		instance->ussdproxy = NULL;
	}
	
	return TRUE;
}

gboolean gmm_dbus_api_get_device_opened(dbus_api_t instance)
{
	if (instance == NULL) return FALSE;
	
	if (instance->curdevice != NULL) {
		return TRUE;
	} else {
		return FALSE;
	}
}

gboolean gmm_dbus_api_update_device_info(dbus_api_t instance)
{
	GVariant *info;
	
	if (instance == NULL) return FALSE; 
	if (instance->curdevice == NULL) return FALSE;
	if ((instance->cardproxy == NULL) || (instance->netproxy == NULL)) return FALSE;
	
	gmm_dbus_api_free_error(instance);
	
	info = g_dbus_proxy_call_sync(instance->netproxy, "GetSignalQuality", NULL, 0, -1, NULL, &instance->error);
	
	if ((info == NULL) && (instance->error != NULL)) {
		instance->curdevice->siglevel = 0;
		g_warning("DBus proxy error: %s", (instance->error->message != NULL) ? instance->error->message : "Unknown");
	} else {
		g_variant_get(info, "(u)", &instance->curdevice->siglevel);
		g_variant_unref(info);
	}
	
	gmm_dbus_api_free_error(instance);
	
	if (instance->curdevice->operatorcode != NULL) g_free(instance->curdevice->operatorcode);
	if (instance->curdevice->operatorname != NULL) g_free(instance->curdevice->operatorname);
	
	info = g_dbus_proxy_call_sync(instance->netproxy, "GetRegistrationInfo", NULL, 0, -1, NULL, &instance->error);
	
	if ((info == NULL) && (instance->error != NULL)) {
		instance->curdevice->operatorcode = g_strdup(_("Unknown"));
		instance->curdevice->operatorname = g_strdup(_("Unknown"));
		instance->curdevice->regstatus = 0;
		g_warning("DBus proxy error: %s", (instance->error->message != NULL) ? instance->error->message : "Unknown");
	} else {
		g_variant_get(info, "((uss))", &instance->curdevice->regstatus, &instance->curdevice->operatorcode, &instance->curdevice->operatorname);
		instance->curdevice->operatorcode = g_strdup(instance->curdevice->operatorcode);
		instance->curdevice->operatorname = g_strdup(instance->curdevice->operatorname);
		g_variant_unref(info);
	}
		
	info = g_dbus_proxy_get_cached_property(instance->netproxy, "AllowedMode");
	
	instance->curdevice->allmode = g_variant_get_uint32(info);
	
	g_variant_unref(info);
	
	info = g_dbus_proxy_get_cached_property(instance->netproxy, "AccessTechnology");
	
	instance->curdevice->mode = g_variant_get_uint32(info);
		
	g_variant_unref(info);
	
	if (instance->curdevice->type == DBUS_API_DEVICE_TYPE_GSM) {
		gmm_dbus_api_free_error(instance);
		
		if (instance->curdevice->imei != NULL) g_free(instance->curdevice->imei);
		
		info = g_dbus_proxy_call_sync(instance->cardproxy, "GetImei", NULL, 0, -1, NULL, &instance->error);
		
		if ((info == NULL) && (instance->error != NULL)) {
			instance->curdevice->imei = g_strdup(_("Unknown"));
			g_warning("DBus proxy error: %s", (instance->error->message != NULL) ? instance->error->message : "Unknown");
		} else {
			g_variant_get(info, "(s)", &instance->curdevice->imei);
			instance->curdevice->imei = g_strdup(instance->curdevice->imei);
			g_variant_unref(info);
		}
		
		gmm_dbus_api_free_error(instance);
		
		if (instance->curdevice->imsi != NULL) g_free(instance->curdevice->imsi);
		
		info = g_dbus_proxy_call_sync(instance->cardproxy, "GetImsi", NULL, 0, -1, NULL, &instance->error);
		
		if ((info == NULL) && (instance->error != NULL)) {
			instance->curdevice->imsi = g_strdup(_("Unknown"));
			g_warning("DBus proxy error: %s", (instance->error->message != NULL) ? instance->error->message : "Unknown");
		} else {
			g_variant_get(info, "(s)", &instance->curdevice->imsi);
			instance->curdevice->imsi = g_strdup(instance->curdevice->imsi);
			g_variant_unref(info);
		}
	} else if (instance->curdevice->type == DBUS_API_DEVICE_TYPE_CDMA) {
		if (instance->curdevice->imei != NULL) g_free(instance->curdevice->imei);
		instance->curdevice->imei = g_strdup(_("Unknown"));
		
		if (instance->curdevice->imsi != NULL) g_free(instance->curdevice->imsi);
		instance->curdevice->imsi = g_strdup(_("Unknown"));
	}
	
	return TRUE;
}

guint gmm_dbus_api_get_device_mode(dbus_api_t instance)
{
	GVariant *info;
	GError *error;
	guint mode;
	
	if ((instance == NULL) || (instance->netproxy == NULL) || (instance->curdevice == NULL)) return 0;
	
	info = g_dbus_proxy_get_cached_property(instance->netproxy, "AccessTechnology");
	
	mode = g_variant_get_uint32(info);
		
	g_variant_unref(info);
	
	return mode;
}

gchar *gmm_dbus_api_get_mode_string(enum _dbus_api_modes mode)
{
	switch (mode) {
		case DBUS_API_MODE_UNKNOWN:
			return "Unknown";
		case DBUS_API_MODE_GSM:
			return "GSM";
		case DBUS_API_MODE_GSM_COMPACT:
			return "Compact GSM";
		case DBUS_API_MODE_GPRS:
			return "GPRS";
		case DBUS_API_MODE_EDGE:
			return "EDGE (ETSI 27.007: \"GSM w/EGPRS\")";
		case DBUS_API_MODE_UMTS:
			return "UMTS (ETSI 27.007: \"UTRAN\")";
		case DBUS_API_MODE_HSDPA:
			return "HSDPA (ETSI 27.007: \"UTRAN w/HSDPA\")";
		case DBUS_API_MODE_HSUPA:
			return "HSUPA (ETSI 27.007: \"UTRAN w/HSUPA\")";
		case DBUS_API_MODE_HSPA:
			return "HSPA (ETSI 27.007: \"UTRAN w/HSDPA and HSUPA\")";
		default:
			return "Unknown";
	}
}

gchar *gmm_dbus_api_get_na_status_string(enum _dbus_api_network_availability status)
{
	switch (status) {
		case DBUS_API_NA_UNKNOWN:
			return "Unknown";
		case DBUS_API_NA_AVAILABLE:
			return "Available";
		case DBUS_API_NA_CURRENT:
			return "Current";
		case DBUS_API_NA_FORBIDDEN:
			return "Forbidden";
		default:
			return "Unknown";
	}
}

gchar *gmm_dbus_api_get_access_tech_string(enum _dbus_api_access_tech status)
{
	switch (status) {
		case DBUS_API_ACCESS_TECH_GSM:
			return "GSM";
		case DBUS_API_ACCESS_TECH_GSM_COMPACT:
			return "GSM Compact";
		case DBUS_API_ACCESS_TECH_UMTS:
			return "UMTS";
		case DBUS_API_ACCESS_TECH_EDGE:
			return "EDGE";
		case DBUS_API_ACCESS_TECH_HSDPA:
			return "HSDPA";
		case DBUS_API_ACCESS_TECH_HSUPA:
			return "HSUPA";
		case DBUS_API_ACCESS_TECH_HSPA:
			return "HSPA";
		default:
			return "Unknown";
	}
}

gchar *gmm_dbus_api_get_reg_status(enum _dbus_api_reg_status status)
{
	switch (status) {
		case DBUS_API_REG_STATUS_IDLE:
			return "Not registered";
		case DBUS_API_REG_STATUS_HOME:
			return "Home network";
		case DBUS_API_REG_STATUS_SEARCHING:
			return "Searching";
		case DBUS_API_REG_STATUS_DENIED:
			return "Registration denied";
		case DBUS_API_REG_STATUS_UNKNOWN:
			return "Unknown status";
		case DBUS_API_REG_STATUS_ROAMING:
			return "Roaming network";
		default:
			return "Unknown status";
	}
}

static void gmm_dbus_api_free_part_sms_messages_foreach(gpointer data, gpointer user_data)
{
	sms_message_t sms;
	
	if (data != NULL) return;
	
	sms = (sms_message_t)data;
	
	if (sms->number != NULL) {
		g_free(sms->number);
		sms->number = NULL;
	}
	
	if (sms->snumber != NULL) {
		g_free(sms->snumber);
		sms->snumber = NULL;
	}
	
	if (sms->text != NULL) {
		g_free(sms->text);
		sms->text = NULL;
	}
}

static void gmm_dbus_api_free_full_sms_messages_foreach(gpointer data, gpointer user_data)
{
	sms_full_message_t sms;
	
	if (data != NULL) return;
	
	sms = (sms_full_message_t)data;
	
	if (sms->number != NULL) {
		g_free(sms->number);
		sms->number = NULL;
	}
	
	if (sms->idents != NULL) {
		g_array_free(sms->idents, TRUE);
		sms->idents = NULL;
	}
	
	if (sms->message != NULL) {
		g_string_free(sms->message, TRUE);
		sms->message = NULL;
	}
}

static gint gmm_dbus_api_sms_message_sort_compare(gconstpointer a, gconstpointer b)
{
	sms_message_t sms1, sms2;
		
	sms1 = (sms_message_t)a;
	sms2 = (sms_message_t)b;
	
	if (sms1->timestamp < sms2->timestamp) {
		return -1;
	} else if (sms1->timestamp > sms2->timestamp) {
		return 1;
	} else {
		return 0;
	}
}

static gint gmm_dbus_api_sms_message_sort_index_compare(gconstpointer a, gconstpointer b)
{
	sms_message_t sms1, sms2;
		
	sms1 = (sms_message_t)a;
	sms2 = (sms_message_t)b;
	
	if (sms1->id < sms2->id) {
		return -1;
	} else if (sms1->id > sms2->id) {
		return 1;
	} else {
		return 0;
	}
}

static gint gmm_dbus_api_sms_message_part_compare(gconstpointer a, gconstpointer b)
{
	sms_full_message_t fullsms;
	sms_message_t newsms;
	
	fullsms = (sms_full_message_t)a;
	newsms = (sms_message_t)b;
	
	if (!g_str_equal(fullsms->number, newsms->number)) return -1;
	
	if (difftime(fullsms->timestamp, newsms->timestamp) > DBUS_API_SMS_TIME_INTERVAL) {
		return 1;
	} else {
		return 0;
	}
}

static void gmm_dbus_api_sms_message_merge_foreach(gpointer data, gpointer user_data)
{
	GSList **smsfulllist;
	sms_message_t sms;
	sms_full_message_t fullsms;
	GSList *smsposition;
	
	sms = (sms_message_t)data;
	smsfulllist = (GSList **)user_data;
	
	if (*smsfulllist == NULL) {
		smsposition = NULL;
	} else {
		smsposition = g_slist_find_custom(*smsfulllist, sms, gmm_dbus_api_sms_message_part_compare);
	}
	
	if 	(smsposition == NULL) {
		fullsms = g_new(struct _sms_full_message, 1);
		fullsms->number = g_strdup(sms->number);
		fullsms->timestamp = sms->timestamp;
		fullsms->idents = g_array_new(FALSE, TRUE, sizeof(guint));
		g_array_append_val(fullsms->idents, sms->id);
		fullsms->message = g_string_new(sms->text);
		*smsfulllist = g_slist_prepend(*smsfulllist, fullsms);
	} else {
		fullsms = smsposition->data;
		g_array_append_val(fullsms->idents, sms->id);
		fullsms->message = g_string_append(fullsms->message, sms->text);
	}
}

static void gmm_dbus_api_sms_add_foreach(gpointer data, gpointer user_data)
{
	sms_full_message_t sms;
	sms_notification_t smsnotify;
	guint i;
	guint ident;
	GVariant *vident;
	GError *error;
	
	sms = (sms_full_message_t)data;
	smsnotify = (sms_notification_t)user_data; 
	
	if (gmm_smsdb_add_sms(smsnotify->device->smsdb, sms)) {
		g_array_append_val(smsnotify->idents, sms->dbid);
		
		for (i=0; i<sms->idents->len; i++) {
			ident = g_array_index(sms->idents, guint, i);
			
			error = NULL;
			
			vident = g_variant_new("(u)", ident);
			
			g_dbus_proxy_call_sync(smsnotify->instance->smsproxy, "Delete", vident, 0, -1, NULL, &error);
			
			if (error != NULL) {
				if (smsnotify->instance->error->message != NULL) {
					g_warning("DBus proxy call error: %s", (smsnotify->instance->error->message != NULL) ? smsnotify->instance->error->message : "Unknown");
				} else {
					g_warning("DBus proxy call error: Unknown");
				}
				g_error_free(error);
			}
			
			//g_variant_unref(vident);
		}
	}
}

void gmm_dbus_api_smsnotify_free(sms_notification_t smsnotify)
{
	if (smsnotify == NULL) return;
	
	g_array_free(smsnotify->idents, TRUE);
	
	g_free(smsnotify);
}

static time_t gmm_dbus_api_str_to_time(const gchar *str)
{
	guint i, len;
	gchar strbuf[3];
	struct tm btime;
	time_t timestamp;
	gint *fields[] = {&btime.tm_year, &btime.tm_mon, &btime.tm_mday,
						&btime.tm_hour,	&btime.tm_min, &btime.tm_sec};
	
	timestamp = time(NULL);
	
	if (str == NULL) return timestamp; 
	
	len = strlen(str);
	
	if (len > 12) {
		if (str[12] == '+') {
			//v.0.4.998 timestamp format
			for (i=0; i<6; i++) {
				strncpy(strbuf, str+(i*2), 2);
				strbuf[2] = '\0';
				*fields[i] = atoi(strbuf);
			}
		} else if (str[8] == ',') {
			//v.0.5.2 timestamp format
			for (i=0; i<6; i++) {
				strncpy(strbuf, str+(i*3), 2);
				strbuf[2] = '\0';
				*fields[i] = atoi(strbuf);
			}
		}
		btime.tm_year += 100;
		btime.tm_mon -= 1;
		timestamp = mktime(&btime);
	}
		
	return timestamp;
}

static gpointer gmm_update_thread(gpointer data)
{
	dbus_api_t instance;
	GVariant *info;
	GError *error;
	GSList *smspartlist, *smsfulllist;
	sms_message_t sms;
	sms_notification_t smsnotify;
	GVariantIter iter, iter2;
	GVariant *child, *anode, *pvalue;
	gsize strlength = 256;
	const gchar *timestr;
	gint command;
	fd_set readset;
	struct timeval tv;
	gint readb;
	guint64 rxbytes, txbytes;
	time_t currenttime;
	guint timeframe;
	gchar *interface;
	
	instance = (dbus_api_t)data;
	
	if (instance == NULL) return NULL;
	
	while (TRUE) {
		FD_ZERO(&readset);
		FD_SET(instance->threadctl[1], &readset);
		
		tv.tv_sec = DBUS_API_THREAD_SLEEP_INTERVAL;
		tv.tv_usec = 0;
		
		if (select(instance->threadctl[1]+1, &readset, NULL, NULL, &tv) > 0) {
			if (FD_ISSET(instance->threadctl[1], &readset)) {
				readb = read(instance->threadctl[1], &command, sizeof(command));
				if (readb > 0) {
					if (command == DBUS_API_THREAD_STOP_CMD) {
						break;
					}
				}
			}
		}
		
		currenttime = time(NULL);
		
		if ((gmm_dbus_api_device_enabled(instance)) && (!instance->nmdisabled)) {
			if (nm_update_device_connection_info(instance->nm, instance->curdevice->id)) {
				if (nm_get_device_traffic(instance->nm, &rxbytes, &txbytes)) {
					if (instance->curdevice->speedchecktime != 0) {
						timeframe = (guint)difftime(currenttime, instance->curdevice->speedchecktime);
						instance->curdevice->sessiontime += timeframe;
						if (instance->curdevice->speedindex < DBUS_API_SPEED_VALUES_NUMBER) {
							instance->curdevice->speedvalues[0][instance->curdevice->speedindex] = (gfloat)((rxbytes - instance->curdevice->rxbytes)*8)/(gfloat)(timeframe*1024); 
							instance->curdevice->speedvalues[1][instance->curdevice->speedindex] = (gfloat)((txbytes - instance->curdevice->txbytes)*8)/(gfloat)(timeframe*1024);
							instance->curdevice->speedindex++;
						} else {
							memmove(&instance->curdevice->speedvalues[0][0], &instance->curdevice->speedvalues[0][1], sizeof(instance->curdevice->speedvalues[0]) - sizeof(instance->curdevice->speedvalues[0][0]));
							memmove(&instance->curdevice->speedvalues[1][0], &instance->curdevice->speedvalues[1][1], sizeof(instance->curdevice->speedvalues[1]) - sizeof(instance->curdevice->speedvalues[1][0]));
							instance->curdevice->speedvalues[0][DBUS_API_SPEED_VALUES_NUMBER-1] = (gfloat)((rxbytes - instance->curdevice->rxbytes)*8)/(gfloat)(timeframe*1024); 
							instance->curdevice->speedvalues[1][DBUS_API_SPEED_VALUES_NUMBER-1] = (gfloat)((txbytes - instance->curdevice->txbytes)*8)/(gfloat)(timeframe*1024);
						}
					}
					instance->curdevice->rxbytes = rxbytes;
					instance->curdevice->txbytes = txbytes;
				} else {
					instance->curdevice->rxbytes = 0;
					instance->curdevice->txbytes = 0;
				}
				instance->curdevice->speedchecktime = currenttime;
				
				instance->curdevice->connected = nm_get_device_enabled(instance->nm);
				
				interface = nm_get_device_interface(instance->nm);
				if (instance->curdevice->netinterface != NULL) g_free(instance->curdevice->netinterface);
				if (interface != NULL) {
					instance->curdevice->netinterface = g_strdup(interface);
				} else {
					instance->curdevice->connected = FALSE;
					instance->curdevice->netinterface = NULL;
				}
				
				if (!instance->curdevice->connected) {
					instance->curdevice->speedindex = 0;
					instance->curdevice->speedvalues[0][0] = 0.0;
					instance->curdevice->speedvalues[1][0] = 0.0;
					instance->curdevice->rxbytes = 0;
					instance->curdevice->txbytes = 0;
				}
				
				(instance->callback)(DBUS_API_EVENT_NET_STATUS, instance->curdevice);
			} else {
				instance->curdevice->speedchecktime = currenttime;
				(instance->callback)(DBUS_API_EVENT_NET_STATUS, NULL);
			}
		} else {
			if (instance->curdevice != NULL) {
				instance->curdevice->speedchecktime = currenttime;
			}
			(instance->callback)(DBUS_API_EVENT_NET_STATUS, NULL);
		}
		
				
		if ((instance->smsproxy != NULL) && (gmm_dbus_api_device_enabled(instance)) && (!instance->smsdisabled)) {
			if ((instance->curdevice->smschecktime == 0) || (difftime(currenttime, instance->curdevice->smschecktime) >= DBUS_API_THREAD_SMS_CHECK_INTERVAL)) {
				smspartlist = NULL;
				smsfulllist = NULL;
				
				error = NULL;
				
				info = g_dbus_proxy_call_sync(instance->smsproxy, "List", NULL, 0, -1, NULL, &error);
				
				if ((info != NULL) && (error == NULL)) {
					g_variant_iter_init(&iter, info);
					
					while ((child = g_variant_iter_next_value(&iter)) != NULL) {
						g_variant_iter_init(&iter2, child);
						while ((anode = g_variant_iter_next_value(&iter2)) != NULL) {
							sms = g_new(struct _sms_message, 1);
							pvalue = g_variant_lookup_value(anode, "number", G_VARIANT_TYPE_STRING);
							if (instance->smsdebug) g_printf("New SMS:\n");
							if (pvalue != NULL) {
								sms->number = (gchar *)g_variant_get_string(pvalue, &strlength);
								sms->number = strdup(sms->number);
								g_variant_unref(pvalue);
							} else {
								sms->number = g_strdup(_("Unknown"));
							}
							if (instance->smsdebug) g_printf(" Number: '%s'\n", sms->number);
							
							pvalue = g_variant_lookup_value(anode, "smsc", G_VARIANT_TYPE_STRING);
							if (pvalue != NULL) {
								sms->snumber = (gchar *)g_variant_get_string(pvalue, &strlength);
								sms->snumber = strdup(sms->snumber);
								g_variant_unref(pvalue);
							} else {
								sms->snumber = g_strdup(_("Unknown"));
							}
							if (instance->smsdebug) g_printf(" Service number: '%s'\n", sms->snumber);
							
							pvalue = g_variant_lookup_value(anode, "text", G_VARIANT_TYPE_STRING);
							if (pvalue != NULL) {
								sms->text = (gchar *)g_variant_get_string(pvalue, &strlength);
								sms->text = strdup(sms->text);
								g_variant_unref(pvalue);
							} else {
								sms->text = g_strdup(_("Unknown"));
							}
							if (instance->smsdebug) g_printf(" Text: '%s'\n", sms->text);
							
							pvalue = g_variant_lookup_value(anode, "timestamp", G_VARIANT_TYPE_STRING);
							if (pvalue != NULL) {
								timestr = g_variant_get_string(pvalue, &strlength);
								sms->timestamp = gmm_dbus_api_str_to_time(timestr);
							} else {
								sms->timestamp = time(NULL);
							}
							if (instance->smsdebug) g_printf(" Timestamp: '%s'\n", timestr);
							
							gsize datasize;
							const guchar *data;
							gint i;
							guint32 dcs;
							
							pvalue = g_variant_lookup_value(anode, "data", G_VARIANT_TYPE_BYTESTRING);
							if (pvalue != NULL) {
								datasize = g_variant_get_size(pvalue);
								if (datasize > 0) {
									if (instance->smsdebug) g_printf(" Binary data: ");
									data = (const guchar *)g_variant_get_data(pvalue);
									for (i=0; i<datasize; i++) {
										if (data[i] < 0x10) {
											if (instance->smsdebug) g_printf("0%1x ", data[i]);
										} else {
											if (instance->smsdebug) g_printf("%2x ", data[i]);
										}
									}
								} else {
									if (instance->smsdebug) g_printf("<Empty vector>");
								}
								if (instance->smsdebug) g_printf("\n");
							}
							
							pvalue = g_variant_lookup_value(anode, "data-coding-scheme", G_VARIANT_TYPE_UINT32);
							if (pvalue != NULL) {
								dcs = g_variant_get_uint32(pvalue);
								if (instance->smsdebug) g_printf(" DCS: '%u'\n", dcs);
							}
							
							pvalue = g_variant_lookup_value(anode, "index", G_VARIANT_TYPE_UINT32);
							if (pvalue != NULL) {
								sms->id = g_variant_get_uint32(pvalue);
								smspartlist = g_slist_prepend(smspartlist, sms);
								g_variant_unref(pvalue);
								if (instance->smsdebug) g_printf(" ID: '%u'\n\n", sms->id);
							} else {
								if (sms->number != NULL) g_free(sms->number);
								if (sms->snumber != NULL) g_free(sms->snumber);
								if (sms->text != NULL) g_free(sms->text);
								g_free(sms);
								//g_warning("Failed to read SMS message index\n");
							}
							g_variant_unref(anode);
						}
						g_variant_unref(child);
					}
					g_variant_unref(info);
					
					if (smspartlist != NULL) {
						//smspartlist = g_slist_sort(smspartlist, gmm_dbus_api_sms_message_sort_compare);
						smspartlist = g_slist_sort(smspartlist, gmm_dbus_api_sms_message_sort_index_compare);
						g_slist_foreach(smspartlist, gmm_dbus_api_sms_message_merge_foreach, &smsfulllist);
						g_slist_foreach(smspartlist, gmm_dbus_api_free_part_sms_messages_foreach, NULL);
						g_slist_free(smspartlist);
					}
					
					if (smsfulllist != NULL) {
						smsfulllist = g_slist_reverse(smsfulllist);
						
						smsnotify = g_new(struct _sms_notification, 1); 
						smsnotify->instance = instance;
						smsnotify->device = instance->curdevice;
						smsnotify->idents = g_array_new(FALSE, FALSE, sizeof(gulong));
						
						g_slist_foreach(smsfulllist, gmm_dbus_api_sms_add_foreach, smsnotify/*instance*/);
						//g_slist_foreach(smsfulllist, gmm_dbus_api_sms_build_notification, smsnotify);
						g_slist_foreach(smsfulllist, gmm_dbus_api_free_full_sms_messages_foreach, NULL);
						g_slist_free(smsfulllist);
						
						(instance->callback)(DBUS_API_EVENT_SMS_ADDED, smsnotify);
					}
					instance->curdevice->smschecktime = currenttime;
				} else {
					//g_warning("DBus error: %s\n", error->message ? error->message : "Unknown");
					g_error_free(error);
					instance->curdevice->smschecktime = currenttime;
				}
			}
		}
		
		if ((instance->limits != NULL) && (instance->curdevice != NULL)) {
			if (instance->limits->trafficenabled) {
				if ((!instance->limits->trafficexecuted) && (instance->limits->trafficfull < (instance->curdevice->rxbytes+instance->curdevice->txbytes))) {
					instance->limits->trafficexecuted = TRUE;
					if (instance->limits->trafficaction == DBUS_API_EVENT_ACTION_DISCONNECT) {
						gmm_dbus_api_device_disconnect(instance);
					}
					(instance->callback)(DBUS_API_EVENT_TRAFFIC_LIMIT, NULL);
				}
			}
			
			if (instance->limits->timeenabled) {
				if ((!instance->limits->timeexecuted) && (instance->limits->timefull < instance->curdevice->sessiontime)) {
					instance->limits->timeexecuted = TRUE;
					if (instance->limits->timeaction == DBUS_API_EVENT_ACTION_DISCONNECT) {
						gmm_dbus_api_device_disconnect(instance);
					}
					(instance->callback)(DBUS_API_EVENT_TIME_LIMIT, NULL);
				}
			}
		}
		
	}
	
	close(instance->threadctl[1]);
	
	return NULL;
}

static void gmm_dbus_api_async_send_sms_handler(GDBusProxy *proxy, GAsyncResult *res, gpointer user_data)
{
	dbus_api_t instance;
	GVariant *result;
	gboolean sent;
	
	instance = (dbus_api_t)user_data;
	
	if (instance == NULL) return;
	
	gmm_dbus_api_free_error(instance);
	
	result = g_dbus_proxy_call_finish(proxy, res, &instance->error);
	
	if ((result != NULL) && (instance->error == NULL)) {
		//g_variant_get(result, "(s)", &instance->curdevice->ussdanswer);
		g_variant_unref(result);
		sent = TRUE;
	} else {
		sent = FALSE;
	}
	
	(instance->callback)(DBUS_API_EVENT_SMS_SENT, GUINT_TO_POINTER(sent));
}

gboolean gmm_dbus_api_send_sms(dbus_api_t instance, gchar *number, gchar *text)
{
	GVariantBuilder *builder;
	GVariant *array, *message;
	gsize length;
	gint i;
	
	if ((instance == NULL) || (number == NULL) || (text == NULL)) return FALSE;
	
	if (instance->connection == NULL) return FALSE;
	if (instance->curdevice == NULL) return FALSE;
	if (instance->smsproxy == NULL) return FALSE;
	
	if (!instance->curdevice->enabled) return FALSE;
	
	if (!gmm_dbus_api_validate_sms_number(number)) return FALSE;
	
	length = strlen(text);
	
	if (length == 0) return FALSE;
	
	builder = g_variant_builder_new(G_VARIANT_TYPE_ARRAY);
	g_variant_builder_add_parsed(builder, "{'number', <%s>}", number);
	g_variant_builder_add_parsed(builder, "{'text', <%s>}", text);
	
	array = g_variant_builder_end(builder);
	
	builder = g_variant_builder_new(G_VARIANT_TYPE_TUPLE);
	g_variant_builder_add_value(builder, array);
	
	message = g_variant_builder_end(builder);
	
	g_dbus_proxy_call(instance->smsproxy,
                     "Send",
                     message,
                     G_DBUS_CALL_FLAGS_NONE,
                     DBUS_API_ASYNC_SEND_SMS_TIMEOUT,
                     NULL,
                     (GAsyncReadyCallback)gmm_dbus_api_async_send_sms_handler,
                     instance);
	
	g_variant_unref(array);
	g_variant_unref(message);
	
	return TRUE;
}

gboolean gmm_dbus_api_validate_sms_number(gchar *number)
{
	gboolean validated;
	gsize length, digitlength, i;
	
	validated = TRUE;
	
	if ((number != NULL) && (number[0] != '\0')) {
		length = strlen(number);
		if (number[0] == '+') {
			digitlength = length-1;
		} else {
			digitlength = length;
		} 
		if ((digitlength < DBUS_API_MIN_SMS_NUMBER_LENGTH) || (digitlength > DBUS_API_MAX_SMS_NUMBER_LENGTH)) {
			validated = FALSE;
		} else {
			for (i=0; i<length; i++) {
				if (((!isdigit(number[i])) && (number[i] != '+')) || ((i != 0) && (number[i] == '+'))) {
					validated = FALSE;
					break;
				}
			}
		}
	} else {
		validated = FALSE;
	}
	
	return validated;
}

static gchar *gmm_dbus_api_gsm7_to_ucs2(gchar *srcstr)
{
	gchar *ussddecstr;
	gsize strsize;
	
	if (srcstr == NULL) return NULL;
	
	srcstr = g_strdup(srcstr);
	strsize = strlen(srcstr);
	if (strsize > 0) {
		ussddecstr = utf8_map_gsm7(srcstr, strsize, &strsize);
		if (ussddecstr != NULL) {
			g_free(srcstr);
			srcstr = utf8_to_gsm7(ussddecstr, strsize, &strsize);
			if (srcstr != NULL) {
				g_free(ussddecstr);
				ussddecstr = ucs2_to_utf8(srcstr, strsize, &strsize);
				if (ussddecstr != NULL) {
					g_free(srcstr);
					srcstr = ussddecstr;
				}
			}
		}
	}
	
	return srcstr;
}

guint gmm_dbus_api_validate_ussd_request(gchar *request)
{
	guint statusid;
	gsize length, i;
	gboolean wrongsym;
	
	statusid = DBUS_API_USSD_VALIDATION_INVALID;
	
	if ((request == NULL) && (request[0] == '\0')) return statusid;
	
	length = strlen(request);
	
	if (length > DBUS_API_MAX_USSD_REQUEST_LENGTH) return statusid;
	
	if ((request[0] == '*') && (request[length-1] == '#') && (length > 2)) {
		wrongsym = FALSE;
		for (i=0; i<length; i++) {
			if ((!isdigit(request[i])) && (request[i] != '*') && (request[i] != '#')) {
				wrongsym = TRUE;
				break;
			}
		}
		if (!wrongsym) {
			statusid = DBUS_API_USSD_VALIDATION_REQUEST;
		} else {
			statusid = DBUS_API_USSD_VALIDATION_RESPONSE;
		}
	} else {
		statusid = DBUS_API_USSD_VALIDATION_RESPONSE;
	}
	
	return statusid;
}

void gmm_dbus_api_cancel_ussd_session(dbus_api_t instance)
{
	if (instance == NULL) return;
	if (instance->ussdproxy == NULL) return;
	
	gmm_dbus_api_free_error(instance);
	
	g_dbus_proxy_call_sync(instance->ussdproxy, "Cancel", NULL, 0, -1, NULL, &instance->error);
	
	if (instance->error != NULL) {
		g_warning("USSD answer error: %s\n", (instance->error->message != NULL) ? instance->error->message : "Unknown");
	}
}

guint gmm_dbus_api_get_ussd_state(dbus_api_t instance)
{
	GVariant *session;
	gchar *state;
	guint stateid;
	gsize strsize;
	
	stateid = DBUS_API_USSD_STATE_UNKNOWN;
	
	if (instance == NULL) return stateid;
	if (instance->connection == NULL) return stateid;
	if (instance->curdevice == NULL) return stateid;
	if (instance->ussdproxy == NULL) return stateid;
	
	if (!instance->curdevice->enabled) return stateid;
	
	session = g_dbus_proxy_get_cached_property(instance->ussdproxy, "State");
	
	if (session == NULL) return stateid;
		
	state = (gchar *)g_variant_get_string(session, &strsize);
	
	if (state != NULL) {
		if (g_str_equal(state, "idle")) {
			stateid = DBUS_API_USSD_STATE_IDLE;
		} else if (g_str_equal(state, "active")) {
			stateid = DBUS_API_USSD_STATE_ACTIVE;
		} else if (g_str_equal(state, "user-response")) {
			stateid = DBUS_API_USSD_STATE_USER_RESPONSE;
		}
	}
	
	g_variant_unref(session);
	
	return stateid;
}

static void gmm_dbus_api_async_ussd_handler(GDBusProxy *proxy, GAsyncResult *res, gpointer user_data)
{
	dbus_api_t instance;
	GVariant *result;
	
	instance = (dbus_api_t)user_data;
	
	if (instance == NULL) return;
	
	gmm_dbus_api_free_error(instance);
	
	result = g_dbus_proxy_call_finish(proxy, res, &instance->error);
	
	instance->curdevice->ussdanswer = NULL;
	
	if (result != NULL) {
		g_variant_get(result, "(s)", &instance->curdevice->ussdanswer);
		
		if (instance->curdevice->ussdencoding == DBUS_API_ENCODING_UCS2) {
			instance->curdevice->ussdanswer = gmm_dbus_api_gsm7_to_ucs2(instance->curdevice->ussdanswer);
		} else {
			instance->curdevice->ussdanswer = g_strdup(instance->curdevice->ussdanswer);
		}
		g_variant_unref(result);
	}
	
	(instance->callback)(DBUS_API_EVENT_USSD_RESULT, instance->curdevice);
}

gboolean gmm_dbus_api_send_ussd_async(dbus_api_t instance, gchar *request, enum _dbus_api_encoding encoding)
{
	GVariant *ussdreq;
	guint validationid, sessionstate;
	gchar *command;
	
	if ((instance == NULL) || (request == NULL)) return FALSE;
	if (instance->connection == NULL) return FALSE;
	if (instance->curdevice == NULL) return FALSE;
	if (instance->ussdproxy == NULL) return FALSE;
	
	if (!instance->curdevice->enabled) return FALSE;
	
	instance->curdevice->ussdencoding = encoding;
	
	validationid = gmm_dbus_api_validate_ussd_request(request);
	
	if (validationid == DBUS_API_USSD_VALIDATION_INVALID) return FALSE;
	
	sessionstate = gmm_dbus_api_get_ussd_state(instance);
	
	if ((sessionstate == DBUS_API_USSD_STATE_UNKNOWN) || (sessionstate == DBUS_API_USSD_STATE_ACTIVE)) return FALSE;
	
	gmm_dbus_api_free_error(instance);
	
	ussdreq = g_variant_new("(s)", request);
	
	if (sessionstate == DBUS_API_USSD_STATE_IDLE) {
		command = "Initiate";
	} else if (sessionstate == DBUS_API_USSD_STATE_USER_RESPONSE) {
		if (validationid == DBUS_API_USSD_VALIDATION_REQUEST) {
			gmm_dbus_api_cancel_ussd_session(instance);
			command = "Initiate";
		} else {
			command = "Respond";
		}
	}
	
	g_dbus_proxy_call(instance->ussdproxy,
                     command,
                     ussdreq,
                     G_DBUS_CALL_FLAGS_NONE,
                     DBUS_API_ASYNC_USSD_TIMEOUT,
                     NULL,
                     (GAsyncReadyCallback)gmm_dbus_api_async_ussd_handler,
                     instance);
                     
   //g_variant_unref(ussdreq);
	
	return TRUE;
}

GSList *gmm_dbus_api_get_networks(dbus_api_t instance)
{
	if (instance == NULL) return NULL;
	if (instance->curdevice == NULL) return NULL;
	
	return instance->curdevice->netlist;
}

static void gmm_dbus_api_free_networks_foreach(gpointer data, gpointer user_data)
{
	network_t network;
	
	if (data != NULL) return;
	
	network = (network_t)data;
	
	if (network->operator_short != NULL) {
		g_free(network->operator_short);
		network->operator_short = NULL;
	}
	
	if (network->operator_long != NULL) {
		g_free(network->operator_long);
		network->operator_long = NULL;
	}
}

static gboolean gmm_dbus_api_free_networks(dbus_api_t instance)
{
	if (instance == NULL) return FALSE;
	if (instance->curdevice == NULL) return FALSE;
	if (instance->curdevice->netlist == NULL) return TRUE;
	
	g_slist_foreach(instance->curdevice->netlist, gmm_dbus_api_free_networks_foreach, NULL);
	
	g_slist_free(instance->curdevice->netlist);
	
	instance->curdevice->netlist = NULL;
	
	return TRUE;
}

static void gmm_dbus_api_async_scan_handler(GDBusProxy *proxy, GAsyncResult *res, gpointer user_data)
{
	dbus_api_t instance;
	GVariant *result;
	network_t network;
	GVariantIter iter, iter2;
	GVariant *child, *anode, *pvalue;
	gsize strlength = 256;
	gchar *tmpstr;
	
	instance = (dbus_api_t)user_data;
	
	if (instance == NULL) return;
	
	gmm_dbus_api_free_error(instance);
	
	result = g_dbus_proxy_call_finish(proxy, res, &instance->error);
	
	if (instance->curdevice->netlist != NULL) {
		gmm_dbus_api_free_networks(instance);
	}
	
	if (result != NULL) {
		g_variant_iter_init(&iter, result);
		while ((child = g_variant_iter_next_value(&iter)) != NULL) {
			g_variant_iter_init(&iter2, child);
			while ((anode = g_variant_iter_next_value(&iter2)) != NULL) {
				network = g_new(struct _network, 1);
				pvalue = g_variant_lookup_value(anode, "operator-num", G_VARIANT_TYPE_STRING);
				if (pvalue != NULL) {
					tmpstr = (gchar *)g_variant_get_string(pvalue, &strlength);
					network->operator_num = atoi(tmpstr);
					g_variant_unref(pvalue);
				} else {
					network->operator_num = 0;
				}
				
				pvalue = g_variant_lookup_value(anode, "status", G_VARIANT_TYPE_STRING);
				if (pvalue != NULL) {
					tmpstr = (gchar *)g_variant_get_string(pvalue, &strlength);
					network->status = atoi(tmpstr);
					g_variant_unref(pvalue);
				} else {
					network->status = 0;
				}
				
				pvalue = g_variant_lookup_value(anode, "access-tech", G_VARIANT_TYPE_STRING);
				if (pvalue != NULL) {
					tmpstr = (gchar *)g_variant_get_string(pvalue, &strlength);
					network->access_tech = atoi(tmpstr);
					g_variant_unref(pvalue);
				} else {
					network->access_tech = 0;
				}
				
				pvalue = g_variant_lookup_value(anode, "operator-long", G_VARIANT_TYPE_STRING);
				if (pvalue != NULL) {
					tmpstr = (gchar *)g_variant_get_string(pvalue, &strlength);
					network->operator_long = g_strdup(tmpstr);
				} else {
					network->operator_long = g_strdup(_("Unknown"));
				}
				
				pvalue = g_variant_lookup_value(anode, "operator-short", G_VARIANT_TYPE_STRING);
				if (pvalue != NULL) {
					tmpstr = (gchar *)g_variant_get_string(pvalue, &strlength);
					network->operator_short = g_strdup(tmpstr);
				} else {
					network->operator_short = g_strdup(_("Unknown"));
				}
				
				instance->curdevice->netlist = g_slist_prepend(instance->curdevice->netlist, network);
				
				g_variant_unref(anode);
			}
			g_variant_unref(child);
		}
		g_variant_unref(result);
		
		if (instance->curdevice->netlist != NULL) {
			instance->curdevice->netlist = g_slist_reverse(instance->curdevice->netlist);
		}
	}
		
	(instance->callback)(DBUS_API_EVENT_SCAN_RESULT, instance->curdevice);
}

gboolean ggmm_dbus_api_scan_networks_async(dbus_api_t instance)
{
	GVariant *info;
	
	if (instance == NULL) return FALSE;
	if (instance->connection == NULL) return FALSE;
	if (instance->curdevice == NULL) return FALSE;
	if (instance->netproxy == NULL) return FALSE;
	
	if (!instance->curdevice->enabled) return FALSE;
	
	g_dbus_proxy_call(instance->netproxy,
                     "Scan",
                     NULL,
                     G_DBUS_CALL_FLAGS_NONE,
                     DBUS_API_ASYNC_SCAN_TIMEOUT,
                     NULL,
                     (GAsyncReadyCallback)gmm_dbus_api_async_scan_handler,
                     instance);
	
	return TRUE;
}
