# -*- Mode: Python; coding: utf-8; indent-tabs-mode: nil; tab-width: 4 -*-
### BEGIN LICENSE
# Copyright (C) 2012 Brian Douglass bhdouglass@gmail.com
# This program is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License version 3, as published 
# by the Free Software Foundation.
# 
# This program is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranties of 
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
# PURPOSE.  See the GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License along 
# with this program.  If not, see <http://www.gnu.org/licenses/>.
### END LICENSE

from gi.repository import Gio, Gtk, GObject, Notify # pylint: disable=E0611
import logging
logger = logging.getLogger('indicator_remindor_lib')

from pygame import mixer
import time, datetime
from datetime import timedelta, date
from dateutil import parser
import subprocess
from os.path import basename

from desktopcouch.records.server import CouchDatabase
from desktopcouch.records.record import Record

from indicator_remindor.AlarmDialog import AlarmDialog
from indicator_remindor.QuickDialog import QuickDialog

from indicator_remindor_lib.helpers import get_builder, show_uri, get_help_uri
from indicator_remindor_lib.indicator_remindorconfig import get_data_file

setup_schedule = True
setup_tree_view = True
alarm_schedule = []
playing_sound = None

# This class is meant to be subclassed by IndicatorRemindorWindow.  It provides
# common functions and some boilerplate.
class Window(Gtk.Window):
    __gtype_name__ = "Window"

    # To construct a new instance of this method, the following notable
    # methods are called in this order:
    # __new__(cls)
    # __init__(self)
    # finish_initializing(self, builder)
    # __init__(self)
    #
    # For this reason, it's recommended you leave __init__ empty and put
    # your initialization code in finish_initializing

    def __new__(cls):
        """Special static method that's automatically called by Python when
        constructing a new instance of this class.

        Returns a fully instantiated BaseIndicatorRemindorWindow object.
        """
        builder = get_builder('IndicatorRemindorWindow')
        new_object = builder.get_object("indicator_remindor_window")
        new_object.finish_initializing(builder)
        return new_object

    def finish_initializing(self, builder):
        """Called while initializing this instance in __new__

        finish_initializing should be called after parsing the UI definition
        and creating a IndicatorRemindorWindow object with it in order to finish
        initializing the start of the new IndicatorRemindorWindow instance.
        """
        # Get a reference to the builder and set up the signals.
        self.builder = builder
        self.ui = builder.get_ui(self, True)
        self.PreferencesDialog = None # class
        self.preferences_dialog = None # instance
        self.AboutDialog = None # class

        #self.settings = Gio.Settings("net.launchpad.indicator-remindor")
        #self.settings.connect('changed', self.on_preferences_changed)

        Notify.init("indicator-remindor")
        mixer.init()
        mixer.init()

        self.connect("delete-event", self.on_destroy)
        self.update_alarms()

        GObject.timeout_add(1000 * 60 * 60 * 12, self.update_alarm_schedule) #update everything every 1/2 daily
        #GObject.timeout_add(1000 * 60, self.update_alarm_schedule) #update minutly

        # Optional Launchpad integration
        # This shouldn't crash if not found as it is simply used for bug reporting.
        # See https://wiki.ubuntu.com/UbuntuDevelopment/Internationalisation/Coding
        # for more information about Launchpad integration.
        try:
            from gi.repository import LaunchpadIntegration # pylint: disable=E0611
            LaunchpadIntegration.add_items(self.ui.helpMenu, 1, True, True)
            LaunchpadIntegration.set_sourcepackagename('indicator-remindor')
        except ImportError:
            pass

        # Optional application indicator support
        # Run 'quickly add indicator' to get started.
        # More information:
        #  http://owaislone.org/quickly-add-indicator/
        #  https://wiki.ubuntu.com/DesktopExperienceTeam/ApplicationIndicators
        try:
            from indicator_remindor import indicator
            # self is passed so methods of this class can be called from indicator.py
            # Comment this next line out to disable appindicator
            self.indicator = indicator.new_application_indicator(self)
        except ImportError:
            pass

    def on_mnu_contents_activate(self, widget, data=None):
        show_uri(self, "ghelp:%s" % get_help_uri())

    def on_mnu_about_activate(self, widget, data=None):
        """Display the about box for indicator-remindor."""
        if self.AboutDialog is not None:
            about = self.AboutDialog() # pylint: disable=E1102
            response = about.run()
            about.destroy()

    #def on_mnu_preferences_activate(self, widget, data=None):
        """Display the preferences window for indicator-remindor."""

        """ From the PyGTK Reference manual
           Say for example the preferences dialog is currently open,
           and the user chooses Preferences from the menu a second time;
           use the present() method to move the already-open dialog
           where the user can see it."""
        #if self.preferences_dialog is not None:
            #logger.debug('show existing preferences_dialog')
            #self.preferences_dialog.present()
        #elif self.PreferencesDialog is not None:
            #logger.debug('create new preferences_dialog')
            #self.preferences_dialog = self.PreferencesDialog() # pylint: disable=E1102
            #self.preferences_dialog.connect('destroy', self.on_preferences_dialog_destroyed)
            #self.preferences_dialog.show()
        # destroy command moved into dialog to allow for a help button

    def on_mnu_close_activate(self, widget, data=None):
        """Signal handler for closing the IndicatorRemindorWindow."""
        #self.destroy()
        self.hide_on_delete()
        return True

	def on_mnu_quit_activate(self, widget, data=None):
		self.on_indicator_quit_activate(widget, data=None)

	def on_indicator_quit_activate(self, widget, data=None):
		Gtk.main_quit()

    def on_destroy(self, widget, data=None):
        self.hide_on_delete()
		return True

	def on_indicator_manage_activate(self, widget, data=None):
		self.show_all()

    #def on_preferences_changed(self, settings, key, data=None):
        #logger.debug('preference changed: %s = %s' % (key, str(settings.get_value(key))))

    #def on_preferences_dialog_destroyed(self, widget, data=None):
        '''only affects gui

        logically there is no difference between the user closing,
        minimising or ignoring the preferences dialog'''
        #logger.debug('on_preferences_dialog_destroyed')
        # to determine whether to create or present preferences_dialog
        #self.preferences_dialog = None

	def on_indicator_sync_activate(self, widget, data=None):
		print "sync"

	def on_mnu_quick_add_activate(self, widget, data=None):
		self.on_quick_add_clicked(widget, data)

	def on_quick_add_clicked(self, widget, data=None):
		quick_dialog = QuickDialog()
		quick_dialog.connect('alarm_added', self.add_to_schedule_slot)
		quick_dialog.run()
		quick_dialog.destroy()
		self.update_alarms()

	def on_mnu_add_activate(self, widget, data=None):
		self.on_add_clicked(widget, data)

	def on_add_clicked(self, widget, data=None):
		alarm_dialog = AlarmDialog()
		alarm_dialog.connect('alarm_added', self.add_to_schedule_slot)
		alarm_dialog.run()
		alarm_dialog.destroy()
		self.update_alarms()

	def on_mnu_delete_activate(self, widget, data=None):
		self.delete_selected()

	def on_delete_clicked(self, widget, data=None):
		self.delete_selected()

	def on_mnu_edi_activate(self, widget, data=None):
		self.on_edit_clicked(widget, data)

	def on_edit_clicked(self, widget, data=None):
		alarm_id, iter = self.get_selected_alarm()
		if alarm_id is not None:
			alarm_dialog = AlarmDialog()
			alarm_dialog.set_edit()
			alarm_dialog.connect('alarm_added', self.add_to_schedule_slot)

			model = self.builder.get_object("alarm_tree").get_model()
			id = model.get_value(iter, 7)

			db = CouchDatabase('indicator-remindor-alarms', create=True)
			rec = db.get_record(id)

			length = 10
			loop = False
			try:
				length = rec["sound_length"]
			except KeyError:
				length = 10
			try:
				loop = rec["sound_loop"]
			except KeyError:
				loop = False

			alarm_dialog.set_all(rec["label"], rec["notes"], rec["hour"], rec["minute"],
				rec["am_pm"], rec["date"], rec["sound"], length, loop, rec["notification"], rec["command"])

			response = alarm_dialog.run()

			if response == Gtk.ResponseType.OK:
				self.delete_selected()

			alarm_dialog.destroy()

	def on_mnu_postpone_activate(self, widget, data=None):
		self.on_postpone_clicked(widget, data)

	def on_postpone_clicked(self, widget, data=None):
		alarm_id, iter = self.get_selected_alarm()
		if alarm_id is not None:
			db = CouchDatabase('indicator-remindor-alarms', create=True)
			rec = db.get_record(alarm_id)

			if rec is not None:
				time_s = self.format_time(rec["hour"], rec["minute"], rec["am_pm"])
				date = time_s + " " + rec["date"]
				d = self.parse_date(time_s, date) + timedelta(minutes=30)
				date = d.strftime("%m/%d/%Y")

				length = 10
				loop = False
				try:
					length = rec["sound_length"]
				except KeyError:
					length = 10
				try:
					loop = rec["sound_loop"]
				except KeyError:
					loop = False

				rec2 = Record({
					"label" : rec["label"],
					"notes" : rec["notes"],
					"hour" : d.strftime("%I"),
					"minute" : d.strftime("%M"),
					"am_pm" : d.strftime("%p"),
					"date" : d.strftime("%m/%d/%Y"),
					"sound" : rec["sound"],
					"sound_length" : length,
					"sound_loop" : loop,
					"notification" : rec["notification"],
					"command" : rec["command"]
				}, record_type='http://wiki.ubuntu.com/Quickly/RecordTypes/IndicatorRemindor/')

				id = db.put_record(rec2)
				self.add_to_schedule(time_s, date, rec["label"], rec["notes"], rec["sound"],
					length, loop, rec["notification"], rec["command"], id)
				db.delete_record(alarm_id)

		self.update_alarms()

	def delete_selected(self):
		alarm_id, iter = self.get_selected_alarm()
		if alarm_id is not None:
			db = CouchDatabase('indicator-remindor-alarms', create=True)
			db.delete_record(alarm_id)

			self.update_alarms()

	def get_selected_alarm(self):
		alarm_tree = self.builder.get_object("alarm_tree")
		model, row = alarm_tree.get_selection().get_selected()

		index = None
		if row is not None:
			if model.iter_is_valid(row):
				alarm_id = model.get_value(row, 7)

		return alarm_id, row

	def update_alarm_schedule(self):
		global setup_schedule
		setup_schedule = True

		for alarm in alarm_schedule:
			GObject.source_remove(alarm)

		logger.debug("updating the whole schedule")

		self.update_alarms()

		return True

	def update_alarms(self):
		db = CouchDatabase('indicator-remindor-alarms', create=True)

        if not db.view_exists("list", "app"):
			map_js = """function(doc) { emit(doc._id, doc) }"""
			db.add_view("list", map_js, None, "app")

		result = db.execute_view("list", "app")
		result_list = list(result)

        alarm_tree = self.builder.get_object("alarm_tree")
        alarm_list_store = Gtk.ListStore(GObject.TYPE_STRING, GObject.TYPE_STRING,
			GObject.TYPE_STRING, GObject.TYPE_STRING, GObject.TYPE_STRING,
			GObject.TYPE_STRING, GObject.TYPE_STRING, GObject.TYPE_STRING,
			GObject.TYPE_STRING, GObject.TYPE_DOUBLE)
		alarm_list_store.set_sort_func(9, self.sort_alarms)
		alarm_list_store.set_sort_column_id(9, Gtk.SortType.DESCENDING)

		global setup_schedule

		for row in result_list:
			time_s = self.format_time(row.value["hour"], row.value["minute"], row.value["am_pm"])

			t = self.timestamp(self.parse_date(time_s, row.value["date"]))
			d = self.parse_date(None, row.value["date"])

			color = "#FFFFFF"
			if datetime.date.today().strftime("%m/%d/%Y") == d.strftime("%m/%d/%Y"):
				color = "#3D8B37"
			if t < self.timestamp(datetime.datetime.now()):
				color = "#999999"
				if "every" in row.value["date"]:
					color = "#FFFFFF"
				elif "next" in row.value["date"]:
					dl = row.value["date"].split()
					today = date.today()
					d2 = parser.parse(dl[4]).date() + timedelta(days=int(dl[1]))
					if d2 > today:
						color = "#FFFFFF"

			notify = "Yes"
			if row.value["notification"] == "False":
				notify = "No"

			if setup_schedule:
				length = 10
				loop = False
				try:
					length = row.value["sound_length"]
				except KeyError:
					length = 10
				try:
					loop = row.value["sound_loop"]
				except KeyError:
					loop = False

				self.add_to_schedule(time_s, row.value["date"], row.value["label"],
					row.value["notes"], row.value["sound"], length, loop,
					row.value["notification"], row.value["command"], row.value["_id"])

			note = row.value["notes"]
			if len(note) > 20:
				note = note[:20] + "..."

			sound = row.value["sound"]
			if sound != "" and sound != None:
				sound = basename(row.value["sound"])

			command = row.value["command"]
			if len(command) > 20:
				command = command[:20] + "..."

			alarm_list_store.append([row.value["label"], note, time_s, row.value["date"],
				sound, notify, command, row.value["_id"], color, t])

		if setup_schedule:
			setup_schedule = False

		alarm_tree.set_model(alarm_list_store)

		global setup_tree_view
		if setup_tree_view:
			setup_tree_view = False

			cell0 = Gtk.CellRendererText()
			cell0.set_property('background-set' , True)
			label_col = Gtk.TreeViewColumn("Label", cell0, text=0, background=8)
			label_col.set_min_width(10)
			label_col.set_resizable(True)
			label_col.set_spacing(0)
			alarm_tree.append_column(label_col)

			cell1 = Gtk.CellRendererText()
			cell1.set_property('background-set' , True)
			notes_col = Gtk.TreeViewColumn("Notes", cell1, text=1, background=8)
			notes_col.set_min_width(10)
			notes_col.set_resizable(True)
			notes_col.set_spacing(0)
			alarm_tree.append_column(notes_col)

			cell2 = Gtk.CellRendererText()
			cell2.set_property('background-set' , True)
			time_col = Gtk.TreeViewColumn("Time", cell2, text=2, background=8)
			time_col.set_min_width(10)
			time_col.set_resizable(True)
			time_col.set_spacing(0)
			alarm_tree.append_column(time_col)

			cell3 = Gtk.CellRendererText()
			cell3.set_property('background-set' , True)
			date_col = Gtk.TreeViewColumn("Date", cell3, text=3, background=8)
			date_col.set_min_width(10)
			date_col.set_resizable(True)
			date_col.set_spacing(0)
			alarm_tree.append_column(date_col)

			cell4 = Gtk.CellRendererText()
			cell4.set_property('background-set' , True)
			sound_col = Gtk.TreeViewColumn("Sound", cell4, text=4, background=8)
			sound_col.set_min_width(10)
			sound_col.set_resizable(True)
			sound_col.set_spacing(0)
			alarm_tree.append_column(sound_col)

			cell5 = Gtk.CellRendererText()
			cell5.set_property('background-set' , True)
			notification_col = Gtk.TreeViewColumn("Notification", cell5, text=5, background=8)
			notification_col.set_min_width(10)
			notification_col.set_resizable(True)
			notification_col.set_spacing(0)
			alarm_tree.append_column(notification_col)

			cell6 = Gtk.CellRendererText()
			cell6.set_property('background-set' , True)
			command_col = Gtk.TreeViewColumn("Command", cell6, text=6, background=8)
			command_col.set_min_width(10)
			command_col.set_resizable(True)
			command_col.set_spacing(0)
			alarm_tree.append_column(command_col)

		return True

	def stop_sound(self, widget, data=None):
		global playing_sound
		if playing_sound != None:
			GObject.source_remove(playing_sound)
			playing_sound = None

		mixer.music.stop()

	def run_alarm(self, label, note, sound, length, loop, notification, command, id):
		db = CouchDatabase('indicator-remindor-alarms', create=True)
		rec = db.get_record(id)

		if rec is not None:
			logger.debug("run_alarm: " + label)

			if notification:
				logger.debug("notification: " + label + " " + note)

				icon = get_data_file("media", "indicator-remindor.svg")
				n = Notify.Notification.new(label, note, icon)
				n.set_timeout(10)
				n.show()

			if command != "":
				logger.debug("running: " + command)
				process = subprocess.Popen(command)

			if sound != "" and sound != None:
				logger.debug("playing: " + sound)
				logger.debug("loop: " + str(loop))
				logger.debug("length: " + str(length))

				if loop:
					loop = 5
				else:
					loop = 0

				global playing_sound
				if playing_sound != None:
					GObject.source_remove(playing_sound)
					playing_sound = None

				mixer.music.stop()
				mixer.music.load(sound)
				mixer.music.play(loop)

				if int(length) > 0 and loop == 0:
					playing_sound = GObject.timeout_add(int(length * 1000), self.stop_sound, None)

	def timestamp(self, date):
		return time.mktime(date.timetuple())

	def parse_date(self, t, d):
		try:
			parser.parse(d) #no need to format
		except ValueError:
			dl = d.split()
			today = date.today()
			format = "%m/%d/%Y"

			if dl[0] == "every":
				days = ["monday", "tuesday", "wednesday", "thursday", "friday",
					"saturday","sunday"]
				if dl[1] in days:
					d = dl[1] #ex: monday
				elif dl[1] == "weekday":
					w = int(today.strftime("%w"))
					if w is not 0 and w is not 6: #today is weekday
						d = today.strftime(format)
					else: #today is weekend
						d = "monday"
				elif dl[1] == "weekend":
					w = int(today.strftime("%w"))
					if w is not 0 and w is not 6: #today is weekday
						d = "saturday"
					else: #today is weekend
						d = today.strftime(format)
				elif dl[1] == "day":
					d = today.strftime("%m/%d/%Y")
				elif dl[1].isdigit() and len(dl) == 2:
					d = parser.parse(dl[1]).strftime(format)
				elif dl[1] == "other":
					d = parser.parse(dl[3]).date()
					while d < today:
						d += timedelta(days=2)
					d = d.strftime(format)
				elif dl[0] == "every" and dl[2] == "days":
					d = parser.parse(dl[4]).date()
					while d < today:
						d += timedelta(days=int(dl[1]))
					d = d.strftime(format)
			else: #dl[0] == "next"
				d = parser.parse(dl[4]).date()
				d2 = d + timedelta(days=int(dl[1]))
				if d <= today and d2 >= today:
					d = today.strftime(format)
				else:
					d = d.strftime(format)

		p = d
		if t is not None:
			p = t + " " + d
		return parser.parse(p)

	def format_time(self, hour, minute, am_pm):
		time_s = str(hour) + ":"
		if minute < 10:
			time_s += "0" + str(minute)
		else:
			time_s += str(minute)

		if am_pm is 0:
			time_s += " am"
		else:
			time_s += " pm"

		return time_s

	def add_to_schedule(self, time_s, date, label, note, sound, length, loop, notification, command, id):
		delay = self.timestamp(self.parse_date(time_s, date)) - self.timestamp(datetime.datetime.now())

		if delay > 0:
			alarm_schedule.append(GObject.timeout_add(int(delay * 1000), self.run_alarm, label, note, sound, length, loop, notification, command, id))

		logger.debug("timer for " + str(delay) + "s from now: " + label)

	def add_to_schedule_slot(self, widget, data=None):
		db = CouchDatabase('indicator-remindor-alarms', create=True)
		rec = db.get_record(data)

		length = 10
		loop = False
		try:
			length = rec["sound_length"]
		except KeyError:
			length = 10
		try:
			loop = rec["sound_loop"]
		except KeyError:
			loop = False

		self.add_to_schedule(self.format_time(rec["hour"], rec["minute"], rec["am_pm"]), rec["date"], rec["label"], rec["notes"], rec["sound"], length, loop, rec["notification"], rec["command"], data)

	def sort_alarms(self, model, row1, row2, data=None):
		sort_column, _ = model.get_sort_column_id()
		value1 = model.get_value(row1, sort_column)
		value2 = model.get_value(row2, sort_column)
		if value1 < value2:
			return -1
		elif value1 > value2:
			return 1
		else:
			return 0
