# -*- Mode: Python; coding: utf-8; indent-tabs-mode: nil; tab-width: 4 -*-
### BEGIN LICENSE
# Copyright (C) 2012 Brian Douglass bhdouglass@gmail.com
# This program is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License version 3, as published 
# by the Free Software Foundation.
# 
# This program is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranties of 
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
# PURPOSE.  See the GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License along 
# with this program.  If not, see <http://www.gnu.org/licenses/>.
### END LICENSE

from gi.repository import Gtk, GObject # pylint: disable=E0611

from datetime import date, timedelta, datetime
from dateutil import parser

from desktopcouch.records.server import CouchDatabase
from desktopcouch.records.record import Record

from indicator_remindor_lib.helpers import get_builder, show_uri, get_help_uri

import gettext
from gettext import gettext as _
gettext.textdomain('indicator-remindor')

class AlarmDialog(Gtk.Dialog):
    __gtype_name__ = "AlarmDialog"
    __gsignals__ = {
		'alarm_added' : (GObject.SIGNAL_RUN_LAST, GObject.TYPE_NONE,
			(GObject.TYPE_STRING,))
	}

    def __new__(cls):
        """Special static method that's automatically called by Python when
        constructing a new instance of this class.

        Returns a fully instantiated AlarmDialog object.
        """
        builder = get_builder('AlarmDialog')
        new_object = builder.get_object('alarm_dialog')
        new_object.finish_initializing(builder)
        return new_object

    def finish_initializing(self, builder):
        """Called when we're finished initializing.

        finish_initalizing should be called after parsing the ui definition
        and creating a AlarmDialog object with it in order to
        finish initializing the start of the new AlarmDialog
        instance.
        """
        # Get a reference to the builder and set up the signals.
        self.builder = builder
        self.ui = builder.get_ui(self)

        t = datetime.now()
        hour = t.hour
        am_pm = 0
        if hour > 12:
			hour -= 12
			am_pm = 1

        self.set_hour(hour)
        self.set_minute(t.minute)
        self.set_am_pm(am_pm)

        label_object = self.builder.get_object("label_entry")
        label_object.connect('icon-release', self.label_clear)

        date_object = self.builder.get_object("date_entry")
        date_object.connect('icon-release', self.date_clear)
        date_object.connect('changed', self.on_date_entry_changed)

        command_object = self.builder.get_object("command_entry")
        command_object.connect('icon-release', self.command_clear)

        sound_switch = self.builder.get_object("sound_switch")
        sound_switch.connect('notify::active', self.on_sound_switch_activate)

        loop_switch = self.builder.get_object("loop_entry")
        loop_switch.connect('notify::active', self.on_loop_switch_activate)

	def on_sound_clear_clicked(self, widget, data=None):
		self.set_sound(None, False)
		self.set_sound_length(10)
		self.set_loop(False)

    def on_btn_add_clicked(self, widget, data=None):
		label_object = self.builder.get_object("label_entry")
		label = label_object.get_text()

		hour_object = self.builder.get_object("hour_entry")
		hour = hour_object.get_value()

		minute_object = self.builder.get_object("minute_entry")
		minute = minute_object.get_value()

		am_pm_object = self.builder.get_object("am_pm_entry")
		am_pm = am_pm_object.get_active()

		date_object = self.builder.get_object("date_entry")
		date = self.parse_date(date_object.get_text())

		global date_ok
		if date == None:
			date_ok = False
		else:
			date_ok = True

			sound_switch = self.builder.get_object("sound_switch")
			switch = sound_switch.get_active()
			if not switch:
				self.on_sound_clear_clicked(None)

			sound_object = self.builder.get_object("sound_entry")
			sound = sound_object.get_filename()

			length_object = self.builder.get_object("length_entry")
			length = length_object.get_value()

			loop_object = self.builder.get_object("loop_entry")
			loop = loop_object.get_active()

			notification_object = self.builder.get_object("notification_entry")
			notification = notification_object.get_active()

			command_object = self.builder.get_object("command_entry")
			command = command_object.get_text()

			notes_object = self.builder.get_object("notes_entry")
			buffer = notes_object.get_buffer()
			notes = buffer.get_text(buffer.get_start_iter(), buffer.get_end_iter(), True)

			db = CouchDatabase('indicator-remindor-alarms', create=True)

			rec = Record({
				"label" : label,
				"notes" : notes,
				"hour" : hour,
				"minute" : minute,
				"am_pm" : am_pm,
				"date" : date,
				"sound" : sound,
				"sound_length" : length,
				"sound_loop" : loop,
				"notification" : notification,
				"command" : command
			}, record_type='http://wiki.ubuntu.com/Quickly/RecordTypes/IndicatorRemindor/')
			r = db.put_record(rec)

			self.emit('alarm_added', r)

    def on_btn_cancel_clicked(self, widget, data=None):
        pass

	def on_date_entry_changed(self, widget, data=None):
		add = self.builder.get_object("btn_add")
		if self.parse_date(widget.get_text()) == None:
			add.set_sensitive(False)
			widget.set_icon_from_stock(0, Gtk.STOCK_DIALOG_ERROR)
		else:
			add.set_sensitive(True)
			widget.set_icon_from_stock(0, None)

	def on_btn_help_clicked(self, widget, data=None):
		show_uri(self, "ghelp:%s" % get_help_uri('add'))
		return True

	def on_sound_switch_activate(self, widget, data=None):
		sound_file_label = self.builder.get_object("sound_file_label")
		length_label = self.builder.get_object("length_label")
		loop_label = self.builder.get_object("loop_label")
		sound_box = self.builder.get_object("sound_box")
		length_box = self.builder.get_object("length_box")
		loop_box = self.builder.get_object("loop_box")

		if widget.get_active():
			sound_file_label.show()
			length_label.show()
			loop_label.show()
			sound_box.show_all()
			length_box.show_all()
			loop_box.show()
		else:
			sound_file_label.hide()
			length_label.hide()
			loop_label.hide()
			sound_box.hide()
			length_box.hide()
			loop_box.hide()

	def on_loop_switch_activate(self, widget, data=None):
		loop_label = self.builder.get_object("loop_label2")
		length_entry = self.builder.get_object("length_entry")

		if widget.get_active():
			self.set_sound_length(0)
			loop_label.show()
			length_entry.set_sensitive(False)
		else:
			loop_label.hide()
			length_entry.set_sensitive(True)

	def label_clear(self, widget, pos, event, data=None):
		self.set_label("")

	def date_clear(self, widget, pos, event, data=None):
		self.set_date("")

	def command_clear(self, widget, pos, event, data=None):
		self.set_command("")

	def set_all(self, label, notes, hour, minute, am_pm, date, sound, length, loop, notification, command):
		self.set_label(label)
		self.set_notes(notes)
		self.set_hour(hour)
		self.set_minute(minute)
		self.set_am_pm(am_pm)
		self.set_date(date)
		self.set_sound(sound)
		self.set_sound_length(length)
		self.set_loop(loop)
		self.set_notification(notification)
		self.set_command(command)

	def set_label(self, label):
		label_object = self.builder.get_object("label_entry")
		label_object.set_text(label)

	def set_notes(self, notes):
		buffer = self.builder.get_object("notes_entry").get_buffer()
		buffer.set_text(notes)

	def set_hour(self, hour):
		hour_object = self.builder.get_object("hour_entry")
		hour_object.set_value(int(hour))

	def set_minute(self, minute):
		minute_object = self.builder.get_object("minute_entry")
		minute_object.set_value(int(minute))

	def set_am_pm(self, am_pm):
		am_pm_object = self.builder.get_object("am_pm_entry")
		am_pm_object.set_active(am_pm)

	def set_date(self, date):
		date_object = self.builder.get_object("date_entry")
		date_object.set_text(date)

	def set_sound(self, sound, activate=True):
		sound_object = self.builder.get_object("sound_entry")
		sound_switch = self.builder.get_object("sound_switch")

		if sound is not None:
			sound_object.set_filename(sound)
			sound_switch.set_active(True)
			self.on_sound_switch_activate(sound_switch)
		else:
			sound_object.unselect_all()

			if activate:
				sound_switch.set_active(False)
				self.on_sound_switch_activate(sound_switch)

	def set_sound_length(self, length):
		length_object = self.builder.get_object("length_entry")
		length_object.set_value(int(length))

	def set_loop(self, loop):
		loop_object = self.builder.get_object("loop_entry")
		loop_object.set_active(loop)

	def set_notification(self, notification):
		notification_object = self.builder.get_object("notification_entry")
		notification_object.set_active(notification)

	def set_command(self, command):
		command_object = self.builder.get_object("command_entry")
		command_object.set_text(command)

	def set_edit(self):
		add_button = self.builder.get_object("btn_add")
		add_button.set_label("Edit")
		image = Gtk.Image()
		image.set_from_stock(Gtk.STOCK_EDIT, Gtk.IconSize.BUTTON)
		add_button.set_image(image)

	def parse_date(self, d):
		d = d.lower()
		today = date.today()
		format = "%m/%d/%Y"

		days = ["monday", "tuesday", "wednesday", "thursday", "friday", "saturday",
			"sunday", "mon", "tues", "wed", "thurs", "fri", "sat", "sun"]
		other_days = ["today", "tomorrow", "tom"]
		other = ["days", "from"]
		first_only = ["next", "nx", "every", "ev"]
		second_only = ["weekday", "weekend", "other", "day"]

		if d == "" or d == None:
			d = "today"

		dl = d.split()
		if len(dl) == 1:
			#either a number or a day or a date
			try:
				parser.parse(d)
			except ValueError:
				#invalid
				if d not in other_days and d not in days: #ex: tomorrow, mon
					return None

			if d.isdigit(): #ex: 5
				d2 = today + timedelta(int(d))
				return d2.strftime(format)
			elif d in days or d in other_days: #ex: monday, tom
				return self.parse_day(d)
			elif len(d) == 1 and not d.isdigit():
				return None
			else: #ex: 7/12/2012
				return parser.parse(d).strftime(format)
		else:
			if dl[-1] == "from":
				return None
			if dl[0] not in first_only:
				try:
					return parser.parse(d).strftime(format)
				except ValueError:
					return None
			if dl[1] not in second_only and dl[1] not in days and not dl[1].isdigit():
				return None
			if len(dl) > 2 and dl[2] not in other:
				return None
			if len(dl) > 3:
				try:
					parser.parse(dl[3])
					if dl[3].isdigit():
						d2 = today + timedelta(int(dl[3]))
						d = d.replace(dl[3], d2.strftime(format))

				except ValueError:
					if len(dl) > 4 and dl[3] in other:
						try:
							parser.parse(dl[4])
							if dl[4].isdigit():
								d2 = today + timedelta(int(dl[4]))
								d = d.replace(dl[4], d2.strftime(format))

						except ValueError:
							return None
					else:
						return None
			if len(dl) > 2 and dl[1].isdigit() and dl[2] == "from":
				return None
			if len(dl) == 2 and dl[0] == "every" and dl[1].isdigit():
				if int(dl[1]) > 31:
					return None

			d = d.replace("ev ", "every ")
			d = d.replace("nx ", "next ")

			if "monday" not in d:
				d = d.replace("mon", "monday")
			elif "tuesday" not in d:
				d = d.replace("tues", "tuesday")
			elif "wednesday" not in d:
				d = d.replace("wed", "wednesday")
			elif "thursday" not in d:
				d = d.replace("thurs", "thursday")
			elif "friday" not in d:
				d = d.replace("fri", "friday")
			elif "saturday" not in d:
				d = d.replace("sat", "saturday")
			elif "sunday" not in d:
				d = d.replace("sun", "sunday")

			if d == "every other":
				d += " from " + today.strftime(format)
			if len(dl) == 3 and dl[0] == "every" and dl[2] == "days":
				d += " from " + today.strftime(format)
			if len(dl) == 2 and dl[0] == "next" and dl[1].isdigit():
				if int(dl[1]) > 31:
					return None

				if int(dl[1]) > today.day:
					return parser.parse(dl[1]).strftime(format)
				else:
					d2 = parser.parse(dl[1])
					cont = True
					counter = 1
					while cont:
						try:
							d2 = d2.replace(month=(d2.month + counter))
							cont = False
						except ValueError:
							counter += 1

					return d2.strftime(format)
			if len(dl) == 3 and dl[0] == "next" and dl[2] == "days":
				d += " from " + today.strftime(format)

			if d.lower() == "next monday":
				d2 = parser.parse("monday") + timedelta(days=7)
				return d2.strftime(format)
			elif d.lower() == "next tuesday":
				d2 = parser.parse("tuesday") + timedelta(days=7)
				return d2.strftime(format)
			elif d.lower() == "next wednesday":
				d2 = parser.parse("wednesday") + timedelta(days=7)
				return d2.strftime(format)
			elif d.lower() == "next thursday":
				d2 = parser.parse("thursday") + timedelta(days=7)
				return d2.strftime(format)
			elif d.lower() == "next friday":
				d2 = parser.parse("friday") + timedelta(days=7)
				return d2.strftime(format)
			elif d.lower() == "next saturday":
				d2 = parser.parse("saturday") + timedelta(days=7)
				return d2.strftime(format)
			elif d.lower() == "next sunday":
				d2 = parser.parse("sunday") + timedelta(days=7)
				return d2.strftime(format)

			return d

	def parse_day(self, d):
		today = date.today()
		format = "%m/%d/%Y"

		if d.lower() == "today":
			return today.strftime(format)
		elif d.lower() == "tomorrow" or d.lower() == "tom":
			tom = today + timedelta(1)
			return tom.strftime(format)
		elif d.lower() == "monday" or d.lower() == "mon":
			d2 = parser.parse("monday")
			return d2.strftime(format)
		elif d.lower() == "tuesday" or d.lower() == "tues":
			d2 = parser.parse("tuesday")
			return d2.strftime(format)
		elif d.lower() == "wednesday" or d.lower() == "wed":
			d2 = parser.parse("wednesday")
			return d2.strftime(format)
		elif d.lower() == "thursday" or d.lower() == "thurs":
			d2 = parser.parse("thursday")
			return d2.strftime(format)
		elif d.lower() == "friday" or d.lower() == "fri":
			d2 = parser.parse("friday")
			return d2.strftime(format)
		elif d.lower() == "saturday" or d.lower() == "sat":
			d2 = parser.parse("saturday")
			return d2.strftime(format)
		elif d.lower() == "sunday" or d.lower() == "sun":
			d2 = parser.parse("sunday")
			return d2.strftime(format)
		else:
			return None

if __name__ == "__main__":
    dialog = AlarmDialog()
    dialog.show()
    Gtk.main()
