# -*- Mode: Python; coding: utf-8; indent-tabs-mode: nil; tab-width: 4 -*-
### BEGIN LICENSE
# Copyright (C) 2012 <Duncan Davis> <duncanjdavis@gmail.com>
# This program is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License version 3, as published 
# by the Free Software Foundation.
# 
# This program is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranties of 
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
# PURPOSE.  See the GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License along 
# with this program.  If not, see <http://www.gnu.org/licenses/>.
### END LICENSE

import gettext
from gettext import gettext as _
gettext.textdomain('download-monitor')

from gi.repository import Gtk, Gio, Unity, GdkPixbuf, Notify, Dbusmenu # pylint: disable=E0611
import logging
logger = logging.getLogger('download_monitor')

from download_monitor_lib import Window
from download_monitor.AboutDownloadMonitorDialog import AboutDownloadMonitorDialog
from download_monitor.PreferencesDownloadMonitorDialog import PreferencesDownloadMonitorDialog
from download_monitor.DownloadMonitor import DownloadMonitor
from download_monitor.camo import *
import copy, time, os, xdg.BaseDirectory
DAILY_SVG = os.path.join(xdg.BaseDirectory.xdg_data_home, 
    'download-monitor-daily.svg')
HOURLY_SVG = os.path.join(xdg.BaseDirectory.xdg_data_home, 
    'download-monitor-hourly.svg')
MONTHLY_SVG = os.path.join(xdg.BaseDirectory.xdg_data_home, 
    'download-monitor-monthly.svg')
OVERVIEW_TOOLBAR_SVG = os.path.join(xdg.BaseDirectory.xdg_data_home, 
    'download-monitor-overview-toolbar.svg')
MONTHLY_TOOLBAR_SVG = os.path.join(xdg.BaseDirectory.xdg_data_home, 
    'download-monitor-monthly-toolbar.svg')
DAILY_TOOLBAR_SVG = os.path.join(xdg.BaseDirectory.xdg_data_home, 
    'download-monitor-daily-toolbar.svg')
HOURLY_TOOLBAR_SVG = os.path.join(xdg.BaseDirectory.xdg_data_home, 
    'download-monitor-hourly-toolbar.svg')
MEDIA_DIR = os.path.join(os.path.dirname(__file__), '..', 'data', 'media')
EXTRAS = False
if not(os.path.exists(MEDIA_DIR)):
    MEDIA_DIR = '/usr/share/download-monitor/media'
if not(os.path.exists(MEDIA_DIR)):
    EXTRAS = True
    MEDIA_DIR = '/opt/extras.ubuntu.com/download-monitor/share/download-monitor/media'

OVERVIEW_TOOLBAR_SVG_ORIG = os.path.join(MEDIA_DIR, 'overview.svg')
MONTHLY_TOOLBAR_SVG_ORIG = os.path.join(MEDIA_DIR, 'monthly.svg')
DAILY_TOOLBAR_SVG_ORIG = os.path.join(MEDIA_DIR, 'daily.svg')
HOURLY_TOOLBAR_SVG_ORIG = os.path.join(MEDIA_DIR, 'hourly.svg')

Notify.init('download-monitor')

# See download_monitor_lib.Window.py for more details about how this class works
class DownloadMonitorWindow(Window):
    __gtype_name__ = "DownloadMonitorWindow"
    NOTIFICATIONS = {
        "down":{
            DownloadMonitor.QUOTA_EXCEEDS_DAILY : _("You have exceeded your download quota for today!"),
            DownloadMonitor.QUOTA_WARN_DAILY : _("You have nearly exceeded your download quota for today. It's advisable to reduce your usage."),
            DownloadMonitor.QUOTA_EXCEEDS_MONTHLY : _("You have exceeded your monthly download quota!"),
            DownloadMonitor.QUOTA_WARN_MONTHLY : _("You have nearly exceeded your monthly download quota. It's advisable to reduce your usage.")
        },
        "up":{
            DownloadMonitor.QUOTA_EXCEEDS_DAILY : _("You have exceeded your upload quota for today!"),
            DownloadMonitor.QUOTA_WARN_DAILY : _("You have nearly exceeded your upload quota for today. It's advisable to reduce your usage."),
            DownloadMonitor.QUOTA_EXCEEDS_MONTHLY : _("You have exceeded your monthly upload quota!"),
            DownloadMonitor.QUOTA_WARN_MONTHLY : _("You have nearly exceeded your monthly upload quota. It's advisable to reduce your usage.")
        },
        "total":{
            DownloadMonitor.QUOTA_EXCEEDS_DAILY : _("You have exceeded your total network usage quota for today!"),
            DownloadMonitor.QUOTA_WARN_DAILY : _("You have nearly exceeded your total network usage quota for today. It's advisable to reduce your usage."),
            DownloadMonitor.QUOTA_EXCEEDS_MONTHLY : _("You have exceeded your total network usage download quota!"),
            DownloadMonitor.QUOTA_WARN_MONTHLY : _("You have nearly exceeded your total network usage download quota. It's advisable to reduce your usage.")
        }
    }
    
    def finish_initializing(self, builder): # pylint: disable=E1002
        """Set up the main window"""
        super(DownloadMonitorWindow, self).finish_initializing(builder)
        self.mon = DownloadMonitor()

        self.notifications = copy.deepcopy(self.__class__.NOTIFICATIONS)
        if EXTRAS:
            self.launcher = Unity.LauncherEntry.get_for_desktop_id(
                "extras-download-monitor.desktop")
        else:
            self.launcher = Unity.LauncherEntry.get_for_desktop_id(
                "download-monitor.desktop")
        self.AboutDialog = AboutDownloadMonitorDialog
        self.PreferencesDialog = PreferencesDownloadMonitorDialog
        self.overview = self.builder.get_object("label5")
        self.button1 = self.builder.get_object("button1")
        self.button1.connect("clicked", self.button_clicked)
        self.box2 = self.builder.get_object("box2")
        self.graph_monthly = self.builder.get_object("image1")
        self.graph_daily = self.builder.get_object("image3")
        self.graph_hourly = self.builder.get_object("image4")
        self.notebook = self.builder.get_object("notebook1")
        self.toolbar = self.builder.get_object("toolbar1")
        ctx = self.toolbar.get_style_context()
        ctx.add_class(Gtk.STYLE_CLASS_PRIMARY_TOOLBAR)
        self.overview_tab = self.builder.get_object("toolbutton1")
        self.overview_tab.connect("toggled", self.goto_overview)
        self.monthly_tab = self.builder.get_object("toolbutton2")
        self.monthly_tab.connect("toggled", self.goto_monthly)
        self.daily_tab = self.builder.get_object("toolbutton3")
        self.daily_tab.connect("toggled", self.goto_daily)
        self.hourly_tab = self.builder.get_object("toolbutton4")
        self.hourly_tab.connect("toggled", self.goto_hourly)
        self.overview_icon = self.builder.get_object("image5")
        self.monthly_icon = self.builder.get_object("image6")
        self.daily_icon = self.builder.get_object("image7")
        self.hourly_icon = self.builder.get_object("image8")
        self.theme = Gio.Settings.new('org.gnome.desktop.interface')
        self.theme.connect('changed::gtk-theme', self.make_icons)
        self.overview_menu = self.builder.get_object("menuitem4")
        self.monthly_menu = self.builder.get_object("menuitem1")
        self.daily_menu = self.builder.get_object("menuitem2")
        self.hourly_menu = self.builder.get_object("menuitem3")
        self.overview_menu.connect("activate", self.goto_overview)
        self.monthly_menu.connect("activate", self.goto_monthly)
        self.daily_menu.connect("activate", self.goto_daily)
        self.hourly_menu.connect("activate", self.goto_hourly)
        self.make_icons()
        self.mon.connect_changed(self.changed)
        self.signal_ok = True
    
        self.ql_monthly_text = _("Monthly: %s")
        self.ql_daily_text = _("Daily: %s")
        self.ql_hourly_text = _("Hourly: %s")
        
        self.ql = Dbusmenu.Menuitem.new()
        self.ql_monthly = Dbusmenu.Menuitem.new()
        self.ql_monthly.property_set(Dbusmenu.MENUITEM_PROP_LABEL, self.ql_monthly_text)
        self.ql_monthly.property_set_bool(Dbusmenu.MENUITEM_PROP_VISIBLE, True)
        self.ql_monthly.connect("item-activated", self.goto_monthly)
        self.ql.child_append(self.ql_monthly)
        self.ql_daily = Dbusmenu.Menuitem.new()
        self.ql_daily.property_set(Dbusmenu.MENUITEM_PROP_LABEL, self.ql_daily_text)
        self.ql_daily.property_set_bool(Dbusmenu.MENUITEM_PROP_VISIBLE, True)
        self.ql_daily.connect("item-activated", self.goto_daily)
        self.ql.child_append(self.ql_daily)
        self.ql_hourly = Dbusmenu.Menuitem.new()
        self.ql_hourly.property_set(Dbusmenu.MENUITEM_PROP_LABEL, self.ql_hourly_text)
        self.ql_hourly.property_set_bool(Dbusmenu.MENUITEM_PROP_VISIBLE, True)
        self.ql_hourly.connect("item-activated", self.goto_hourly)
        self.ql.child_append(self.ql_hourly)
        self.ql_quotas = Dbusmenu.Menuitem.new()
        self.ql_quotas.property_set(Dbusmenu.MENUITEM_PROP_LABEL, _("Set Quotas"))
        self.ql_quotas.property_set_bool(Dbusmenu.MENUITEM_PROP_VISIBLE, True)
        self.ql_quotas.connect("item-activated", self.button_clicked)
        self.ql.child_append(self.ql_quotas)
        self.launcher.set_property("quicklist", self.ql)
        
        self.notify("", True)
        
        self.changed()
        
    def make_icons(self, *a):
        #FIXME: doesn't work 2nd time for some reason
        c = Colour.from_widget(self.toolbar).get_inverse()
        inf = open(OVERVIEW_TOOLBAR_SVG_ORIG)
        of = open(OVERVIEW_TOOLBAR_SVG, 'w')
        c.camo_rgb(inf, of, '#dd4814')
        inf.close()
        of.close()
        inf = open(MONTHLY_TOOLBAR_SVG_ORIG)
        of = open(MONTHLY_TOOLBAR_SVG, 'w')
        c.camo_rgb(inf, of, '#dd4814')
        inf.close()
        of.close()
        inf = open(DAILY_TOOLBAR_SVG_ORIG)
        of = open(DAILY_TOOLBAR_SVG, 'w')
        c.camo_rgb(inf, of, '#dd4814')
        inf.close()
        of.close()
        inf = open(HOURLY_TOOLBAR_SVG_ORIG)
        of = open(HOURLY_TOOLBAR_SVG, 'w')
        c.camo_rgb(inf, of, '#dd4814')
        inf.close()
        of.close()
        i = GdkPixbuf.Pixbuf.new_from_file_at_size(OVERVIEW_TOOLBAR_SVG, 32, 32)
        self.overview_icon.set_from_pixbuf(i)
        i = GdkPixbuf.Pixbuf.new_from_file_at_size(MONTHLY_TOOLBAR_SVG, 32, 32)
        self.monthly_icon.set_from_pixbuf(i)
        i = GdkPixbuf.Pixbuf.new_from_file_at_size(DAILY_TOOLBAR_SVG, 32, 32)
        self.daily_icon.set_from_pixbuf(i)
        i = GdkPixbuf.Pixbuf.new_from_file_at_size(HOURLY_TOOLBAR_SVG, 32, 32)
        self.hourly_icon.set_from_pixbuf(i)

    def changed(self, *a):
        self.mon.write_svg_daily(DAILY_SVG)
        self.graph_daily.set_from_file(DAILY_SVG)
        self.mon.write_svg_monthly(MONTHLY_SVG)
        self.graph_monthly.set_from_file(MONTHLY_SVG)
        self.mon.write_svg_hourly(HOURLY_SVG)
        self.graph_hourly.set_from_file(HOURLY_SVG)
        um = self.mon.get_human_size(self.mon.get_usage(self.mon.get_month(0))[2])
        ud = self.mon.get_human_size(self.mon.get_usage(self.mon.get_day(0))[2])
        uh = self.mon.get_human_size(self.mon.get_usage(self.mon.get_hour(0))[2])
        self.ql_monthly.property_set(Dbusmenu.MENUITEM_PROP_LABEL, self.ql_monthly_text % um)
        self.ql_daily.property_set(Dbusmenu.MENUITEM_PROP_LABEL, self.ql_daily_text % ud)
        self.ql_hourly.property_set(Dbusmenu.MENUITEM_PROP_LABEL, self.ql_hourly_text % uh)
        wd = self.mon.check_quota("down")
        for wd in wd:
            if (wd in self.notifications["down"]) and (type(self.notifications["down"][wd])==str):
                self.notify(self.notifications["down"][wd])
                self.notifications["down"][wd] = time.time()
        wu = self.mon.check_quota("up")
        for wu in wu:
            if wu and (wu in self.notifications["up"]) and (type(self.notifications["up"][wu])==str):
                self.notify(self.notifications["up"][wu])
                self.notifications["up"][wu] = time.time()
        wt = self.mon.check_quota("total")
        for wt in wt:
            if wt and (wt in self.notifications["total"]) and (type(self.notifications["total"][wt])==str):
                self.notify(self.notifications["up"][wt])
                self.notifications["total"][wt] = time.time()
        q = self.mon.get_quota("down")+self.mon.get_quota("up")+self.mon.get_quota("total")
        if q > 0:
            self.box2.hide()
            tot = self.mon.get_monthly_proportion("total")
            up = self.mon.get_monthly_proportion("up")
            down = self.mon.get_monthly_proportion("down")
            #print(repr((tot,up,down)))
            progress = self.mon.greatest(self.mon.greatest(tot,down),up)
            #print(progress)
            self.launcher.set_property("progress", progress)
            self.launcher.set_property("progress_visible", True)
        else:
            self.box2.show()
            self.launcher.set_property("progress_visible", False)
        self.overview.set_markup(self.mon.get_overview_text())
        
    def notify(self, message, dummy=False):
        n = Notify.Notification.new("Download Monitor", message, "download-monitor")
        pb = GdkPixbuf.Pixbuf.new_from_file(os.path.join(MEDIA_DIR, 'download-monitor.svg'))
        n.set_icon_from_pixbuf(pb)
        if not dummy:
            n.show()
                
    def button_clicked(self, *a):
        self.on_mnu_preferences_activate(None)
    
    def goto_overview(self, *a):
        self.deiconify()
        if self.signal_ok:
            self.notebook.set_current_page(0)
            self.signal_ok = False
            self.overview_tab.set_active(True)
            self.monthly_tab.set_active(False)
            self.daily_tab.set_active(False)
            self.hourly_tab.set_active(False)
            self.signal_ok = True
        
    def goto_monthly(self, *a):
        self.deiconify()
        if self.signal_ok:
            self.notebook.set_current_page(1)
            self.signal_ok = False
            self.overview_tab.set_active(False)
            self.monthly_tab.set_active(True)
            self.daily_tab.set_active(False)
            self.hourly_tab.set_active(False)
            self.signal_ok = True
        
    def goto_daily(self, *a):
        self.deiconify()
        if self.signal_ok:
            self.notebook.set_current_page(2)
            self.signal_ok = False
            self.overview_tab.set_active(False)
            self.monthly_tab.set_active(False)
            self.daily_tab.set_active(True)
            self.hourly_tab.set_active(False)
            self.signal_ok = True
        
    def goto_hourly(self, *a):
        self.deiconify()
        if self.signal_ok:
            self.notebook.set_current_page(3)
            self.signal_ok = False
            self.overview_tab.set_active(False)
            self.monthly_tab.set_active(False)
            self.daily_tab.set_active(False)
            self.hourly_tab.set_active(True)
            self.signal_ok = True

