from gi.repository import Gio, Gtk
import subprocess, time, os, sys
from download_monitor.Chart import make_svg
import gettext
from gettext import gettext as _
gettext.textdomain('download-monitor')

KiB = 1024.0
MiB = KiB*KiB
GiB = MiB*KiB

OVERVIEW = (
    _('<span><b>This Month:</b> <span foreground="#00ff00" >Uploaded <b>'),
    _('</b></span>, <span foreground="#ff0000" >Downloaded <b><b>'),
    _('</b></b></span>, <span foreground="#0000ff" >Total <b>'),
    _('</b></span>\n\n<b>Today:</b> <span foreground="#00ff00" >Uploaded <b>'),
    _('</b></span>, <span foreground="#ff0000" >Downloaded <b>'),
    _('</b></span>, <span foreground="#0000ff" >Total <b>'),
    _('</b></span>\n\n<b>Since HOUR:00:</b> <span foreground="#00ff00" >Uploaded <b>'),
    _('</b></span>, <span foreground="#ff0000" >Downloaded <b>'),
    _('</b></span>, <span foreground="#0000ff" >Total <b>'),
    _('</b></span>\n\n<span size="large"><b>All Time:</b> <span foreground="#00ff00" >Uploaded <b>'),
    _('</b></span>, <span foreground="#ff0000" >Downloaded <b>'),
    _('</b></span>, <span foreground="#0000ff" >Total <b>'),
    '</b></span></span></span>'
)

class DownloadMonitor(object):
    MONTHLY = -3
    DAILY = -2
    HOURLY = -1
    QUOTA_OK = 0
    QUOTA_EXCEEDS_HOURLY = 1
    QUOTA_WARN_HOURLY = 2
    QUOTA_EXCEEDS_DAILY = 3
    QUOTA_WARN_DAILY = 4
    QUOTA_EXCEEDS_MONTHLY = 5
    QUOTA_WARN_MONTHLY = 6
    def __init__(self, quota=-1):
        self.update()
        self.settings = Gio.Settings("net.launchpad.download-monitor")
        self.on_change = []
        self.settings.connect("changed", self.changed)
        self.mon = Gio.File.new_for_path('/var/lib/vnstat').monitor_directory(
            Gio.FileMonitorFlags.NONE,None)
        self.mon.connect("changed", self.changed)
        self.colours = ("#00FF00", "#FF0000", "#0000FF")

        
    def connect_changed(self, fn):
        self.on_change.append(fn)
        
    def changed(self, *a):
        self.update()
        for fn in self.on_change:
            fn()
        
    def get_quota(self, t="down"):
        return (self.settings.get_uint("%squotagib"%t) * KiB + 
            self.settings.get_uint("%squotamib"%t)) * MiB
            
    def get_quota_t(self, t="down"):
        qm = self.get_quota(t)
        qd = qm / 31.0
        qh = qd / 24.0
        return (qm, qd, qh)
        
    def get_monthly_proportion(self, t="down"):
        qd = self.get_quota("down")
        qu = self.get_quota("up")
        qt = self.get_quota("total")
        u = self.get_month(0)
        if qd > 0 and t == "down":
            #print("down " + self.get_human_size(u[0]))
            #print("quota " + self.get_human_size(qd))
            return u[0]/qd
        elif qu > 0 and t == "up":
            #print("up " + str(u[1]))
            return u[1]/qu
        elif qt > 0 and t == "total":
            #print("total " + str(u[0]))
            return (u[0]+u[1])/qt
        return 0.0
            
    def get_daily_proportion(self, t="down"):
        qd = self.get_quota("down") / 31.0
        qu = self.get_quota("up") / 31.0
        qt = self.get_quota("total") / 31.0
        u = self.get_day(0)
        if qd > 0 and t == "down":
            return u[0]/qd
        elif qu > 0 and t == "up":
            return u[1]/qu
        elif qt > 0 and t == "total":
            return (u[0]+u[1])/qt
        return 0.0
        
    def check_quota(self, t="down"):
        c = self.__class__
        qm, qd, qh = self.get_quota_t(t)
        h,d,m = (0,0,0)
        if t == "down":
            h = self.get_hour(0)[0]
            d = self.get_day(0)[0]
            m = self.get_month(0)[0]
        elif t == "up":
            h = self.get_hour(0)[1]
            d = self.get_day(0)[1]
            m = self.get_month(0)[1]
        elif t == "total":
            h = self.get_hour(0)
            h = h[0] + h[1]
            d = self.get_day(0)
            d = d[0] + d[1]
            m = self.get_month(0)
            m = m[0] + m[1]
        r = []
        if qh > 0:
            if h > qh:
                r.append(c.QUOTA_EXCEEDS_HOURLY)
            elif h > qh - 0.05*qh:
                r.append(c.QUOTA_WARN_HOURLY)
        if qd > 0:
            if d > qd:
                r.append(c.QUOTA_EXCEEDS_DAILY)
            elif d > qd - 0.05*qd:
                r.append(c.QUOTA_WARN_DAILY)
        if qm > 0:
            if m > qm:
                r.append(c.QUOTA_EXCEEDS_MONTHLY)
            elif m > qm - 0.05*qm:
                r.append(c.QUOTA_WARN_MONTHLY)
        return r
        
        
    def iflist(self):
        return filter(lambda a:not(a.startswith('.')),
            os.listdir('/var/lib/vnstat'))
            
    def parse_1(self, number, timestamp, Mrx, Mtx, Krx, Ktx, n=None, *rest):
        return (float(Mrx)*MiB+float(Krx)*KiB, float(Mtx)*MiB+float(Ktx)*KiB, 
            float(timestamp))
        
    def parse_2(self, number, timestamp, Krx, Ktx, n=None, *rest):
        return (float(Krx)*KiB, float(Ktx)*KiB, float(timestamp))
        
    def greatest(self, a, b):
        if a > b:
            return a
        return b
        
    def merge(self, a, b):
        return (a[0]+b[0], a[1]+b[1], self.greatest(a[2],b[2]))
        
    def get_hour_iface(self, iface, hour):
        hours = self.db[iface]['hours']
        if len(hours) > hour:
            return hours[hour]
        return (0,0,0)
        
    def get_hour(self, hour):
        total = (0,0,0)
        for iface in self.db:
            h = self.get_hour_iface(iface, hour)
            total = self.merge(total, h)
        return total
            
    def get_day_iface(self, iface, day):
        days = self.db[iface]['days']
        if len(days) > day:
            return days[day]
        return (0,0,0)
        
    def get_day(self, day):
        total = (0,0,0)
        for iface in self.db:
            d = self.get_day_iface(iface, day)
            total = self.merge(total, d)
        return total
            
    def get_month_iface(self, iface, month):
        months = self.db[iface]['months']
        if len(months) > month:
            return months[month]
        return (0,0,0)
        
    def get_month(self, month):
        total = (0,0,0)
        for iface in self.db:
            m = self.get_month_iface(iface, month)
            total = self.merge(total, m)
        return total
            
    def get_total_iface(self, iface):
        return self.db[iface]['total']
            
    def get_total(self):
        total = (0,0,0)
        for iface in self.db:
            t = self.get_total_iface(iface)
            total = self.merge(total, t)
        return total
            
    def get_usage(self, u):
        return (u[1], u[0], u[0]+u[1])
        
    def update(self):
        self.db = {}
        sortfn = lambda a,b:cmp(b[2],a[2])
        for iface in self.iflist():
            try:
                t = subprocess.check_output("vnstat --dumpdb -i %s"%iface, shell=True)
                db = {
                    #"total" : (,),
                    #"current" : (,),
                    "hours" : [],
                    "days" : [],
                    "months" : []
                }
                ttrx = 0.0
                tttx = 0.0
                crx = 0.0
                ctx = 0.0
                for line in t.split("\n"):
                    items = line.split(";")
                    key = items.pop(0)
                    if key == "totalrx":
                        ttrx += float(items[0])*MiB
                    elif key == "totalrxk":
                        ttrx += float(items[0])*KiB
                    elif key == "totaltx":
                        tttx += float(items[0])*MiB
                    elif key == "totaltxk":
                        tttx += float(items[0])*KiB
                    elif key == "currx":
                        crx = float(items[0])+0.0
                    elif key == "curtx":
                        ctx = float(items[0])+0.0
                    elif key == "h":
                        db['hours'].append(self.parse_2(*items))
                    elif key == "d":
                        db['days'].append(self.parse_1(*items))
                    elif key == "m":
                        db['months'].append(self.parse_1(*items))
                    db['total'] = (ttrx,tttx,0.0)
                    db['current'] = (crx, ctx, time.time())
                    self.db[iface] = db
                
                db['hours'].sort(sortfn)
            except Exception, e:
                sys.stderr.write(("Error reading database for %s: "%
                    iface)+repr(e)+"\n")
                
    def flatten(self, a):
        if not (type(a) == tuple): 
            return (a,)
        elif len(a) == 0: 
            return ()
        else: 
            return self.flatten(a[0]) + self.flatten(a[1:]) 
                
    def get_overview(self):
        return self.flatten((self.get_usage(self.get_month(0)),
        self.get_usage(self.get_day(0)),
        self.get_usage(self.get_hour(0)),
        self.get_usage(self.get_total())))
        
    def get_human_size(self, bytes):
        if bytes >= GiB:
            return "%.2fGiB" % (bytes / GiB)
        if bytes >= MiB:
            return "%.2fMiB" % (bytes / MiB)
        if bytes >= KiB:
            return "%.2fKiB" % (bytes / KiB)
        else:
            return "%iB" % bytes
            
    def interleave(self, a, b):
        r = []
        i = 0
        l = self.greatest(len(a),len(b))
        while i < l:
            if i < len(a):
                r.append(a[i])
            if i < len(b):
                r.append(b[i])
            i += 1
        return r
        
    def get_overview_text(self):
        return (''.join(self.interleave(OVERVIEW, 
            map(self.get_human_size, self.get_overview())))).replace('HOUR',
            time.strftime("%H"))
            
    def make_svg_daily(self):
        qu = self.get_quota("up") / 31.0
        qd = self.get_quota("down") / 31.0
        qt = self.get_quota("total") / 31.0
        lines = [(0, '#000000')]
        if qu > 0:
            lines.append((qu, '#00FF00'))
        if qd > 0:
            lines.append((qd, '#FF0000'))
        if qt > 0:
            lines.append((qt, '#0000FF'))
        d = 0
        bars = []
        while d < 31:
            t = self.get_day(d)
            if t[2] > 0:
                u = self.get_usage(t)
                bars.append((time.strftime("%d %b %Y", time.localtime(t[2])), 
                    (map(self.get_human_size, u), u, self.colours)))
            d += 1
        return make_svg({'bars':bars,'lines':lines}, offsetx=10, margin=80, maxbarwidth=530)
        
    def make_svg_monthly(self):
        qu = self.get_quota("up")
        qd = self.get_quota("down")
        qt = self.get_quota("total")
        lines = [(0, '#000000')]
        if qu > 0:
            lines.append((qu, '#00FF00'))
        if qd > 0:
            lines.append((qd, '#FF0000'))
        if qt > 0:
            lines.append((qt, '#0000FF'))
        m = 0
        bars = []
        while m < 13:
            t = self.get_month(m)
            if t[2] > 0:
                u = self.get_usage(t)
                bars.append((time.strftime("%b %Y", time.localtime(t[2])), 
                    (map(self.get_human_size, u), u, self.colours)))
            m += 1
        return make_svg({'bars':bars,'lines':lines}, offsetx=10, margin=60, maxbarwidth=550)
        
    def make_svg_hourly(self):
        qu = self.get_quota("up") / 744.0
        qd = self.get_quota("down") / 744.0
        qt = self.get_quota("total") / 744.0
        lines = [(0, '#000000')]
        if qu > 0:
            lines.append((qu, '#00FF00'))
        if qd > 0:
            lines.append((qd, '#FF0000'))
        if qt > 0:
            lines.append((qt, '#0000FF'))
        h = 0
        bars = []
        while h < 25:
            t = self.get_hour(h)
            if t[2] > 0:
                u = self.get_usage(t)
                bars.append((time.strftime("%H:00 %d %b %Y", time.localtime(t[2])), 
                    (map(self.get_human_size, u), u, self.colours)))
            h += 1
        return make_svg({'bars':bars,'lines':lines}, offsetx=10, margin=110)
        
    def write_svg_daily(self, f):
        f = open(f, 'w')
        f.write(self.make_svg_daily())
        
    def write_svg_monthly(self, f):
        f = open(f, 'w')
        f.write(self.make_svg_monthly())
        
    def write_svg_hourly(self, f):
        f = open(f, 'w')
        f.write(self.make_svg_hourly())
        
        
if __name__ == '__main__':
    mon = DownloadMonitor()
    print(mon.get_overview_text())


