# -*- Mode: Python; coding: utf-8; indent-tabs-mode: nil; tab-width: 4 -*-
### BEGIN LICENSE
# Copyright (C) 2012 Pete Burgers <deltify81@gmail.com>
# This program is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License version 3, as published 
# by the Free Software Foundation.
# 
# This program is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranties of 
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
# PURPOSE.  See the GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License along 
# with this program.  If not, see <http://www.gnu.org/licenses/>.
### END LICENSE

import os
import sqlite3
import ConfigParser
import json
import logging
logger = logging.getLogger(__name__)

import deltify.utils as utils            

class Browser:
    pass

class FirefoxBrowser(Browser):
    PROFILE_PATH = os.path.expanduser("~/.mozilla/firefox/")
    PROFILE_FILE = "profiles.ini"
    BOOKMARKS_DB_FILE = "places.sqlite"
    TEMPORARY_DB_FILE = "deltify_places.sqlite"
    SQL_GET_BOOKMARKS = ("SELECT moz_bookmarks.title,moz_places.url FROM moz_bookmarks "
                         "LEFT JOIN moz_places WHERE moz_bookmarks.fk = moz_places.id "
                         "AND moz_bookmarks.title != 'null' AND moz_places.url LIKE '%http%';")
    TITLE_INDEX = 0
    URI_INDEX = 1

    def get_profiles(self):
        profiles = []
        try:
            parser = ConfigParser.SafeConfigParser()
            parser.read(os.path.join(self.PROFILE_PATH, self.PROFILE_FILE))
            for section in parser.sections():
                if section.startswith("Profile"):
                    default = None
                    if parser.has_option(section, "Default"):
                        default = parser.get(section, "Default")
                    path = parser.get(section, "Path")
                    name = parser.get(section, "Name")
                    if default == 1:
                        # Make the default first in the list
                        profiles.insert(0, (name, path))
                    else:
                        profiles.append((name, path))
        except Exception as error:
            logger.warn(error)
            pass
        return profiles

    def get_bookmarks(self, profile):
        # Create a copy of the places DB
        db_path = os.path.join(self.PROFILE_PATH, profile, self.BOOKMARKS_DB_FILE)
        temp_db_path = utils.create_temporary_copy(db_path, self.TEMPORARY_DB_FILE)
        conn = sqlite3.connect(temp_db_path)

        for bookmark in conn.execute(self.SQL_GET_BOOKMARKS).fetchall():
            yield ({'title': bookmark[self.TITLE_INDEX], 
                    'uri': bookmark[self.URI_INDEX]})

        # Close and delete DB copy
        conn.close()
        try:
            os.remove(temp_db_path)
        except OSError:
            pass

class ChromeBrowser(Browser):
    PROFILE_PATH = os.path.expanduser("~/.config/google-chrome/")
    PROFILE_FILE = "Local State"
    BOOKMARKS_DB_FILE = "Bookmarks"

    def get_profiles(self):
        profiles = []
        try:
            data = json.load(open(os.path.join(self.PROFILE_PATH, self.PROFILE_FILE)))
            profile_data = data['profile']
            for path in profile_data['info_cache']:
                name = profile_data['info_cache'][path]['name']
                if path == profile_data['last_used']:
                    # Make the default first in the list
                    profiles.insert(0, (name, path))
                else:
                    profiles.append((name, path))
        except Exception as error:
            logger.warn(error)
            pass
        return profiles
    
    def get_bookmarks(self, profile):
        bookmarks = []
        data = json.load(open(os.path.join(self.PROFILE_PATH, profile, self.BOOKMARKS_DB_FILE)))
        # Walk through children looking for bookmarks
        for root in data['roots']:
            bookmarks.extend(self._extract_bookmarks(data['roots'][root]))
        return bookmarks

    def _extract_bookmarks(self, item):
        bookmarks = []
        if 'url' in item and 'name' in item:
            bookmarks.append({'title': item['name'],
                              'uri': item['url']})
        if 'children' in item:
            for child in item['children']:
                bookmarks.extend(self._extract_bookmarks(child))

        return bookmarks

class ChromiumBrowser(ChromeBrowser):
    """ Identical to Chrome, but a different profile path """
    PROFILE_PATH = os.path.expanduser("~/.config/chromium/")


browsers = {"firefox": FirefoxBrowser,
            "chrome": ChromeBrowser,
            "chromium": ChromiumBrowser,
            }

def browser_from_name(name):
    BrowserClass = browsers[name.lower()]
    return BrowserClass()
