# -*- Mode: Python; coding: utf-8; indent-tabs-mode: nil; tab-width: 4 -*-
### BEGIN LICENSE
# Copyright (C) 2012 Pete Burgers <deltify81@gmail.com>
# This program is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License version 3, as published 
# by the Free Software Foundation.
# 
# This program is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranties of 
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
# PURPOSE.  See the GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License along 
# with this program.  If not, see <http://www.gnu.org/licenses/>.
### END LICENSE

import sqlite3
import os
import cPickle as pickle # faster pickle module
import datetime
import string
import logging
logger = logging.getLogger(__name__)

from gi.repository import Gtk # pylint: disable=E0611
from gi.repository import Gio # pylint: disable=E0611

import deltify.settings as settings
import deltify.fields as fields
import deltify.utils as utils
import deltify.browser as browser

BOOKMARKS_DB_PATH = os.path.join(settings.SETTINGS_PATH, "bookmarks.pickle")
BOOKMARKS_DB_VERSION = 1

CHARSET_AUTO = "auto"

FIELDS = fields.Fields([("rowid", int),
                        ("title", str),
                        ("uri", str),
                        ("filename", str),
                        ("scan_rate_mins", int, -1), # -1 means use the default
                        ("change_threshold", int, -1), # -1 means use the default
                        ("character_set", str, CHARSET_AUTO),
                        ("is_changed", bool, False),
                        ("is_error", bool, False),
                        ("is_skipped", bool, False),
                        ("error", str),
                        ("error_function", str),
                        ("error_traceback", str),
                        ("new_scan_time", int, 0), # Time format is UTC UNIX timestamp
                        ("old_scan_time", int, 0),
                        ("last_scan_time", int, 0),
                        ("next_scan_time", int, 0),
                        ("insert_style", str, "background-color: yellow !important;"),
                        ("delete_style", str, ""),
                        ])

class BookmarksStore(Gtk.ListStore):
    def __init__(self, browser_name, profile_path, bookmark_root_folder):
        Gtk.ListStore.__init__(self, *FIELDS.types)
        self.uris = [] # Keep a list of uris, in the same order as the ListStore items

        self._import_browser_bookmarks(browser_name, profile_path)
		self.load(update_only=True)
        self.save()

    def _import_browser_bookmarks(self, browser_name, profile_path):
        for bookmark in browser.browser_from_name(browser_name).get_bookmarks(profile_path):
            # Create a unique filename based on the Title and bookmark number
            filename = "".join([char for char in bookmark['title'] 
                                if char in string.ascii_letters][:50])
            filename += str(len(self))
            # We're keying off the URIs, so ensure they are unique
            if bookmark['uri'] not in self.uris:
                self.uris.append(bookmark['uri'])
                self.append(title=bookmark['title'], uri=bookmark['uri'], filename=filename)
    
    def load(self, update_only=False):
        """ If update_only is set, only update existing bookmarks """
        if not os.path.exists(BOOKMARKS_DB_PATH):
            logger.warn("No bookmarksDB found.")
            return

        try:
            with open(BOOKMARKS_DB_PATH, "rb") as db_file:
                version = pickle.load(db_file)
                if version != BOOKMARKS_DB_VERSION:
                    raise ValueError("Unsupported bookmarksDB version: %d" % version)
                bookmarks = pickle.load(db_file)
        except EOFError:
            logger.warn("Incomplete bookmarksDB - ignoring")
            return

        for bookmark in bookmarks:
            uri = bookmark["uri"]
            if uri in self.uris:
                self.update(self.uris.index(uri), update_title=False, **bookmark)
            elif not update_only:
                self.append(**bookmark)
            else:
                logger.info("Bookmark has been deleted: %s", bookmark["title"])

    def save(self):
        bookmarks = []
        for bookmark in self:
            bookmarks.append(FIELDS.dict_for(bookmark))

        with open(BOOKMARKS_DB_PATH, "wb") as db_file:
            pickle.dump(BOOKMARKS_DB_VERSION, db_file)
            pickle.dump(bookmarks, db_file)

    def append(self, **kwds):
        """Append a bookmark to the liststore. Any fields that aren't passed in take their default values."""
        Gtk.ListStore.append(self, FIELDS.values_for(**kwds))

    def update(self, index, update_title, **kwds):
        """
        Update the bookmark at the specified index of the liststore.
        If update_title is false, don't update the title field.
        """
        if not update_title:
            kwds["title"] = self[index][FIELDS.title]
        self[index] = FIELDS.values_for(**kwds)

    def update_with_scan(self, index, scan):
        old_filename = self[index][FIELDS.filename] + ".old.html"
        new_filename = self[index][FIELDS.filename] + ".new.html"
        diff_filename = self[index][FIELDS.filename] + ".diff.html"
        self[index][FIELDS.last_scan_time] = scan.time
        self[index][FIELDS.is_error] = False
        self[index][FIELDS.is_changed] = False

        if scan.error:
            self[index][FIELDS.is_error] = True
            self[index][FIELDS.error] = scan.error
            self[index][FIELDS.error_function] = scan.error_function
            self[index][FIELDS.error_traceback] = scan.error_traceback
            return

        if self[index][FIELDS.old_scan_time] == 0:
            # First time this page has been scanned
            logger.debug("Scan complete, first scan")
            utils.save_scan(old_filename, scan.data)
            self[index][FIELDS.old_scan_time] = scan.time
        else:
            # Check what's changed since the last scan. Ignore HTML tags
            diff_data, words_highlighted = scan.diff(new_filename, strip_tags=True)

            settings = Gio.Settings("net.launchpad.deltify")
            default_change_threshold = int(settings.get_string("default-change-threshold"))

            change_threshold = self[index][FIELDS.change_threshold]
            if change_threshold == fields.DEFAULT_VALUE:
                change_threshold = default_change_threshold
            logger.debug("Change threshold is %d" % change_threshold)

            if words_highlighted == 0:
                logger.info("Scan complete, no change")
            elif words_highlighted <= change_threshold:
                logger.info("Scan complete, change below threshold (%d words)", words_highlighted)
            else:
                logger.info("Scan complete, change detected above threshold (%d words)" 
                            % words_highlighted)
                if not self[index][FIELDS.is_changed]:
                    # This is a new change - overwrite .old with .new
                    utils.move_and_overwrite_scan(source=new_filename, dest=old_filename)
                    self[index][FIELDS.old_scan_time] = self[index][FIELDS.new_scan_time]

                self[index][FIELDS.is_changed] = True

        utils.save_scan(new_filename, scan.data)
        self[index][FIELDS.new_scan_time] = scan.time
        # Create diff between new and old pages, preserving HTML tags
        diff_data, _ = scan.diff(old_filename, strip_tags=False)
        utils.save_scan(diff_filename, diff_data)

    def get_diff_data(self, tree_iter):
        filename = self[tree_iter][FIELDS.filename] + ".diff.html"
        return utils.add_header(utils.load_scan(filename),
                                self[tree_iter][FIELDS.insert_style], 
                                self[tree_iter][FIELDS.delete_style])
    
    def mark_as_viewed(self, tree_iter):
        self[tree_iter][FIELDS.is_changed] = False
        self.save()

    def toggle_skipped(self, tree_iter):
        self[tree_iter][FIELDS.is_skipped] = not self[tree_iter][FIELDS.is_skipped]
        self.save()

    def set_change_threshold(self, tree_iter, change_threshold):
        self[tree_iter][FIELDS.change_threshold] = change_threshold
        self.save()

    def get_num_changes(self):
        changes = 0
        for bookmark in self:
            if bookmark[FIELDS.is_changed]:
                changes += 1
        return changes
