# -*- Mode: Python; coding: utf-8; indent-tabs-mode: nil; tab-width: 4 -*-
### BEGIN LICENSE
# Copyright (C) 2012 Pete Burgers <deltify81@gmail.com>
# This program is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License version 3, as published 
# by the Free Software Foundation.
# 
# This program is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranties of 
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
# PURPOSE.  See the GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License along 
# with this program.  If not, see <http://www.gnu.org/licenses/>.
### END LICENSE

import os
import webbrowser

import gettext
from gettext import gettext as _
gettext.textdomain('deltify')

from gi.repository import Gtk # pylint: disable=E0611
from gi.repository import GObject
from gi.repository import WebKit
from gi.repository import Unity
from gi.repository import Notify

import logging
logger = logging.getLogger(__name__)

import deltify_lib
from deltify.AboutDeltifyDialog import AboutDeltifyDialog
from deltify.PreferencesDeltifyDialog import PreferencesDeltifyDialog
from deltify.PropertiesDeltifyDialog import PropertiesDeltifyDialog
from deltify.DeltifyAssistant import DeltifyAssistant

from deltify.bookmarks import BookmarksStore, FIELDS
import deltify.settings as settings
import deltify.scanner as scanner
import deltify.task as task
import deltify.utils as utils
import deltify.renderers as renderers

#@TODO: Make translateable
ERROR_HTML = """
<html>
<head>
<style> 
#msg { text-align: center; }
body { background: #fff0f0; }
</style>
</head><body>
<div id="msg"><h2>Scan Error</h2>Sorry, this page could not be scanned: %(error)s</div>
</body></html>
"""
THROBBER_HTML = """
<html><head>
<style> #msg { position: absolute; left: 50%%; top: 50%%; }</style>
</head>
<body><div id="msg">Loading...</div></body></html>
"""

# See deltify_lib.Window.py for more details about how this class works
class DeltifyWindow(deltify_lib.Window):
    __gtype_name__ = "DeltifyWindow"
    
    def finish_initializing(self, builder): # pylint: disable=E1002
        """Set up the main window"""
        super(DeltifyWindow, self).finish_initializing(builder)

        self.AboutDialog = AboutDeltifyDialog
        self.PreferencesDialog = PreferencesDeltifyDialog
        self.DeltifyAssistant = DeltifyAssistant
        self.PropertiesDialog = PropertiesDeltifyDialog

        self.web_view_scroll = self.builder.get_object("web_view_scroll")
        self.bookmarks_tree = self.builder.get_object("bookmarks_tree")
        self.progress_bar = self.builder.get_object("progress_bar")
        self.scan_button = self.builder.get_object("scan_button")
        self.view_button = self.builder.get_object("view_button")
        self.skip_button = self.builder.get_object("skip_button")
        self.mnu_view = self.builder.get_object("mnu_view")
        self.mnu_properties = self.builder.get_object("mnu_properties")
        self.mnu_skip = self.builder.get_object("mnu_skip")
        self.mnu_change_source = self.builder.get_object("mnu_change_source")
        self.bookmarks_context_menu = self.builder.get_object("bookmarks_context_menu")
        self.browser_icons = {'firefox': self.builder.get_object("firefox_image"),
                              'chrome': self.builder.get_object("chrome_image"),
                              'chromium': self.builder.get_object("chromium_image")}

        # Manually connect the skip button/menu, so we can disable the handler when overriding its state
        self.skip_button_handler = self.skip_button.connect("toggled", self._on_skip_button_toggled)
        self.mnu_skip_handler = self.mnu_skip.connect("toggled", self._on_mnu_skip_toggled)

        # When packaged into /opt/, the desktop id is different.
        # Unsure how to handle this - can we use the app URI?
        self.launchers = [Unity.LauncherEntry.get_for_desktop_id("deltify.desktop"),
                          Unity.LauncherEntry.get_for_desktop_id("extras-deltify.desktop")]
        Notify.init("Deltify")

        self.scan_task = None
        self.properties = None

        self.settings = settings.Settings()
        self.settings.load()

        # WebKit view
        self.web_view = WebKit.WebView()
        self.web_view.connect("button-press-event", self.on_webview_button_event)
        self.web_view.connect("button-release-event", self.on_webview_button_event)
        self.web_view.set_can_focus(False)
        self.web_view_scroll.add(self.web_view)
        self.web_view.show()

        # Bookmarks sidebar
        self.bookmarks_store = None
        if self.settings.browser_profile is None:
            # Show the bookmarks import assistant immediately after this window has been created
            GObject.idle_add(self.show_assistant, True)
        else:
            self.load_bookmarks_store()
            self.view_button.set_icon_widget(self.browser_icons[self.settings.browser])

        self.on_bookmarks_tree_selection_changed()

        # Title column
        renderer = Gtk.CellRendererText()
        column = Gtk.TreeViewColumn("Title", renderer, text=FIELDS.title)
        column.set_cell_data_func(renderer, renderers.render_title)
        self.bookmarks_tree.append_column(column)

        self.update_launcher_count()

    def on_webview_button_event(self, button, event):
        """ Don't allow any webview navigation - block all mouse clicks by returning True """
        return True

    def on_mnu_change_source_activate(self, menu):
        self.show_assistant()

    # Break callback naming convention, since we want to hook it up ourselves
    def _on_mnu_skip_toggled(self, button):
        self._on_skip_button_toggled()

    def on_mnu_view_activate(self, button):
        self.on_view_button_clicked()

    def show_assistant(self, first_run=False):
        assistant = self.DeltifyAssistant() # pylint: disable=E1102
        assistant.set_transient_for(self)
        if first_run:
            # If it's the first time Deltify has run, cancelling 
            # the assistant closes the application
            assistant.show(on_close=self.on_assistant_close,
                           on_cancel=self.destroy)
        else:
            assistant.show(on_close=self.on_assistant_close)


    def on_assistant_close(self, browser, profile):
        self.settings.browser = browser
        self.settings.browser_profile = profile
        self.settings.save()
        
        self.view_button.set_icon_widget(self.browser_icons[browser])
        self.load_bookmarks_store()

    def load_bookmarks_store(self):
        self.bookmarks_store = BookmarksStore(self.settings.browser, 
                                              self.settings.browser_profile,
                                              self.settings.bookmark_root_folder)
        self.bookmarks_tree.set_model(self.bookmarks_store)

    def get_selected(self):
        selection = self.bookmarks_tree.get_selection()
        return selection.get_selected()

    def on_destroy(self, window, data=None):
        try:
            if self.scan_task is not None:
                self.scan_task.stop()
            if self.bookmarks_store is not None:
                self.bookmarks_store.save()
        finally:
            super(DeltifyWindow, self).on_destroy(window, data)

    def on_bookmarks_tree_selection_changed(self, selection=None):
        if selection is None:
            model, treeiter = self.get_selected()
        else:
            model, treeiter = selection.get_selected()

        if treeiter is None:
            is_skipped = False
            self.mnu_properties.set_state(Gtk.StateType.INSENSITIVE)
            self.mnu_skip.set_state(Gtk.StateType.INSENSITIVE)
            self.mnu_view.set_state(Gtk.StateType.INSENSITIVE)
            self.skip_button.set_state(Gtk.StateType.INSENSITIVE)
            self.view_button.set_state(Gtk.StateType.INSENSITIVE)
            self.web_view.load_string("", "text/html", "", "")
        else:
            is_skipped = model[treeiter][FIELDS.is_skipped]
            self.mnu_properties.set_state(Gtk.StateType.NORMAL)
            self.mnu_skip.set_state(Gtk.StateType.NORMAL)
            self.mnu_view.set_state(Gtk.StateType.NORMAL)
            self.skip_button.set_state(Gtk.StateType.NORMAL)
            self.view_button.set_state(Gtk.StateType.NORMAL)

            # Show "loading" message
            self.show_throbber()
            # Start loading the page
            GObject.idle_add(self.show_diff, model, treeiter)

            model.mark_as_viewed(treeiter)
            self.update_launcher_count()

        self.update_skipped_state(is_skipped)

    def on_bookmarks_tree_button_press_event(self, treeview, event):
        # Don't show the popup menu - it still needs work
        return
        if event.button == 3:
            x = int(event.x)
            y = int(event.y)
            time = event.time
            pthinfo = treeview.get_path_at_pos(x, y)
            if pthinfo is not None:
                path, col, cellx, celly = pthinfo
                treeview.grab_focus()
                treeview.set_cursor(path, col, 0)
                self.bookmarks_context_menu.popup( None, None, None, None, event.button, time)
            return True

    def on_mnu_properties_activate(self, widget):
        if self.properties is not None:
            # Only show one at a time
            return
        model, treeiter = self.get_selected()
        self.properties = self.PropertiesDialog()
        self.properties.run(model, treeiter)
        change_threshold = self.properties.get_change_threshold()
        if change_threshold is not None:
            model.set_change_threshold(treeiter, change_threshold)
            
        self.properties.destroy()
        self.properties = None

    def on_scan_button_clicked(self, button):
        self.scan_button.set_state(Gtk.StateType.INSENSITIVE)
        self.mnu_change_source.set_state(Gtk.StateType.INSENSITIVE)
        self.progress_bar.set_fraction(0)
        self.update_progress(0)
        self.progress_bar.set_visible(True)
        
        self.scan_task = task.Task(self.scan_all, 
                                   loop_callback=self.on_scan_progress,
                                   complete_callback=self.on_scan_finished)
        self.scan_task.start()

    # Break callback naming convention, since we want to hook it up ourselves
    def _on_skip_button_toggled(self, button=None):
        model, treeiter = self.get_selected()
        model.toggle_skipped(treeiter)
        self.update_skipped_state(model[treeiter][FIELDS.is_skipped])

    def on_view_button_clicked(self, button=None):
        model, treeiter = self.get_selected()
        if treeiter is not None:
            webbrowser.open(model[treeiter][FIELDS.uri])

    def scan_all(self):
        num_bookmarks = len(self.bookmarks_store)
        for index, bookmark in enumerate(self.bookmarks_store):
            if bookmark[FIELDS.is_skipped]:
                yield ((bookmark, index))
                continue
            scan = scanner.Scan()
            scan.run(bookmark)
            self.bookmarks_store.update_with_scan(index, scan)
            yield((bookmark, index))
            self.bookmarks_store.save()
        
    def on_scan_progress(self, bookmark, index):
        # Redraw the tree
        self.bookmarks_tree.queue_draw()
        # Deselect if the current selection has just been updated
        model, selected_bookmark = self.get_selected()
        if selected_bookmark and model[selected_bookmark][FIELDS.uri] == bookmark[FIELDS.uri]:
            logger.debug("Selected bookmark has changed state - deselecting")
            self.bookmarks_tree.get_selection().unselect_all()

        self.update_progress(index + 1)
        self.update_launcher_count()

    def on_scan_finished(self):
        self.scan_button.set_state(Gtk.StateType.NORMAL)
        self.mnu_change_source.set_state(Gtk.StateType.NORMAL)
        self.progress_bar.set_visible(False)
        self.update_launcher_progress(0, show=False)
        changes = self.bookmarks_store.get_num_changes()
        if changes == 0:
            message = _("No bookmarks have changed.")
        else:
            message = gettext.ngettext("One bookmark has changed.", "%d bookmarks have changed." % changes, changes)
        notification = Notify.Notification.new(_("Deltify Scan Complete"), 
                                               message,
                                               deltify_lib.helpers.get_media_file("deltify.svg"))
        notification.show()

    def update_progress(self, index):
        fraction = float(index) / len(self.bookmarks_store)
        self.progress_bar.set_fraction(fraction)
        self.update_launcher_progress(fraction, show=True)
        
        text = " "
        if index < len(self.bookmarks_store):
            # Add a little padding to the left
            text = " " + self.bookmarks_store[index][FIELDS.title]
        self.progress_bar.set_text(text)
        

    def show_throbber(self):
        self.web_view.load_string(THROBBER_HTML, 
                                  mime_type="text/html", 
                                  encoding="utf-8",
                                  base_uri="")

    def show_diff(self, model, treeiter):
        bookmark = model[treeiter]
        if bookmark[FIELDS.is_error]:
            self.show_error(model, treeiter)
        else:
            uri = bookmark[FIELDS.uri]
            diff_content = model.get_diff_data(treeiter)
            self.web_view.load_string(diff_content, mime_type="text/html", 
                                      encoding=utils.SCAN_FILE_ENCODING, base_uri=uri)

    def show_error(self, model, treeiter):
        error = model[treeiter][FIELDS.error]
        self.web_view.load_string(ERROR_HTML % vars(),
                                  mime_type="text/html", 
                                  encoding="utf-8",
                                  base_uri="")

    def update_skipped_state(self, is_skipped):
        # Update the skip button & menu state, but don't tell the event handlers
        self.skip_button.handler_block(self.skip_button_handler)
        self.mnu_skip.handler_block(self.mnu_skip_handler)
        self.skip_button.set_active(is_skipped)
        self.mnu_skip.set_active(is_skipped)
        self.skip_button.handler_unblock(self.skip_button_handler)
        self.mnu_skip.handler_unblock(self.mnu_skip_handler)

    def update_launcher_count(self):
        if self.bookmarks_store is None:
            num_changes = 0
        else:
            num_changes = self.bookmarks_store.get_num_changes()

        for launcher in self.launchers:
            launcher.set_property("count", num_changes)
            launcher.set_property("count_visible", (num_changes > 0))

    def update_launcher_progress(self, fraction, show):
        for launcher in self.launchers:
            launcher.set_property("progress", fraction)
            launcher.set_property("progress_visible", show)
