from gi.repository import Gtk
from cuttlefish.events import EventManager
from cuttlefish.plugins import CuttlePlugin
from cuttlefish.params import BoolParam
			
	

class ReflexUI:
	def __init__(self, container, ctrl, reflexes):
		self._container = container
		self._ctrl = ctrl
		self._reflexes = reflexes
		self._reflexes.connect('save', self.on_save)
		self._reflexes.connect('delete', self.on_delete)
		self._frames = {}
		self._eventMgr = EventManager()
	
		for reflex in self._reflexes.list():
			self._create(reflex)	

	def _set_details_text(self, label, reflex):
		e = self._eventMgr.get_class(reflex['event']['type'])
		if e.NAME != 'UNKOWN' and reflex['event']['isActive']:
			label.set_markup('<b>%s</b><small>\nTriggering \'%s\' will cause a reaction consisting of %d actions.</small>' % 
				(reflex['name'], e.NAME, len(reflex['actions'])))
		else:
			label.set_markup('<b>%s</b>\n<small>Manually starting this reflex will cause a reaction consisting of %d actions.</small>' % 
				(reflex['name'], len(reflex['actions'])))

	def on_save(self, sender, reflex):
		if reflex.record_id in self._frames:
			self._update(reflex)
		else:
			self._create(reflex)
		
	def on_delete(self, sender, record_id):
		if record_id in self._frames:
			self._delete(record_id)

	def _create(self, reflex):
		# container
		frame = Gtk.VBox(False, 8)
		hbox = Gtk.HBox(False, 0)
		btns = Gtk.VButtonBox()

		# data
		lblDetails = Gtk.Label()
		lblDetails.set_use_markup(True)
		self._set_details_text(lblDetails, reflex)
		
		#settings = Gtk.Settings.get_default()
		#tmp = settings.props.gtk_button_images
		#settings.props.gtk_button_images = True
		btnEdit = Gtk.Button('gtk-edit', use_stock=True)
		#btnDelete = Gtk.Button('gtk-delete', use_stock=True)

		# signals
		#btnDelete.connect('clicked', self._ctrl.on_delete_clicked, reflex.record_id)
		btnEdit.connect('clicked', self._ctrl.on_edit_clicked, reflex.record_id)

		# render
		hbox.pack_start(lblDetails, False, False, 10)
		btns.add(btnEdit)
		#btns.add(btnDelete)
		hbox.pack_end(btns, False, False, 10)
		frame.add(hbox)
		frame.add(Gtk.Separator())
		frame.show_all()
		self._container.add(frame)
		self._frames[reflex.record_id] = frame


	def _delete(self, record_id):
		self._frames[record_id].destroy()
		del self._frames[record_id]
		
	def _update(self, reflex):
		lblDetails = self._frames[reflex.record_id].get_children()[0].get_children()[0]
		self._set_details_text(lblDetails, reflex)



class DynamicUI:
	def __init__(self, frame):
		self._elements = {}
		self._editor = None
		self._frame = frame

	def clear(self):
		for key, ui in self._elements.items():
			ui.destroy()
		self._elements = {}

	def setup(self, element):
		if not hasattr(element, 'Editor'):
			self._elements = {}
			self._editor = None
			return

		self._params = element.get_params()
		self._editor = element.Editor()
		self._editor.set_params(self._params)
		if hasattr(element, 'triggerOnStartup') and hasattr(element.triggerOnStartup, '__call__') and 'triggerOnStartup' not in self._params:
			self._params['triggerOnStartup'] = False
		
		self._elements = self._editor.begin()
		order = hasattr(self._editor, 'ORDER') and self._editor.ORDER or self._elements		

		if hasattr(element, 'triggerOnStartup') and hasattr(element.triggerOnStartup, '__call__'):
			self._elements['triggerOnStartup'] = BoolParam('Also check on startup')
			if 'triggerOnStartup' not in order:
				order.append('triggerOnStartup')

		for key in order:
			lbl   =  self._elements[key].get_label()
			widget = self._elements[key].get_widget()

			lbl.set_halign(Gtk.Align.END)
			widget.set_halign(Gtk.Align.START)
			
			self._frame.add(lbl)
			self._frame.attach_next_to(widget, lbl, Gtk.PositionType.RIGHT, 1, 1)
			self._elements[key].set_value(self._params[key])
			
		self._frame.show_all()

	def get_values(self):
		result = {}
		for (key, ui) in self._elements.items():
			tmp = ui.get_value()
			if not tmp is None:
				result[key] = tmp
			else:
				result[key] = self._params[key]

		if self._editor:
			self._editor.set_params(result)
			self._editor.finish(self._elements)
			result = self._editor.get_params()
		return result

	def set_sensitive(self, sensitive):
		for key, ui in self._elements.items():
			ui.set_sensitive(sensitive)