from cuttlefish.actions import CuttleAction
from cuttlefish.events import CuttleEvent, DBusEvent
from cuttlefish.plugins import CuttlePlugin
from cuttlefish.params import SelectParam, BoolParam

from pyudev import Context, Monitor
from subprocess import Popen, PIPE
import threading
import dbus


class USBDeviceChanged(CuttleEvent):
	CATEGORY = "Hardware"
	PARAMS = {
		'devid' : ':',
		'devname' : ''
	}

	class Editor(CuttlePlugin.Editor):
		ORDER = ['devid']

		def _list_devs(self):
			devs = {}
			p = Popen(['lsusb'], stdout=PIPE)
			out, err = p.communicate()
			for dev in map(lambda x : x.split(' ', 6)[5:], out[:-1].split('\n')):
				devs[dev[0]] = dev[1]

			if self._params['devid'] != ':' and self._params['devid'] not in devs:
				devs[self._params['devid']] = '%s (currently not connected)' % self._params['devname']

			return devs


		def begin(self):
			return {
				'devid' : SelectParam('Monitored device', self._list_devs, str)
				} 

		def finish(self, ui):
			self._params['devname'] =  ui['devid'].get_caption()
			if '(currently not connected)' in self._params['devname']:
				self._params['devname'] = self._params['devname'].replace(' (currently not connected)', '')


	def __init__(self, isAvail):
		CuttleEvent.__init__(self)
		self._ctx = Context()
		self._isAvail = isAvail

	def triggerOnStartup(self):
		if self._is_avail() == self._isAvail:
			self.trigger()
		

	def setup(self):		
		self._killSwitch = threading.Event()
		self._thread = threading.Thread(target=self._monitor).start()
		
	def teardown(self):
		self._killSwitch.set()		


	def _monitor(self):
		isAvail = self._is_avail()

		while not self._killSwitch.isSet():
			oldIsAvail = isAvail
			isAvail = self._is_avail()

			if isAvail != oldIsAvail and isAvail == self._isAvail:
				self.trigger()

			self._killSwitch.wait(1)

	def _is_avail(self):
		idVendor, idProduct = self._params['devid'].split(':')
		devs = [d for d in self._ctx.list_devices().match_attribute('idVendor', idVendor).match_attribute('idProduct', idProduct)]
		# FIXME ugly, since maybe buggy with two devices of the same kind (at least not tested...)
		return len(devs) > 0
		


class USBDeviceAdded(USBDeviceChanged, CuttlePlugin):
	NAME = "USB device plugged in"
	DESCP = "React when a specified USB device gets plugged in"
	
	def __init__(self):
		USBDeviceChanged.__init__(self, True)
		CuttlePlugin.__init__(self)

			

class USBDeviceRemoved(USBDeviceChanged, CuttlePlugin):
	NAME ="USB device unplugged"
	DESCP = "React when a specified USB device gets plugged off"

	def __init__(self):
		USBDeviceChanged.__init__(self, False)
		CuttlePlugin.__init__(self)
			




class PowerChanged(DBusEvent):
	CATEGORY = 'Hardware'

	def __init__(self, isOnBatteryValue):
		DBusEvent.__init__(self, self.on_prop_changed, "Changed", "org.freedesktop.UPower", '/org/freedesktop/UPower', DBusEvent.USE_SYSTEM_BUS)
		self._isOnBatteryValue = isOnBatteryValue
				
	def triggerOnStartup(self):
		if self._is_on_battery() == self._isOnBatteryValue:
			self.trigger()

	def setup(self):		
		self._isOnBattery = self._is_on_battery()
		DBusEvent.setup(self)
		
	def on_prop_changed(self):
		tmp = self._is_on_battery()
		if tmp != self._isOnBattery:
			self._isOnBattery = tmp
			if self._isOnBattery == self._isOnBatteryValue:
				self.trigger()


	def _is_on_battery(self):
		obj = self._bus.get_object('org.freedesktop.UPower', '/org/freedesktop/UPower')
		props = dbus.Interface(obj, 'org.freedesktop.DBus.Properties')
		
		return props.Get('org.freedesktop.UPower', 'OnBattery')


class PowerDisconnected(PowerChanged, CuttlePlugin):
	NAME  = 'Power Cable unplugged'
	DESCP = 'React when a the power cable is unplugged'
	
	def __init__(self):
		PowerChanged.__init__(self, True)
		CuttlePlugin.__init__(self)


class PowerConnected(PowerChanged, CuttlePlugin):
	NAME  = 'Power Cable plugged in'
	DESCP = 'React when a the power cable is plugged in'
	
	def __init__(self):
		PowerChanged.__init__(self, False)
		CuttlePlugin.__init__(self)
