from cuttlefish.actions import CuttleAction
from cuttlefish.events import CuttleEvent
from cuttlefish.plugins import CuttlePlugin
from cuttlefish.params import StringParam, BoolParam, SelectAppParam, FolderParam, FileParam

from gi.repository import Wnck, Gio
import subprocess
import os
import os.path
import signal
import psutil

class RunApplication(CuttleAction, CuttlePlugin):
	NAME = 'Start application'
	DESCP = 'Starts an application'
	CATEGORY = 'Applications'
	PARAMS = {
		'path' : 'firefox',
		'wait' : False
	}

	class Editor(CuttlePlugin.Editor):
		ORDER = ['path', 'wait']

		def begin(self):
			return {
				'path': SelectAppParam('Application'),
				'wait': BoolParam('Wait until application exits')
			}

	def __init__(self):
		CuttleAction.__init__(self)
		CuttlePlugin.__init__(self)
		self._proc = None
		
	def execute(self):
		argv = self._params['path'].split(' ')
		self._proc = subprocess.Popen(argv)
		
		if self._params['wait']:
			self._proc.wait()

	def interrupt(self):
		if self._params['wait']:
			self._proc.kill()


class RunApplicationEx(CuttleAction, CuttlePlugin):
	NAME = 'Start application (advanced mode)'
	DESCP = 'Starts any executable application with extra parameters'
	CATEGORY = 'Applications'
	PARAMS = {
		'path': '/usr/bin/firefox', 
		'params': '', 
		'wdir': '', 
		'term': False,
		'wait' : False
	}

	class Editor(CuttlePlugin.Editor):
		ORDER = ['path', 'params', 'wdir', 'term', 'wait']

		def begin(self):
			return {
			'path': FileParam('Executable'),
			'params': StringParam('Parameters'),
			'wdir': FolderParam('Working directory'),
			'term' : BoolParam('Run in terminal'),
			'wait': BoolParam('Wait until application exits')
		}

	def __init__(self):
		CuttleAction.__init__(self)
		CuttlePlugin.__init__(self)
		self._proc = None
		
	def execute(self):
		argv = (self._params['path'] + ' ' + self._params['params']).split(' ')
		if self._params['term']:
			settings = Gio.Settings("org.gnome.desktop.default-applications.terminal")
			argv.insert(0, settings.get_string('exec'))
			execArg = settings.get_string('exec-arg')
			if execArg:
				argv.insert(1, execArg)

		wdir = self._params['wdir']
		if wdir == '':
			wdir = None
		self._proc = subprocess.Popen(argv,cwd=wdir)
		
		if self._params['wait']:
			self._proc.wait()

	def interrupt(self):
		if self._params['wait']:
			self._proc.kill()
		


class KillApplication(CuttleAction, CuttlePlugin):
	NAME = 'Stop an application'
	DESCP = 'Kill all instances of an application'
	CATEGORY = 'Applications'
	PARAMS = {
		'name' : 'firefox'
	}

	class Editor(CuttlePlugin.Editor):
		def begin(self):
			return {
				'name': SelectAppParam('Application')
			}
		
	def __init__(self):
		CuttleAction.__init__(self)
		CuttlePlugin.__init__(self)

	def execute(self):
		p = subprocess.Popen(['ps', '-A', 'h'], stdout=subprocess.PIPE)
		out, err = p.communicate()
		for line in out.splitlines():
			if self._params['name'].upper() in line.upper():
				pid = int(line.split(None, 1)[0])
				os.kill(pid, signal.SIGKILL)



class WnckScreenMonitor(CuttleEvent):
	CATEGORY = 'Applications'
	

	def __init__(self, event):
		CuttleEvent.__init__(self)
		self._screen = Wnck.Screen.get_default()
		self._event = event

	def setup(self):
		self._screen.force_update()
		self._hid = self._screen.connect(self._event, self.on_screen_event)

	def teardown(self):
		self._screen.disconnect(self._hid)


	def _app_to_pid(self, app):
		pid = app.get_pid()
		if pid == 0:
			# seems to be QT-window, getting pid from first window
			if app.get_n_windows() > 0:
				return app.get_windows()[0].get_pid()
			else:
				return 0
		else:
			return pid

	def _get_proc_name(self, proc):
		name = proc.name
		if '/' in name:
			# process started by interpreter (e.g. python /usr/bin/terminator)
			# Try to get clean name from cmdline: python foo/bar.py --> bar
			if len(proc.cmdline) > 1:
				name = os.path.basename(proc.cmdline[1]).split('.',1)[0]
		return name

	def _count_siblings(self, procOrName):
		pids = []
		ppids = []
		if isinstance(procOrName, str):
			thisName = procOrName
		else:
			thisName = self._get_proc_name(procOrName)

		for pid in psutil.get_pid_list():
			try:
				proc = psutil.Process(pid)
				otherName = self._get_proc_name(proc)
				if thisName == otherName:
					pids.append(proc.pid)
					ppids.append(proc.ppid)
			except:
				pass

		parented = [ppid for ppid in ppids if ppid in pids]		
		return len(pids) - len(parented)

	def on_screen_event(self, screen, data):
		pass


class ApplicationStart(WnckScreenMonitor, CuttlePlugin):
	NAME = 'Application starts'
	DESCP = "React when a specified application starts"
	PARAMS =  {
		'application' : 'firefox',
		'firstOnly' : False
	}
	
	class Editor(CuttlePlugin.Editor):
		ORDER = ['application', 'firstOnly']

		def begin(self):
			return {
				'application' : SelectAppParam('Application'), 
				'firstOnly' : BoolParam('Trigger only the first instance')
			}
		
	def __init__(self):
		WnckScreenMonitor.__init__(self, "window-opened")
		CuttlePlugin.__init__(self)


	def on_screen_event(self, screen, data):
		pid = self._app_to_pid(data.get_application())

		if pid > 0:
			proc = psutil.Process(pid)
			psname = self._get_proc_name(proc)
			suspect = os.path.basename(self._params['application'])
			
			if psname == suspect:
				if self._params['firstOnly']:
					if self._count_siblings(proc) == 1:
						self.trigger()
				else:
					self.trigger()



class ApplicationEnd(WnckScreenMonitor, CuttlePlugin):
	NAME = 'Application stops'
	DESCP = "React when a specified application finishes"
	CATEGORY = 'Applications'
	PARAMS =  {
		'application' : 'firefox',
		'lastOnly' : False
	}
	
	class Editor(CuttlePlugin.Editor):
		ORDER = ['application', 'lastOnly']

		def begin(self):
			return {
				'application' : SelectAppParam('Application'), 
				'lastOnly' : BoolParam('Trigger only the last instance')
			}
		
	def __init__(self):
		WnckScreenMonitor.__init__(self, "application-closed")
		CuttlePlugin.__init__(self)


	def setup(self):
		WnckScreenMonitor.setup(self)
		# need to manage pids on my own, because when the application-closed event is fired, the
		# proc-structure is not valid anymore
		self._pids = {}
		for pid in psutil.get_pid_list():
			try:
				_proc = psutil.Process(pid)
				psname = self._get_proc_name(_proc)
				self._pids[pid] = psname
			except:
				pass

		self._hid_open = self._screen.connect("window-opened", self.on_window_opened)
		
	def teardown(self):
		WnckScreenMonitor.teardown(self)
		self._screen.disconnect(self._hid_open)
		self._pids = {}


	def on_window_opened(self, screen, window):
		pid = self._app_to_pid(window.get_application())
		if pid > 0:
			self._pids[pid] = self._get_proc_name(psutil.Process(pid))

	
	def on_screen_event(self, screen, data):
		pid = self._app_to_pid(data)

		if pid in self._pids:
			psname = self._pids[pid]
			suspect = os.path.basename(self._params['application'])
			del self._pids[pid]
			
			if psname == suspect:
				if self._params['lastOnly']:
					if self._count_siblings(psname) == 0:
						self.trigger()
				else:
					self.trigger()