from gi.repository import Gtk, GdkPixbuf, GMenu, Gio, Pango
import os
import os.path
from xdg import IconTheme


class Param:
	def __init__(self, label):
		self._label = Gtk.Label(label)
		self._widget = None
		
	def get_label(self):
		return self._label
	
	def get_widget(self):
		self._widget = self._create_widget()
		return self._widget

	def set_value(self, value):
		pass
		
	def get_value(self):
		return None
		
	def destroy(self):
		self._label.destroy()
		if self._widget:
			self._widget.destroy()
		
	def set_sensitive(self, sensitive):
		self._widget.set_sensitive(sensitive)

		
class StringParam(Param):
	def __init__(self, label):
		Param.__init__(self, label)

	def _create_widget(self):
		return Gtk.Entry()
		
	def set_value(self, value):
		self._widget.set_text(value)
		
	def get_value(self):
		return self._widget.get_text()		


class BoolParam(Param):
	def __init__(self, label):
		Param.__init__(self, label)

	def _create_widget(self):
		return Gtk.Switch()
		
	def set_value(self, value):
		self._widget.set_active(value)
		
	def get_value(self):
		return self._widget.get_active()		


class SelectParam(Param):
	def __init__(self, label, options, idxType=int):
		Param.__init__(self, label)
		self._model = Gtk.ListStore(idxType, str)
		if hasattr(options, '__call__'):
			self._reloadCallback = options
			options = self._reloadCallback()
		else:
			self._reloadCallback = None

		for value, label in options.items():
			self._model.append([value, label])


	def _on_btnReload_click(self, widget, data=None):
		tmp = self.get_value()

		self._model.clear()
		for value, label in self._reloadCallback().items():
				self._model.append([value, label])

		self.set_value(tmp)


	def _create_widget(self):
		hbox = Gtk.HBox()
		cb = Gtk.ComboBox()
		cb.set_model(self._model)
		cr = Gtk.CellRendererText()
		cr.props.ellipsize = Pango.EllipsizeMode.END
		cb.pack_start(cr, True)
		cb.add_attribute(cr, 'text', 1)

		hbox.add(cb)
		if self._reloadCallback:
			btnReload = Gtk.Button()
			btnReload.set_image(Gtk.Image.new_from_stock(Gtk.STOCK_REFRESH, Gtk.IconSize.BUTTON))
			btnReload.connect('clicked', self._on_btnReload_click)
			hbox.add(btnReload)

		return hbox
		
	def set_value(self, value):
		for idx, row in enumerate(self._model):
			if value == row[0]:
				self._widget.get_children()[0].set_active(idx)
				break
		
	def get_value(self):
		idx = self._widget.get_children()[0].get_active()		
		if idx >= 0:
			return self._model[idx][0]

	def get_caption(self):
		idx = self._widget.get_children()[0].get_active()
		if idx < 0:
			return ''
		else:
			return self._model[idx][1]


class IntParam(Param):
	def __init__(self, label, adjustment=None):
		Param.__init__(self, label)
		self._adjustment = adjustment

	def _create_widget(self):
		spinner = Gtk.SpinButton()
		if self._adjustment:
			spinner.set_adjustment(self._adjustment)
		else:
			import sys
			spinner.set_adjustment(Gtk.Adjustment(0, 0, sys.maxint, 1, 0, 0))
		spinner.set_numeric(True)
		spinner.set_update_policy(Gtk.SpinButtonUpdatePolicy.IF_VALID)
		return spinner
		
	def set_value(self, value):
		self._widget.set_value(int(value))
		
	def get_value(self):
		return self._widget.get_value()


class FileParam(Param):
	def __init__(self, label, filters=None):
		Param.__init__(self, label)
		self._filters = filters
		self._path = ""

	def _create_widget(self):
		btn = Gtk.Button('...')
		btn.connect('clicked', self.on_btn_click)
		return btn
		
	def set_value(self, value):
		self._path = value
		s = os.path.basename(value)
		if s == '':
			s = '...'
		self._widget.set_label(s)
		
	def get_value(self):
		return self._path

	def on_btn_click(self, widget, data=None):
		dialog = Gtk.FileChooserDialog("Select File", None, 
										Gtk.FileChooserAction.OPEN,
										(Gtk.STOCK_CANCEL, Gtk.ResponseType.CANCEL,
										Gtk.STOCK_OPEN, Gtk.ResponseType.OK))
		dialog.set_default_response(Gtk.ResponseType.OK)

		if self._filters:
			for f in self._filters:
				dialog.add_filter(f)

		dialog.set_filename(self._path)
		response = dialog.run()
		if response == Gtk.ResponseType.OK:
			self._path = dialog.get_filename()
			self._widget.set_label(os.path.basename(self._path))
		dialog.destroy()



class FolderParam(Param):
	def __init__(self, label):
		Param.__init__(self, label)
		self._path = ""

	def _create_widget(self):
		btn = Gtk.Button('...')
		btn.connect('clicked', self.on_btn_click)
		return btn
		
	def set_value(self, value):
		self._path = value
		s = os.path.basename(value)
		if s == '':
			s = '...'
		self._widget.set_label(s)
		
	def get_value(self):
		return self._path

	def on_btn_click(self, widget, data=None):
		dialog = Gtk.FileChooserDialog("Select Folder", None, 
										Gtk.FileChooserAction.SELECT_FOLDER,
										(Gtk.STOCK_CANCEL, Gtk.ResponseType.CANCEL,
										Gtk.STOCK_OPEN, Gtk.ResponseType.OK))
		dialog.set_default_response(Gtk.ResponseType.OK)

		dialog.set_filename(self._path)
		response = dialog.run()
		if response == Gtk.ResponseType.OK:
			self._path = dialog.get_filename()
			self._widget.set_label(os.path.basename(self._path))
		dialog.destroy()


class SelectAppParam(Param):
	def __init__(self, label):
		Param.__init__(self, label)
		self._model = Gtk.ListStore(str, GdkPixbuf.Pixbuf, str)
		self._model.set_sort_column_id(2, Gtk.SortType.ASCENDING)
		t = GMenu.Tree.new('applications.menu', GMenu.TreeFlags.NONE)
		t.load_sync()
		self._theme = Gtk.IconTheme.get_default()
		self._size = 24
		self._list_entries(t.get_root_directory())

	def _list_entries(self, dir):
		iter = dir.iter()
		itemtype = iter.next()
		while itemtype != GMenu.TreeItemType.INVALID:
			if itemtype == GMenu.TreeItemType.ENTRY:
				appinfo = iter.get_entry().get_app_info()
				i = appinfo.get_icon()
				try:
					if isinstance(i, Gio.ThemedIcon):
						icon = self._theme.choose_icon(i.get_names(), self._size, Gtk.IconLookupFlags.USE_BUILTIN)
						if icon:
							icon = icon.load_icon()
						else:
							icon = self._theme.load_icon('application-x-executable', self._size, Gtk.IconLookupFlags.USE_BUILTIN)

					elif isinstance(i, Gio.FileIcon):
						icon = self._theme.load_icon('application-x-executable', self._size, Gtk.IconLookupFlags.USE_BUILTIN)
						# FIXME: Acccessing i.get_file() causes core dump
						#try:
						#	icon = GdkPixbuf.Pixbuf.new_from_file_at_size(i.get_file().get_path(), self._size, self._size)
				except:
					icon = self._theme.load_icon('application-x-executable', self._size, Gtk.IconLookupFlags.USE_BUILTIN)
					
				self._model.append([appinfo.get_executable(), icon, appinfo.get_name()])
			elif itemtype == GMenu.TreeItemType.DIRECTORY:
				self._list_entries(iter.get_directory())
			itemtype = iter.next()

	def _create_widget(self):
		cb = Gtk.ComboBox()
		cb.set_model(self._model)
		tr = Gtk.CellRendererText()
		ir = Gtk.CellRendererPixbuf()

		tr.props.ellipsize = Pango.EllipsizeMode.END
		cb.pack_start(ir, False)
		cb.pack_start(tr, False)
		cb.add_attribute(ir, 'pixbuf', 1)
		cb.add_attribute(tr, 'text', 2)
		return cb
		
	def set_value(self, value):
		iter = self._model.get_iter_first()

		while iter:
			if self._model.get_value(iter, 0) == value:
				self._widget.set_active_iter(iter)
				return
			iter = self._model.iter_next(iter)

		
	def get_value(self):
		idx = self._widget.get_active()		
		return self._model[idx][0]


class TimeParam(Param):

	def __init__(self, label):
		Param.__init__(self, label)

	def _create_widget(self):
		hbox = Gtk.HBox()
		
		sHour = Gtk.SpinButton()
		sHour.set_adjustment(Gtk.Adjustment(0, 0, 60, 1, 0, 0))
		sHour.set_numeric(True)
		sHour.set_update_policy(Gtk.SpinButtonUpdatePolicy.IF_VALID)
		hbox.add(sHour)
		
		sMin = Gtk.SpinButton()
		sMin.set_adjustment(Gtk.Adjustment(0, 0, 60, 1, 0, 0))
		sMin.set_numeric(True)
		sMin.set_update_policy(Gtk.SpinButtonUpdatePolicy.IF_VALID)
		hbox.add(sMin)
		
		return hbox


	def set_value(self, value):
		hour, mins = value
		sHour, sMin = self._widget.get_children()
		sHour.set_value(int(hour))
		sMin.set_value(int(mins))
		
	def get_value(self):
		sHour, sMin = self._widget.get_children()
		return (sHour.get_value(), sMin.get_value())


class MultiSelectParam(Param):
	def __init__(self, label, options, idxType=int):
		Param.__init__(self, label)
		self._model = Gtk.ListStore(idxType, str)
		if hasattr(options, '__call__'):
			self._reloadCallback = options
			options = self._reloadCallback()
		else:
			self._reloadCallback = None

		for value, label in options.items():
			self._model.append([value, label])


	def _on_btnReload_click(self, widget, data=None):
		tmp = self.get_value()

		self._model.clear()
		for value, label in self._reloadCallback().items():
				self._model.append([value, label])

		self.set_value(tmp)


	def _create_widget(self):
		hbox = Gtk.ScrolledWindow()
		hbox.props.width_request = 150
		hbox.props.height_request = 100
		tv = Gtk.TreeView()
		tv.props.headers_visible = False
		tv.set_model(self._model)
		tv.get_selection().set_mode(Gtk.SelectionMode.MULTIPLE)
		cr = Gtk.CellRendererText()
		column = Gtk.TreeViewColumn("Action", Gtk.CellRendererText(), text=1)
		tv.append_column(column) 
		cr.props.ellipsize = Pango.EllipsizeMode.END
		
		hbox.add(tv)
		if self._reloadCallback:
			btnReload = Gtk.Button()
			btnReload.set_image(Gtk.Image.new_from_stock(Gtk.STOCK_REFRESH, Gtk.IconSize.BUTTON))
			btnReload.connect('clicked', self._on_btnReload_click)
			hbox.add(btnReload)

		return hbox
		
	def set_value(self, value):
		sel = self._widget.get_children()[0].get_selection()
		for v in value:
			sel.select_path(v)

	def get_value(self):
		model, paths = self._widget.get_children()[0].get_selection().get_selected_rows()
		return [model[model.get_iter(path)][0] for path in paths]

	def get_captions(self):
		model, paths = self._widget.get_children()[0].get_selection().get_selected_rows()
		return [model[model.get_iter(path)][1] for path in paths]