from cuttlefish.reflexes import REFLEXES
from cuttlefish.ui import ReflexUI
from cuttlefish.events import EventManager
from cuttlefish.actions import ActionManager
from cuttlefish.plugins import format_plugin_exc
from cuttlefish_lib.helpers import get_media_file

import threading
from gi.repository import Notify, Gio, GObject
 
Notify.init("cuttlefish")
import logging

class Handler(threading.Thread):
	def __init__(self, actions, finished_callback, reflex):
		threading.Thread.__init__(self)
		self._logger = logging.getLogger('cuttlefish.engine.Handler')
		self._reflex = reflex
		self._actions = actions
		self._dead = threading.Event()
		self._activeAction = None
		self._finished_callback = finished_callback

	def run(self):
		for action in self._actions:
			self._activeAction = action
			try:
				self._logger.info("Executing action '%s' for '%s'" % (action.NAME, self._reflex['name']))
				try:
					action.execute()
				except:
					action.logger.error(format_plugin_exc())
			finally:
				if self._dead.isSet():
					break
		if not self._dead.isSet():
			self._logger.info("Finished reaction for '%s'" % self._reflex['name'])
			self._finished_callback(threading.current_thread().getName())

	def interrupt(self):
		self._dead.set()
		if self._activeAction:
			self._activeAction.interrupt()


class Engine(GObject.GObject):
	__gsignals__ = {
			'startup': (GObject.SIGNAL_RUN_FIRST, None, ()),
			'shutdown': (GObject.SIGNAL_RUN_FIRST, None, ())
		}

	def __init__(self):
		GObject.GObject.__init__(self)
		self._logger = logging.getLogger('cuttlefish.engine.Engine')
		self._eventMgr = EventManager()
		self._actionMgr = ActionManager()
		self._reflexes = REFLEXES
		self._settings = Gio.Settings("net.launchpad.cuttlefish")
			
	def on_save(self, sender, reflex):
		if reflex.record_id in self._events:
			self._update_reflex(reflex)
		else:
			self._add_reflex(reflex)
		
	def on_delete(self, sender, record_id):
		if record_id in self._events:
			self._delete_reflex(record_id)
	
	def _add_reflex(self, reflex):
		e = self._eventMgr.get_instance(reflex)
		self._events[reflex.record_id] = e

		e.hid = e.connect('triggered', self.on_event_triggered, reflex.record_id)
		try:
			e.setup()
		except:
			e.logger.error(format_plugin_exc())
			e.disconnect(e.hid)
			del self._events[reflex.record_id]
	
	def _update_reflex(self, reflex):
		self._delete_reflex(reflex.record_id)
		self._add_reflex(reflex)
		   

	def _delete_reflex(self, record_id):
		e = self._events[record_id]
		try:
			try:
				e.teardown()
			except:
				e.logger.error(format_plugin_exc())
		finally:
			e.disconnect(e.hid)
			del self._events[record_id]
			
	def setup(self):
		self._threads = {}
		self._events = {}
		
		for reflex in self._reflexes.list():
			self._add_reflex(reflex)
			event = self._events[reflex.record_id]
			if hasattr(event, 'triggerOnStartup') and hasattr(event.triggerOnStartup, '__call__'):
				params = event.get_params()
				if 'triggerOnStartup' in params and params['triggerOnStartup'] == True:
					event.triggerOnStartup()
		

		self._saveId = self._reflexes.connect('save', self.on_save)
		self._delId = self._reflexes.connect('delete', self.on_delete)
		self.emit('startup')

		

	def teardown(self):
		self.emit('shutdown')
		
		self._reflexes.disconnect(self._saveId)
		self._reflexes.disconnect(self._delId)

		for rid, e in self._events.items():
			self._delete_reflex(rid)

		for tname, thread in self._threads.items():
			thread.interrupt()

		self._events = {}
		self._threads = {}


	def execute(self, widget, record_id):
		self.on_event_triggered(self._events[record_id], record_id, True)
		
	
	def on_event_triggered(self, event, record_id, force=False):
		reflex = self._reflexes.get(record_id)
		if reflex['event']['isActive'] or force:
			if force:
				self._logger.info("Manually triggered reflex '%s'" % reflex['name'])
			else:
				self._logger.info("reflex '%s' triggered by '%s'" % (reflex['name'], event.NAME))

			if self._settings.get_boolean("notifications") and reflex['showNotification']:
				n = Notify.Notification.new('Cuttlefish', reflex['name'], get_media_file("tentacle.png"))
				n.show()

			actions = []
			for i in xrange(len(reflex['actions'])):
				actions.append(self._actionMgr.get_instance(reflex, i))
			t = Handler(actions, self.on_handle_finished, reflex)
			self._threads[t.getName()] = t
			t.start()

	def on_handle_finished(self, tname):
		if tname in self._threads:
			del self._threads[tname]

ENGINE = Engine()