# -*- Mode: Python; coding: utf-8; indent-tabs-mode: nil; tab-width: 4 -*-
### BEGIN LICENSE
# Copyright (C) 2012 Alexander von Bremen-Kuehne <noneed4anick@gmail.com>
# This program is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License version 3, as published 
# by the Free Software Foundation.
# 
# This program is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranties of 
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
# PURPOSE.  See the GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License along 
# with this program.  If not, see <http://www.gnu.org/licenses/>.
### END LICENSE

import gettext
from gettext import gettext as _
gettext.textdomain('cuttlefish')

from gi.repository import GObject, Gtk, Pango # pylint: disable=E0611
import logging
logger = logging.getLogger('cuttlefish')

from cuttlefish_lib import Window
from cuttlefish.AboutCuttlefishDialog import AboutCuttlefishDialog
from cuttlefish.PreferencesCuttlefishDialog import PreferencesCuttlefishDialog

from cuttlefish.reflexes import REFLEXES
from cuttlefish.plugins import MANAGER, EXAMPLE_CODE
from cuttlefish.ui import ReflexUI, DynamicUI
from cuttlefish.events import EventManager
from cuttlefish.actions import ActionManager
from cuttlefish.SelectDialog import SelectDialog
from cuttlefish.PluginbeginnerDialog import PluginbeginnerDialog
from cuttlefish.engine import ENGINE
import os
import os.path
from subprocess import Popen



# See cuttlefish_lib.Window.py for more details about how this class works
class CuttlefishWindow(Window):
    __gtype_name__ = "CuttlefishWindow"
    
    def finish_initializing(self, builder): # pylint: disable=E1002
        """Set up the main window"""
        super(CuttlefishWindow, self).finish_initializing(builder)
        # hack to remove uneeded seperator
        self.ui.helpMenu.get_children()[1].set_visible(False)

        self.AboutDialog = AboutCuttlefishDialog
        self.PreferencesDialog = PreferencesCuttlefishDialog
        
        self._eventMgr = EventManager()
        self._actionMgr = ActionManager()
        self._reflexes = REFLEXES

        self._editAction = None
        self._editItem = None
        self._editActions = Gtk.ListStore(GObject.TYPE_PYOBJECT, str)

        self._bUI = ReflexUI(self.ui.bevList, self, self._reflexes)
        self._eventUI = DynamicUI(self.ui.tblBasicConf)
        self._actionUI = DynamicUI(self.ui.tblActionConf)

        column = Gtk.TreeViewColumn("Action", Gtk.CellRendererText(), text=1)
        self.ui.tvActions.append_column(column)     

        self.ui.togEvent.connect('notify::active', self.on_togEvent_toggle)
        self.ui.togNotification.connect('notify::active', self.on_togNotification_toggle)
        self.ui.togIndicator.connect('notify::active', self.on_togIndicator_toggle)

        self.ui.toolbar.get_style_context().add_class(Gtk.STYLE_CLASS_PRIMARY_TOOLBAR)
        self.ui.tbActions.get_style_context().add_class(Gtk.STYLE_CLASS_INLINE_TOOLBAR)
        self._toggleEmptyViewIfNeeded()
        self.connect('delete-event', self.on_delete_window)

    def on_mnu_close_window_activate(self, widget, data=None):
        self.on_delete_window(widget, data)

    def on_delete_window(self, widget, data=None):
        self.hide()
        if self._editItem:
            self.on_tbGoBack_clicked(None, None)
        return True

    def _toggleEmptyViewIfNeeded(self):
        if len(self._reflexes.list()) == 0:
            self.ui.lblEmpty.show()
        else:        
            self.ui.lblEmpty.hide()
        
    #basic conf
    def on_txtName_changed(self, widget, data=None):
        self._editItem['name'] = widget.get_text()

    def on_togEvent_toggle(self, widget, data=None):
        self._editItem['event']['isActive'] = widget.get_active()
        self.ui.btnEvent.set_sensitive(self._editItem['event']['isActive'])
        self._eventUI.set_sensitive(self._editItem['event']['isActive'])


    def on_togNotification_toggle(self, widget, data=None):
        self._editItem['showNotification'] = widget.get_active()


    def on_togIndicator_toggle(self, widget, data=None):
        self._editItem['listInIndicator'] = widget.get_active()
        
        
    def on_btnEvent_clicked(self, widget, data=None):
        dlg = SelectDialog()
        dlg.set_title("Please select an event type")
        model = self._eventMgr.get_liststore()
        dlg.set_models(model, self._eventMgr.get_categories())

        idx = dlg.run()
        dlg.hide()

        if idx > -1:
            oldType = self._editItem['event']['type']
            newType = model[idx][0]
            
            if oldType == newType:
                return
                
            self._editItem['event']['type'] = newType
            
            event = self._eventMgr.get_instance(self._editItem)
            self.ui.btnEvent.set_label(event.NAME)
            self._eventUI.clear()
            self._eventUI.setup(event)
                
        

    def on_pluginsReload_activate(self, widget, data=None):
        rid = self._editItem and self._editItem.record_id or None

        if rid:
            self.on_tbGoBack_clicked(widget, data)
        
        ENGINE.teardown()
        MANAGER.reload()
        ENGINE.setup()

        if rid:
            self.on_edit_clicked(widget, rid)


    def on_pluginsEdit_activate(self, widget, data=None):
        plugindir = os.path.expanduser('~/.cuttlefish/plugins')
        if not os.path.exists(plugindir):
            os.mkdir(plugindir)

        pluginfile = os.path.join(plugindir, 'myplugins.py')
        if not os.path.isfile(pluginfile):
            with open(pluginfile, 'w+') as f:
                f.write(EXAMPLE_CODE)
            dlg = PluginbeginnerDialog()
            dlg.run()
            dlg.destroy()
        Popen(['gnome-open', pluginfile])
            


    # action conf
    def on_tbNewAction_clicked(self, widget, data=None):
        dlg = SelectDialog()
        dlg.set_title("Please select an action type")
        model = self._actionMgr.get_liststore()
        dlg.set_models(model, self._actionMgr.get_categories())

        actionIdx = dlg.run()
        dlg.hide()

        if actionIdx > -1:
            action = self._actionMgr.get_class(model[actionIdx][0])()
            self._editActions.append([action, action.NAME])
            self.ui.tvActions.set_cursor(len(self._editActions)-1)
            self._checkActionControls()


    def on_tbRemoveAction_clicked(self, widget, data=None):
        path, col = self.ui.tvActions.get_cursor()
        if path:
            self._editAction = None
            i = self._editActions.get_iter(path)
            self._editActions.remove(i)
            self._checkActionControls()

    def on_tbMvActionDown_clicked(self, widget, data=None):
        path, col = self.ui.tvActions.get_cursor()
        if path:
            i = self._editActions.get_iter(path)
            n = self._editActions.iter_next(i)
            if n:
                self._editActions.swap(i, n)
                self._checkActionControls()

    def on_tbMvActionUp_clicked(self, widget, data=None):
        path, col = self.ui.tvActions.get_cursor()
        if path:
            i = self._editActions.get_iter(path)
            if path.prev():
                p = self._editActions.get_iter(path)
                self._editActions.swap(i, p)
                self._checkActionControls()
      

    def on_tvActions_cursor_changed(self, widget, data=None):
        if self._editAction:
            self._editAction.set_params(self._actionUI.get_values())

        self._checkActionControls()
        self._actionUI.clear()

        path, col = self.ui.tvActions.get_cursor()
        if path:
            i = self._editActions.get_iter(path)
            self._editAction = self._editActions.get_value(i, 0)
            self._actionUI.setup(self._editAction)
        else:
            self._editAction = None


    def _checkActionControls(self):
        path, col = self.ui.tvActions.get_cursor()
        if path:
            idx = path.get_indices()[0]
        else:
            idx = -1

        count = len(self._editActions)

        self.ui.tbRemoveAction.set_sensitive(count > 0)
        self.ui.tbMvActionUp.set_sensitive(idx > 0)
        self.ui.tbMvActionDown.set_sensitive(idx < count-1)

    # show / hide conf
    def on_edit_clicked(self, widget, record_id):
        self.ui.tbNew.hide()
        self.ui.main.hide()
        self.ui.conf.show()
        self.ui.tbGoBack.show()
        self.ui.tbDuplicate.show()
        self.ui.tbDelete.show()
        self.ui.conf.set_current_page(0)
        
        self._editItem = self._reflexes.get(record_id)
        self.ui.txtName.set_text(self._editItem['name'])
        self.ui.togEvent.set_active(self._editItem['event']['isActive'])
        self.ui.togNotification.set_active(self._editItem['showNotification'])
        self.ui.togIndicator.set_active(self._editItem['listInIndicator'])
        self._eventUI.setup(self._eventMgr.get_instance(self._editItem))
        
        event = self._eventMgr.get_class(self._editItem['event']['type'])
        self.ui.btnEvent.set_label(event.NAME)

        self._editAction = None
        self._editActions.clear()
        self.ui.tvActions.set_model(self._editActions)

        for i in xrange(len(self._editItem['actions'])):
            action = self._actionMgr.get_instance(self._editItem, i)
            self._editActions.append([action, action.NAME])
            if i == 0:
                self._editAction = action
                self.ui.tvActions.set_cursor(0)
                
        self.ui.btnEvent.set_sensitive(self._editItem['event']['isActive'])
        self._eventUI.set_sensitive(self._editItem['event']['isActive'])


    def on_tbGoBack_clicked(self, widget, data=None):
        self._editItem['event']['params'].update(self._eventUI.get_values())
        if self._editAction:
            self._editAction.set_params(self._actionUI.get_values())
            self._editAction = None

        actions = []
        for row in self._editActions:
            a = row[0]
            actions.append({
                'type' : a.__class__.__name__,
                'params': a.get_params()
                })
        self._editItem['actions'] = actions     

        self._editActions.clear()
        self._reflexes.save(self._editItem)
        self._editItem = None
        
        self._eventUI.clear()
        self.ui.tbGoBack.hide()
        self.ui.tbDelete.hide()
        self.ui.tbDuplicate.hide()
        self.ui.conf.hide()
        self.ui.main.show()
        self.ui.tbNew.show()
        
    
    def on_tbNew_clicked(self, widget, data=None):
        newItem = self._reflexes.create()
        self._reflexes.save(newItem)
        self._toggleEmptyViewIfNeeded()    
        self.on_edit_clicked(widget, newItem.record_id)
        

    def on_tbDelete_clicked(self, widget, data=None):        
        dialog = Gtk.MessageDialog(self, 0, Gtk.MessageType.QUESTION, Gtk.ButtonsType.YES_NO, "Do you really want to delete this reflex?")
        response = dialog.run()
        if response == Gtk.ResponseType.YES:
            record_id = self._editItem.record_id
            self.on_tbGoBack_clicked(widget, data)
            self._reflexes.delete(record_id)
            self._toggleEmptyViewIfNeeded()
        dialog.destroy()

    def on_tbDuplicate_clicked(self, widget, data=None):
        newItem = self._reflexes.create()
        newItem['name'] = 'Duplicate of %s' % self._editItem['name']
        newItem['event'] = self._editItem['event']
        newItem['actions'] = self._editItem['actions']

        self.on_tbGoBack_clicked(widget, data)
        self._reflexes.save(newItem)
        self.on_edit_clicked(widget, newItem.record_id)


    def on_delete_clicked(self, widget, record_id):
        dialog = Gtk.MessageDialog(self, 0, Gtk.MessageType.QUESTION, Gtk.ButtonsType.YES_NO, "Do you really want to delete this reflex?")
        response = dialog.run()
        if response == Gtk.ResponseType.YES:
            self._reflexes.delete(record_id)
            self._toggleEmptyViewIfNeeded()
        dialog.destroy()