# -*- Mode: Python; coding: utf-8; indent-tabs-mode: nil; tab-width: 4 -*-
### BEGIN LICENSE
# Copyright (C) 2012 John Vrbanac <john.vrbanac (at) linux.com>
# This program is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License version 3, as published 
# by the Free Software Foundation.
# 
# This program is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranties of 
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
# PURPOSE.  See the GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License along 
# with this program.  If not, see <http://www.gnu.org/licenses/>.
### END LICENSE

from gi.repository import GObject
import os
import urlparse
import pygst
pygst.require("0.10")
import gst
import logging
logger = logging.getLogger('cuckoo')


class AudioPlayer(GObject.GObject):
    # Consts
    PLAYING = "playing"
    STOPPED = "stopped"
    STATE_CHANGED = "state-changed"

    # Signals
    __gsignals__ = {
            STATE_CHANGED: (GObject.SIGNAL_RUN_FIRST, GObject.TYPE_NONE, (GObject.TYPE_STRING,))
    }

    # Member Variables
    __loop = False
    __player = None
    __playing = False
    __task_id = ""

    def __setup_player(self):
        # Resorting to playbin2 for now
        self.__player = gst.element_factory_make("playbin2", "player")
        fakesink = gst.element_factory_make("fakesink", "fakesink")
        self.__player.set_property("video-sink", fakesink)
        bus = self.__player.get_bus()
        bus.add_signal_watch()
        bus.connect("message", self.on_message)

    def __init__(self):
        GObject.GObject.__init__(self)
        self.__setup_player()

    def stop(self):
        self.__playing = False
        self.__player.set_state(gst.STATE_NULL)
        self.emit(self.STATE_CHANGED, self.STOPPED)

    def play(self, filename, volume=1.5, loop=False, task_id=""):
        filename_is_uri = filename.find("file://") >= 0

        self.__loop = loop

        if not filename_is_uri:
            filename = "file://" + filename

        self.__player.set_property("uri", filename)
        self.set_volume(volume)
        self.__player.set_state(gst.STATE_PLAYING)
        self.__playing = True
        self.__task_id = task_id

        self.emit(self.STATE_CHANGED, self.PLAYING)
        return True

    def set_volume(self, value):
        self.__player.set_property("volume", value)

    def on_message(self, player_bus, message):
        t = message.type
        if t == gst.MESSAGE_EOS:
            if self.__loop:
                logger.debug("Looping Audio")
                self.__player.seek_simple(gst.FORMAT_TIME, gst.SEEK_FLAG_FLUSH, 0)
            else:
                self.__player.set_state(gst.STATE_NULL)
                self.__playing = False
                self.emit(self.STATE_CHANGED, self.STOPPED)

    def is_playing(self):
        return self.__playing

    def get_task_id(self):
        return self.__task_id

GObject.type_register(AudioPlayer)
