# -*- Mode: Python; coding: utf-8; indent-tabs-mode: nil; tab-width: 4 -*-
### BEGIN LICENSE
# Copyright (C) 2012 John Vrbanac <john.vrbanac (at) linux.com>
# This program is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License version 3, as published 
# by the Free Software Foundation.
# 
# This program is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranties of 
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
# PURPOSE.  See the GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License along 
# with this program.  If not, see <http://www.gnu.org/licenses/>.
### END LICENSE

from gi.repository import Notify
from gi.repository import GObject
from cuckoo.CuckooAlarmData import CuckooAlarmData
from cuckoo.CuckooAlarmList import CuckooAlarmList
from cuckoo.AudioPlayer import AudioPlayer
from cuckoo_lib.helpers import get_media_file
import time
import logging
logger = logging.getLogger('cuckoo')


class AlarmController():
    Alarmlist = CuckooAlarmList()

    notification = None
    temp = False
    players = []
    __parent_window = None

    def __init__(self, parent_window):
        self.__parent_window = parent_window

        Notify.init("cuckoo")
        GObject.timeout_add(1000, self.check_alarms)

    def toggle_all(self, toggle):
        logger.debug("Toggling all alarms:" + str(toggle))
        for alarm in AlarmController.Alarmlist.get_alarm_data_array():
            self.stop_alarm(alarm)

        AlarmController.Alarmlist.toggle_all(toggle)
        # Force save for reload of ui
        AlarmController.Alarmlist.convert_and_save(AlarmController.Alarmlist.get_alarm_data_array())

    def stop_alarm(self, alarm):
        logger.debug("Stopping Alarm")
        alarm.set_playing(False)

        for player in self.players:
            if player.get_task_id() == alarm._id:
                player.stop()
                self.players.remove(player)

        self.update_launcher()

    def play_alarm(self, alarm):
        logger.debug("Playing Alarm")
        alarm.set_playing(True)
        self.send_notification(alarm)

        player = AudioPlayer()
        player.play(alarm.get_audio_filename(), alarm.get_volume(), True, alarm._id)
        self.players.append(player)

        self.update_launcher()

    def update_launcher(self):
        # Enclosing in try/except if launchers aren't supported
        try:
            num_active = len(self.players)
            if num_active > 0:
                self.__parent_window.launcher.is_urgent(True)
            else:
                self.__parent_window.launcher.is_urgent(False)

            self.__parent_window.launcher.set_count(num_active)
        except:
            pass

    def current_formatted_time(self):
        return time.strftime(CuckooAlarmData.TIME_FORMAT).lstrip('0')

    def send_notification(self, alarm):
        if self.notification == None:
            self.notification = Notify.Notification.new("Cuckoo", "", get_media_file("cuckoo.svg"))
            self.notification.set_hint_string("append", "allowed")
        self.notification.update("Cuckoo", "Alarm: " + alarm.get_time(), get_media_file("cuckoo.svg"))
        self.notification.show()

    def check_alarms(self):
        for alarm in AlarmController.Alarmlist.get_alarm_data_array():
            if alarm.get_activated() and not alarm.get_playing():
                if alarm.get_time() == self.current_formatted_time():
                    self.play_alarm(alarm)

            elif not alarm.get_activated() and alarm.get_playing():
                self.stop_alarm(alarm)
            elif alarm.get_playing():
                self.update_launcher()
        return True
