# -*- Mode: Python; coding: utf-8; indent-tabs-mode: nil; tab-width: 4 -*-
### BEGIN LICENSE
# Copyright (C) 2012 Mike Kroger <mikek01@telkomsa.net>
# This program is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License version 3, as published 
# by the Free Software Foundation.
# 
# This program is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranties of 
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
# PURPOSE.  See the GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License along 
# with this program.  If not, see <http://www.gnu.org/licenses/>.
### END LICENSE

import gettext
from gettext import gettext as _
gettext.textdomain('wizzrss')

import gtk
import glib
import pango
import webkit
import jswebkit
import feedreader
import sqlite3
import os
import time
import Queue
import prefs
from BeautifulSoup import BeautifulSoup
from urlparse import urlparse

from wizzrss_lib import Window
from wizzrss.AboutWizzrssDialog import AboutWizzrssDialog
from wizzrss.PreferencesWizzrssDialog import PreferencesWizzrssDialog
from wizzrss_lib.helpers import get_media_file

import ctypes
libsoup = ctypes.CDLL('/usr/lib/i386-linux-gnu/libsoup-2.4.so.1')
libwebkit = ctypes.CDLL('/usr/lib/libwebkitgtk-1.0.so.0')

# See wizzrss_lib.Window.py for more details about how this class works
class WizzrssWindow(Window):
    __gtype_name__ = "WizzrssWindow"
    
    def finish_initializing(self, builder): # pylint: disable=E1002
        """Set up the main window"""
        super(WizzrssWindow, self).finish_initializing(builder)

        self.AboutDialog = AboutWizzrssDialog
        self.PreferencesDialog = PreferencesWizzrssDialog

        # Code for other initialization actions should be added here.

        self.websettings = webkit.WebSettings()
        user_agent = self.websettings.get_property("user_agent")
        user_agent = "Wizz RSS (Ubuntu)/12.07.11" + user_agent
        self.websettings.set_property("user_agent", user_agent)

        self.home_path = os.getenv("HOME")
        self.db_path = "/wizzrss/wizzrss.sqlite"

        if not os.path.exists(self.home_path + "/wizzrss"):
            os.makedirs(self.home_path + "/wizzrss")

        session = libwebkit.webkit_get_default_session()

        cookiejar = libsoup.soup_cookie_jar_text_new(self.home_path + "/wizzrss/cookies.txt", False)
        libsoup.soup_session_add_feature(session, cookiejar)

        # setup sqlite stuff
        if os.path.isfile(self.home_path + self.db_path) == False:
            self.conn = sqlite3.connect(self.home_path + self.db_path)

            # history
            self.conn.execute("CREATE TABLE history (histid INTEGER PRIMARY KEY AUTOINCREMENT, uri TEXT, hitcount INTEGER, lasthitdate INTEGER)")
            self.conn.execute("CREATE INDEX histuriindex ON history(uri)")
            self.conn.execute("CREATE INDEX histdateindex ON history(lasthitdate)")

            # feed items
            self.conn.execute("CREATE TABLE items (itemid INTEGER PRIMARY KEY AUTOINCREMENT, feedid INTEGER, itemtitle TEXT, itemsummary TEXT, itemdate TEXT, itemlink TEXT, enclink TEXT, itemtype TEXT, readdate INTEGER)")
            self.conn.execute("CREATE INDEX itemuriindex ON items(itemlink)")
            self.conn.execute("CREATE INDEX itemidindex ON items(feedid)")

            # feeds
            self.conn.execute("CREATE TABLE feeds (feedid INTEGER PRIMARY KEY AUTOINCREMENT, treeorder INTEGER, name TEXT, uri TEXT, treepath TEXT, type TEXT, hitcount INTEGER, lasthitdate INTEGER, wlstatus INTEGER, expanded INTEGER)")
            self.conn.execute("INSERT INTO feeds (treeorder, name, uri, treepath, type, hitcount, wlstatus, expanded) VALUES (1, 'Wizz RSS Feeds', '', Null, 'folder', 0, 0, 1)")
            self.conn.execute("INSERT INTO feeds (treeorder, name, uri, treepath, type, hitcount, wlstatus, expanded) VALUES (2, 'Wizz RSS Blog', 'http://wizzrss.blat.co.za/feed/', '0', 'feed', 0, 0, 0)")
            self.conn.execute("INSERT INTO feeds (treeorder, name, uri, treepath, type, hitcount, wlstatus, expanded) VALUES (3, 'Ubuntu Canonical', 'http://www.canonical.com/rss.xml', '0', 'feed', 0, 0, 0)")
            self.conn.execute("INSERT INTO feeds (treeorder, name, uri, treepath, type, hitcount, wlstatus, expanded) VALUES (4, 'Example Sub-Folder', '', '0', 'folder', 0, 0, 1)")
            self.conn.execute("INSERT INTO feeds (treeorder, name, uri, treepath, type, hitcount, wlstatus, expanded) VALUES (5, 'BBC: Science', 'http://feeds.bbci.co.uk/news/science_and_environment/rss.xml?edition=int', '0:2', 'feed', 0, 0, 0)")
            self.conn.execute("INSERT INTO feeds (treeorder, name, uri, treepath, type, hitcount, wlstatus, expanded) VALUES (6, 'BBC: Technology', 'http://feeds.bbci.co.uk/news/technology/rss.xml?edition=int', '0:2', 'feed', 0, 0, 0)")
            self.conn.execute("INSERT INTO feeds (treeorder, name, uri, treepath, type, hitcount, wlstatus, expanded) VALUES (7, 'Times LIVE Breaking News', 'http://www.timeslive.co.za/?service=rss', '0:2', 'feed', 0, 0, 0)")

            # prefs
            self.conn.execute("CREATE TABLE prefs (prefname TEXT PRIMARY KEY, prefvalue TEXT)")
        else:
            self.conn = sqlite3.connect(self.home_path + self.db_path)

        self.prefs = prefs.prefs(self.db_path)
        self.prefs.connect("pref_changed", self.pref_change_handler)
        self.prefs.create_pref("main_window_width", "800")
        self.prefs.create_pref("main_window_height", "600")
        self.prefs.create_pref("main_split_pos", "250")
        self.prefs.create_pref("tree_split_pos", "150")
        self.prefs.create_pref("item_split_pos", "150")
        self.prefs.create_pref("found_split_pos", "250")
        self.prefs.create_pref("current_page", "0")
        self.prefs.create_pref("last_feed_id", "0")
        self.prefs.create_pref("cache_days", "7")
        self.prefs.create_pref("show_read", "1")
        self.prefs.create_pref("enable_tree_lines", "1")
        self.prefs.create_pref("feed_tree_hint", "0")
        self.prefs.create_pref("item_list_hint", "0")
        self.prefs.create_pref("feed_tree_font_size", "10")
        self.prefs.create_pref("item_list_font_size", "10")
        self.prefs.create_pref("show_summary_pane", "1")
        self.prefs.create_pref("mark_enc_and_item", "0")
        self.prefs.flush()

        # some global stuff
        self.show_read = int(self.prefs.get_pref("show_read"))
        self.feed_tree_font_size = self.prefs.get_pref("feed_tree_font_size")
        self.item_list_font_size = self.prefs.get_pref("item_list_font_size")
        self.last_feed_id = self.prefs.get_pref("last_feed_id")
        self.mark_enc_and_item = int(self.prefs.get_pref("mark_enc_and_item"))
        self.cursor = self.conn.cursor()
        self.current_row = -1
        self.current_tree_iter = None
        self.tree_order = 0
        self.browsertabs = self.builder.get_object("browsertabs")
        self.drag_data = ""

        # clear old stuff out of history
        self.cache_days = int(self.prefs.get_pref("cache_days"))
        now = int(time.time())
        now = now - (86400 * self.cache_days)
        self.conn.execute("DELETE FROM history WHERE lasthitdate < ?", [str(now)])

        # set window size
        width = self.prefs.get_pref("main_window_width")
        height = self.prefs.get_pref("main_window_height")
        self.set_default_size(int(width), int(height))
        self.set_position(gtk.WIN_POS_CENTER)

        # set main splitter position
        pos = self.prefs.get_pref("main_split_pos")
        self.builder.get_object("main_splitter").set_position(int(pos))

        # set feed tree splitter position
        pos = self.prefs.get_pref("tree_split_pos")
        self.builder.get_object("tree_splitter").set_position(int(pos))

        # set feed item splitter position
        pos = self.prefs.get_pref("item_split_pos")
        self.builder.get_object("item_splitter").set_position(int(pos))

        # set found item splitter position
        pos = self.prefs.get_pref("found_split_pos")
        self.builder.get_object("found_splitter").set_position(int(pos))

        # set the current tab
        page = self.prefs.get_pref("current_page")
        self.builder.get_object("tabs").set_current_page(int(page))

        # message and confirm dialog
        self.message_dialog = self.builder.get_object("message_dialog")
        self.confirm_dialog = self.builder.get_object("confirm_dialog")
        self.no_button = self.builder.get_object("no_button")

        # add folder dialog stuff
        self.add_folder_dialog = self.builder.get_object("add_folder_dialog")
        self.add_folder_title = self.builder.get_object("add_folder_title")

        # add feed dialog stuff
        self.add_feed_dialog = self.builder.get_object("add_feed_dialog")
        self.add_feed_title = self.builder.get_object("add_feed_title")
        self.add_feed_uri = self.builder.get_object("add_feed_uri")

        # feed properties dialog stuff
        self.feed_properties_dialog = self.builder.get_object("feed_properties_dialog")
        self.feed_title_prop = self.builder.get_object("feed_title_prop")
        self.feed_url_prop = self.builder.get_object("feed_url_prop")
        self.feed_hitcount_prop = self.builder.get_object("feed_hitcount_prop")
        self.feed_lastread_prop = self.builder.get_object("feed_lastread_prop")
        self.feed_watched_prop = self.builder.get_object("feed_watched_prop")

        # folder properties dialog stuff
        self.folder_properties_dialog = self.builder.get_object("folder_properties_dialog")
        self.folder_title_prop = self.builder.get_object("folder_title_prop")

        # found feeds dialog stuff
        self.found_feeds_dialog = self.builder.get_object("found_feeds_dialog")
        self.found_dialog_msg = self.builder.get_object("found_dialog_msg")
        found_scrolled = self.builder.get_object("found_scrolled")
        self.found_list = webkit.WebView()
        self.found_list.connect("navigation-policy-decision-requested", self.nav_request_policy_decision)
        self.found_list.set_settings(self.websettings)
        found_scrolled.add(self.found_list)
        self.found_list.show()

        # feed tree context menu stuff
        self.feed_tree_popup = self.builder.get_object("feed_tree_popup")
        self.add_folder_mi = self.builder.get_object("add_folder_mi")
        self.add_feed_mi = self.builder.get_object("add_feed_mi")
        self.properties_mi = self.builder.get_object("properties_mi")
        self.delete_mi = self.builder.get_object("delete_mi")

        # item list context menu stuff
        self.item_list_popup = self.builder.get_object("item_list_popup")
        self.open_enc_link_mi = self.builder.get_object("open_enc_link_mi")
        self.mark_read_mi = self.builder.get_object("mark_read_mi")
        self.mark_unread_mi = self.builder.get_object("mark_unread_mi")
        self.mark_enc_read_mi = self.builder.get_object("mark_enc_read_mi")
        self.mark_enc_unread_mi = self.builder.get_object("mark_enc_unread_mi")


        self.show_read_button = self.builder.get_object("show_read_button")
        self.hide_read_button = self.builder.get_object("hide_read_button")
        if self.show_read == True:
            self.show_read_button.set_active(True)
        else:
            self.hide_read_button.set_active(True)
        self.show_read_id = self.show_read_button.connect("clicked", self.show_read_clicked)
        self.hide_read_id = self.hide_read_button.connect("clicked", self.hide_read_clicked)

        # pixbufs used by the feed tree and the item list
        self.pixbuf_none = gtk.gdk.pixbuf_new_from_file(get_media_file("None.gif", True))
        self.pixbuf_read = gtk.gdk.pixbuf_new_from_file(get_media_file("Read.gif", True))
        self.pixbuf_unread = gtk.gdk.pixbuf_new_from_file(get_media_file("Unread.gif", True))
        self.pixbuf_tick = gtk.gdk.pixbuf_new_from_file(get_media_file("tick.png", True))
        self.pixbuf_partread = gtk.gdk.pixbuf_new_from_file(get_media_file("PartRead.gif", True))
        self.pixbuf_folder = gtk.gdk.pixbuf_new_from_file(get_media_file("Folder.png", True))
        self.pixbuf_feed = gtk.gdk.pixbuf_new_from_file(get_media_file("Feed.png", True))

        # images to be used by the throbber
        self.throbber = gtk.Image()
        self.throbber.set_from_file(get_media_file("throbber.gif", True))
        self.throbber.show()

        self.throbber_bg = gtk.Image()
        self.throbber_bg.set_from_file(get_media_file("busybg.png", True))
        self.throbber_bg.show()

        # set images for toolbar buttons on the wizz rss tab
        self.set_button_image("find.png", "find_feeds_button")
        self.set_button_image("help.png", "help_button")
        self.set_button_image("options.png", "preferences_button")
        self.set_button_image("weather.png", "weather_button")
        self.set_button_image("show.png", "show_read_button")
        self.set_button_image("hide.png", "hide_read_button")

        # set image for the throbber button
        self.throbber_button = self.builder.get_object("throbber_button")
        self.throbber_button.set_icon_widget(self.throbber_bg)

        # setup the item list ---------------------------------------------------------------------------------------------
        self.item_list = self.builder.get_object("item_list")
        self.item_list_store = gtk.ListStore(str, str, str, str, int, str, int, str, str, gtk.gdk.Pixbuf, gtk.gdk.Pixbuf)

        # 0 = Item title
        # 1 = Item summary
        # 2 = Item date
        # 3 = Item URI
        # 4 = Item URI read - true or false
        # 5 = Enclosure URI
        # 6 = Enclosure URI read - true or false
        # 7 = Type - title or item
        # 8 = Cell font
        # 9 = Pixbuf for the read/unread/partread image

        self.ils_item_title = 0
        self.ils_item_summary = 1
        self.ils_item_date = 2
        self.ils_item_uri = 3
        self.ils_item_read = 4
        self.ils_enc_uri = 5
        self.ils_enc_read = 6
        self.ils_item_type = 7
        self.ils_cell_font = 8
        self.ils_pixbuf = 9
        self.ils_pixbuf_tick = 10

        self.item_list.set_model(self.item_list_store)
        self.item_list.set_search_equal_func(self.search_func)
        self.item_list.set_rules_hint(int(self.prefs.get_pref("item_list_hint")))
        self.item_list.enable_model_drag_dest([('text/html', 0, 0)], gtk.gdk.ACTION_DEFAULT | gtk.gdk.ACTION_MOVE)

        # columns for the item list
        picrenderer = gtk.CellRendererPixbuf()
        txtrenderer = gtk.CellRendererText()
        txtrenderer.set_property("ellipsize", pango.ELLIPSIZE_END)
        column0 = gtk.TreeViewColumn("", picrenderer, pixbuf = self.ils_pixbuf)
	    column1 = gtk.TreeViewColumn("", txtrenderer, text = self.ils_item_title, font = self.ils_cell_font)
        column1.set_expand(True)
        column2 = gtk.TreeViewColumn("", picrenderer, pixbuf = self.ils_pixbuf_tick)
        self.item_list.append_column(column0)
        self.item_list.append_column(column1)
        self.item_list.append_column(column2)
        self.item_list.columns_autosize()

        # event handlers for the item list
        self.item_list.connect("button-press-event", self.item_clicked)
        self.item_list.connect("motion-notify-event", self.item_hover)
        self.item_list.connect("key-release-event", self.item_key_released)

        # setup the feed tree ---------------------------------------------------------------------------------------------
        self.feed_tree = self.builder.get_object("feed_tree")
        self.feed_tree_store = gtk.TreeStore(str, str, str, str, str, int, str, int, gtk.gdk.Pixbuf, gtk.gdk.Pixbuf)

        # 0 = Feed title
        # 1 = Feed URI
        # 2 = Cell font
        # 3 = Database ID
        # 4 = Type - folder or feed
        # 5 = Hit count
        # 6 = Last hit date
        # 7 = Watch list status
        # 8 = Pixbuf for the watched image
        # 9 = Pixbuf for folder or feed image

        self.fts_feed_title = 0
        self.fts_feed_uri = 1
        self.fts_cell_font = 2
        self.fts_feed_id = 3
        self.fts_type = 4
        self.fts_hitcount = 5
        self.fts_lasthitdate = 6
        self.fts_wl_status = 7
        self.fts_pixbuf = 8
        self.fts_ff_pixbuf = 9

        self.feed_tree.set_model(self.feed_tree_store)
        self.pop_tree()
        self.feed_tree.set_search_equal_func(self.search_func)
        self.feed_tree.set_enable_tree_lines(int(self.prefs.get_pref("enable_tree_lines")))
        self.feed_tree.set_rules_hint(int(self.prefs.get_pref("feed_tree_hint")))
        self.feed_tree.enable_model_drag_dest([("GTK_TREE_MODEL_ROW", gtk.TARGET_SAME_WIDGET, 0), ('text/html', 0, 1)], gtk.gdk.ACTION_DEFAULT | gtk.gdk.ACTION_MOVE)

        # columns for the feed tree
        celltext = gtk.CellRendererText()
        celltext.set_property("ellipsize", pango.ELLIPSIZE_END)
        cellimage = gtk.CellRendererPixbuf()
        column = gtk.TreeViewColumn("", cellimage, pixbuf = self.fts_ff_pixbuf)
        column0 = gtk.TreeViewColumn("", celltext, text = self.fts_feed_title, font = self.fts_cell_font)
        column0.set_expand(True)
        column1 = gtk.TreeViewColumn("", cellimage, pixbuf = self.fts_pixbuf)
        self.feed_tree.append_column(column)
        self.feed_tree.append_column(column0)
        self.feed_tree.append_column(column1)
        self.feed_tree.columns_autosize()

        # event handlers for the feed tree
        self.feed_tree.connect("button-release-event", self.feed_clicked)

        # browser for displaying various information at the bottom of the Wizz RSS tab
        self.scrolled_msg = self.builder.get_object("scrolled_msg")
        self.scrolled_msg.set_visible(int(self.prefs.get_pref("show_summary_pane")))
        self.msg = webkit.WebView()
        self.msg.connect("navigation-policy-decision-requested", self.nav_request_policy_decision)
        self.msg.set_settings(self.websettings)
        self.scrolled_msg.add(self.msg)
        self.msg.show()

        # browser for displaying various information at the bottom of the Found Items tab
        scrolled_found_msg = self.builder.get_object("scrolled_found_msg")
        self.found_msg = webkit.WebView()
        self.found_msg.set_settings(self.websettings)
        scrolled_found_msg.add(self.found_msg)
        self.found_msg.show()

        notebookPage(self.browsertabs, "http://wizzrss.com/ubuntu.php", self)

        self.load_items()

    # handle pref changes
    def pref_change_handler(self, prefs, pref_name, pref_value):
        if pref_name == "enable_tree_lines":
            self.feed_tree.set_enable_tree_lines(pref_value)
        elif pref_name == "feed_tree_hint":
            self.feed_tree.set_rules_hint(pref_value)
        elif pref_name == "item_list_hint":
            self.item_list.set_rules_hint(pref_value)
        elif pref_name == "feed_tree_font_size":
            self.feed_tree_font_size = str(pref_value)
            self.feed_tree_change_font_size(self.feed_tree_store.get_iter_root())
        elif pref_name == "item_list_font_size":
            self.item_list_font_size = str(pref_value)
            self.item_list_change_font_size()
        elif pref_name == "show_summary_pane":
            self.scrolled_msg.set_visible(pref_value)
        elif pref_name == "mark_enc_and_item":
            self.mark_enc_and_item = pref_value
        elif pref_name == "cache_days":
            self.cache_days = pref_value
        else:
            print pref_name
            print pref_value

    # set image for various toolbar buttons
    def set_button_image(self, image_name, button_name):
        image = gtk.Image()
        image.set_from_file(get_media_file(image_name, True))
        image.show()
        button = self.builder.get_object(button_name)
        button.set_icon_widget(image)

    # add a folder or a feed to the feeds table and get the id
    def add_folder_feed(self, title):
        self.cursor.execute("INSERT INTO feeds (name) VALUES (?)", [title])
        self.conn.commit()
        return self.cursor.lastrowid

    # populate the feed tree store from the database
    def pop_tree(self):
        paths = []
        for row in self.cursor.execute("SELECT * FROM feeds ORDER BY treeorder"):
            if row[4]:
                iter = self.feed_tree_store.get_iter_from_string(row[4])
            else:
                iter = None
            if row[5] == "feed" and row[6] == 0:
                font = "Bold " + self.feed_tree_font_size
            else:
                font = "Normal " + self.feed_tree_font_size
            if row[8] == 0:
                pixbuf = self.pixbuf_none
            else:
                pixbuf = self.pixbuf_tick
            if row[5] == "feed":
                ff_pixbuf = self.pixbuf_feed
            else:
                ff_pixbuf = self.pixbuf_folder
            it = self.feed_tree_store.append(iter, [row[2], row[3], font, row[0], row[5], row[6], row[7], row[8], pixbuf, ff_pixbuf])
            if row[9] == 1:
                path = self.feed_tree_store.get_path(it)
                paths.append(path)

        for path in paths:
            self.feed_tree.expand_row(path, open_all = False)

    # persist current settings & close db connection
    def destroy_event(self, widget, event = None):
        self.save_tree(self.feed_tree_store.get_iter_root(), 0)
        self.prefs.set_pref("main_window_width", self.allocation.width)
        self.prefs.set_pref("main_window_height", self.allocation.height)
        self.prefs.set_pref("main_split_pos", self.builder.get_object("main_splitter").get_position())
        self.prefs.set_pref("tree_split_pos", self.builder.get_object("tree_splitter").get_position())
        self.prefs.set_pref("item_split_pos", self.builder.get_object("item_splitter").get_position())
        self.prefs.set_pref("found_split_pos", self.builder.get_object("found_splitter").get_position())
        self.prefs.set_pref("current_page", self.builder.get_object("tabs").get_current_page())
        self.prefs.set_pref("show_read", self.show_read)
        self.prefs.set_pref("last_feed_id", self.last_feed_id)
        self.prefs.flush()

        self.cursor.close()
        self.conn.close()

    # parse the feed at feeduri
    def read_feed(self, feeduri, feedid, treerow):

        self.throbber_button.set_icon_widget(self.throbber)
        q = Queue.Queue()
        fr = feedreader.feedreader(feeduri, feedid, treerow, q, self.cache_days, self.db_path)
        fr.start()

        glib.timeout_add(300, self.check_queue, q)

    def search_func(self, model, column, key, iter):
        element = model.get_value(iter, column)
        if key.lower() in element.lower():
            return False
        else:
            return True

    # check to see if the feed parser thread has put anything into the queue
    def check_queue(self = None, queue = None):
        try:
            status = queue.get(False)
            self.throbber_button.set_icon_widget(self.throbber_bg)

            if status[0] == 1: # 1 - the feed was parsed successfully 
                if len(status[3]) > 0:
                    self.feed_tree_store[status[2]][self.fts_feed_uri] = status[3]
                self.last_feed_id = status[1]
                self.load_items()
            elif status[0] == 2:
                self.message_dialog.set_property("text", "Invalid feed XML")
                self.message_dialog.set_property("secondary-text", "The XML of the feed you tried to read is not valid")
                self.message_dialog.show()
            return False
        except Queue.Empty:
            return True

    # iterates recursively through the feed tree model and sets the font size
    def feed_tree_change_font_size(self, iter):
        while iter:
            path = self.feed_tree_store.get_path(iter)
            if self.feed_tree_store[path][self.fts_type] == "feed" and self.feed_tree_store[path][self.fts_hitcount] == 0:
                font = "Bold " + self.feed_tree_font_size
            else:
                font = "Normal " + self.feed_tree_font_size
            self.feed_tree_store[path][self.fts_cell_font] = font

            if self.feed_tree_store.iter_has_child(iter):
                self.feed_tree_change_font_size(self.feed_tree_store.iter_children(iter))

            iter = self.feed_tree_store.iter_next(iter)

    # iterates through the item list model and sets the font size
    def item_list_change_font_size(self):
        for row in self.item_list_store:
            if row[self.ils_item_type] == "title":
                font = "Bold " + self.item_list_font_size
            else:
                if len(row[self.ils_enc_uri]) == 0:
                    font = "Normal " + self.item_list_font_size
                else:
                    font = "Oblique " + self.item_list_font_size
            row[self.ils_cell_font] = font

    # load feed items from the database into the feed item list
    def load_items(self):
        self.current_row = -1
        self.item_list_store.clear()
        for row in self.cursor.execute("SELECT * FROM items WHERE feedid = ?", [self.last_feed_id]):
            if row[7] == "title":
                self.item_list_store.append([row[2], row[3], row[4], row[5], 0, "", 0, "title", "Bold " + self.item_list_font_size, self.pixbuf_none, self.pixbuf_none])
            else:
                read = self.check_history(row[5])
                enc_read = self.check_history(row[6])

                if len(row[6]) == 0:
                    font = "Normal " + self.item_list_font_size
                else:
                    font = "Oblique " + self.item_list_font_size

                if self.must_show(read, enc_read) == True:
                    pixbuf = self.get_indicator(read, enc_read)
                    self.item_list_store.append([row[2], row[3], row[4], row[5], read, row[6], enc_read, "item", font, pixbuf, self.pixbuf_none])

    # should the feed item be shown or not
    def must_show(self, read, enc_read):
        if self.show_read == True:
            return True
        else:
            if (read == 2 and enc_read == 2) or (read == 2 and enc_read == 0) or (read == 0 and enc_read == 2):
                return False
            else:
                return True

    # show read feed items
    def show_read_clicked(self, event):
        self.show_read_button.handler_block(self.show_read_id)
        self.hide_read_button.handler_block(self.hide_read_id)
        self.show_read_button.set_active(True)
        self.hide_read_button.set_active(False)
        self.show_read_button.handler_unblock(self.show_read_id)
        self.hide_read_button.handler_unblock(self.hide_read_id)
        self.show_read = True
        self.load_items()

    # hide read feed items
    def hide_read_clicked(self, event):
        self.show_read_button.handler_block(self.show_read_id)
        self.hide_read_button.handler_block(self.hide_read_id)
        self.hide_read_button.set_active(True)
        self.show_read_button.set_active(False)
        self.show_read_button.handler_unblock(self.show_read_id)
        self.hide_read_button.handler_unblock(self.hide_read_id)
        self.show_read = False
        self.load_items()

    # check if the URI is in history
    # return 1 if URI not found in history
    # return 2 if URI is found in history
	def check_history(self, uri):
		if len(uri) == 0:
			return 0
		else:
            cur = self.conn.cursor()
			cur.execute("SELECT COUNT(uri) FROM history WHERE uri = ?", [uri])
			row = cur.fetchone()
			if row[0] == 0:
				return 1
			else:
				return 2
            cur.close()

    # feed clicked in the feed tree
    def feed_clicked(self, obj, event):
        path = self.feed_tree.get_path_at_pos(int(event.x), int(event.y))
        if path:
            if event.button == 1: # left click
                uri = self.feed_tree_store[path[0]][self.fts_feed_uri]
                id = self.feed_tree_store[path[0]][self.fts_feed_id]
                if len(uri) > 0:
                    self.feed_tree_store[path[0]][self.fts_cell_font] = "Normal " + self.feed_tree_font_size
                    now = int(time.time())
                    self.feed_tree_store[path[0]][self.fts_hitcount] += 1
                    self.feed_tree_store[path[0]][self.fts_lasthitdate] = now
                    self.read_feed(uri, id, path[0])
            elif event.button == 3:
                self.current_tree_iter = self.feed_tree_store.get_iter(path[0])
                if self.feed_tree_store[path[0]][self.fts_type] == "folder":
                    self.add_folder_mi.set_sensitive(True)
                    self.add_feed_mi.set_sensitive(True)
                else:
                    self.add_folder_mi.set_sensitive(False)
                    self.add_feed_mi.set_sensitive(False)
                if self.feed_tree_store[path[0]][self.fts_feed_id] == "1":
                    self.delete_mi.set_sensitive(False)
                    self.properties_mi.set_sensitive(True)
                else:
                    self.delete_mi.set_sensitive(True)
                    self.properties_mi.set_sensitive(True)
                self.feed_tree_popup.popup(None, None, None, event.button, event.time)

    # feed item clicked in the item list
    def item_clicked(self, widget, event):
        path = self.item_list.get_path_at_pos(int(event.x), int(event.y))
        if path:
            row = path[0][0]
            uri = self.item_list_store[row][self.ils_item_uri]
            if event.button == 1: # left click
                if len(uri) > 0:
                    self.update_history(uri, True, row)
                    id = self.browsertabs.get_current_page()
                    nbPage = self.browsertabs.get_nth_page(id)
                    nbPage.goto_page(uri)
            elif event.button == 2: # middle click
                self.open_new_tab(widget, uri)
            elif event.button == 3: # right click
                if self.item_list_store[row][self.ils_item_type] == "title":
                    self.mark_read_mi.set_sensitive(False)
                    self.mark_unread_mi.set_sensitive(False)
                else:
                    self.mark_read_mi.set_sensitive(True)
                    self.mark_unread_mi.set_sensitive(True)

                if len(self.item_list_store[row][self.ils_enc_uri]) > 0:
                    self.open_enc_link_mi.set_sensitive(True)
                    self.mark_enc_read_mi.set_sensitive(True)
                    self.mark_enc_unread_mi.set_sensitive(True)
                else:
                    self.open_enc_link_mi.set_sensitive(False)
                    self.mark_enc_read_mi.set_sensitive(False)
                    self.mark_enc_unread_mi.set_sensitive(False)
                self.item_list_popup.popup(None, None, None, event.button, event.time)

    # open the selected feed enclosure in the current tab
    def open_enclosure_link(self, widget):
        uri = self.item_list_store[self.current_row][self.ils_enc_uri]
        if len(uri) > 0:
            self.update_history(uri, True, self.current_row)
            id = self.browsertabs.get_current_page()
            nbPage = self.browsertabs.get_nth_page(id)
            nbPage.goto_page(uri)

    # open the selected feed item in a new tab
    # only allows a maximum of 10 tabs to be opened because if we open too many tabs, things go stupid
    def open_new_tab(self, widget, uri = None):
        if self.browsertabs.get_n_pages() < 10:
            if (not uri):
                uri = self.item_list_store[self.current_row][self.ils_item_uri]
            if len(uri) > 0:
                self.update_history(uri, True, self.current_row)
                notebookPage(self.browsertabs, uri, self)
        else:
            self.message_dialog.set_property("text", "The maximum number of tabs is 10.")
            self.message_dialog.set_property("secondary-text", "The maximum number of tabs is already open.\rPlease close at least one existing tab before\rtrying to open another one.")
            self.message_dialog.show()

    # open unread feed items in new tabs
    # only allows a maximum of 10 tabs to be opened because if we open too many tabs, things go stupid
    def open_new_tabs(self, widget):
        for row in self.item_list_store:
            if row[self.ils_item_type] == "item" and row[self.ils_item_read] == 1 and self.browsertabs.get_n_pages() < 10:
                uri = row[self.ils_item_uri]
                if len(uri) > 0:
                    self.update_history(uri, False, row.path[0])
                    notebookPage(self.browsertabs, uri, self)
        self.conn.commit()
        if self.browsertabs.get_n_pages() == 10:
            self.message_dialog.set_property("text", "The maximum number of tabs is 10.")
            self.message_dialog.set_property("secondary-text", "The maximum number of tabs was opened.\rSome unread items may not have been opened\ron a tab.")
            self.message_dialog.show()

    # preview feeds items selected from the item list context menu
    def preview_feed(self, widget):
        id = self.browsertabs.get_current_page()
        nbPage = self.browsertabs.get_nth_page(id)
        nbPage.goto_page("about:feed")

    # generate the feed preview html and display it in the current tab
    def preview_details(self = None, widget = None):

        html = "<!DOCTYPE HTML PUBLIC '-//W3C//DTD HTML 4.01//EN' 'http://www.w3.org/TR/html4/strict.dtd'>"
        html += "<html>"
        html += "<head>"
        html += "<meta http-equiv='Content-Type' content='text/html;charset=utf-8'>"
        html += "<style type='text/css'>"
        html += "body {background: #f0f0f0;}"
        html += ".feedtitle {border: 1px solid darkred; background: #fafafa; padding: 1em 1em 1em 1em; margin: 1em 2em 1em 2em;}"
        html += ".feedtitletext {font-family: arial, sans-serif; font-size: 1.5em; font-weight: bold; color: darkred; padding-bottom: .5em;}"
        html += ".feedsubtitletext {font-family: arial, sans-serif; font-size: 1em; font-weight: bold; color: black; padding-bottom: 15px;}"
        html += ".itemdetail {border: 1px solid #d0d0d0; padding: 8px 10px 7px 20px; margin: 2em 2em 0em 2em;}"
        html += ".itemdetail[read='false'] {background: #fafafa;}"
        html += ".itemdetail[read='false'] .iteminfo {color: black;}"
        html += ".itemdetail[read='true'] {	background: #f0f0f0;}"
        html += ".itemdetail[read='true'] .iteminfo {color: gray;}"
        html += ".itemtitlespan {font-family: arial, sans-serif; font-size: 1.1em; font-weight: bold; color: darkred; border: #d0d0d0 1px solid; background: #fafafa; padding: 1px 15px; white-space: nowrap; position: relative; top: -1.1em; max-width: 400px; overflow: hidden; text-overflow: ellipsis;}"
        html += ".iteminfo {font-family: arial, sans-serif; font-size: 1em; padding-bottom: 0.5em;}"
        html += ".itemdate {font-family: arial, sans-serif; font-size: 0.8em; font-style: italic; padding-top: 1em; padding-bottom: 0em;}"
        html += "</style>"
        html += "<title>Wizz RSS Feed Preview</title>"
        html += "</head>"
        html += "<body><div id='divid'>"

        for row in self.item_list_store:
            if row[self.ils_item_type] == "title":
                html += "<div class='feedtitle'>"
                html += "<div class='feedtitletext'>" + row[self.ils_item_title] + "</div>"
                html += "<div class='feedsubtitletext'>" + row[self.ils_item_summary] + "</div>"
                if len(row[self.ils_item_uri]) > 0:
                    html += "<input type='button' value='Home' onclick='location.href=\"" + row[self.ils_item_uri] + "\"'/>"
                html += "</div>"
            else:
                if row[self.ils_item_read] == 2:
                    html += "<div class='itemdetail' read='true'>"
                else:
                    html += "<div class='itemdetail' read='false'>"
                html += "<div class='itemtitlediv'><div class='itemtitlespan'>" + row[self.ils_item_title] + "</div></div>"
                html += "<div class='iteminfo'><div class='itemdesc'>" + row[self.ils_item_summary] + "</div>"
                html += "<div class='itemdate'>" + row[self.ils_item_date] + "</div></div>"
                if len(row[self.ils_item_uri]) > 0:
                    html += "<input type='button' value='Read more...' onclick='location.href=\"" + row[self.ils_item_uri] + "\"'/>"
                if len(row[self.ils_enc_uri]) > 0:
                    html += " <input type='button' value='Enclosure' onclick='location.href=\"" + row[self.ils_enc_uri] + "\"'/>"
                html += "</div>"

        html += "</div></body>"

        id = self.browsertabs.get_current_page()
        nbPage = self.browsertabs.get_nth_page(id)
        nbPage.show_html(html)

    # mark all feed items in the item list as read
    def mark_all_read(self, widget):
        for row in self.item_list_store:
            uri, item_read, enc_uri, enc_read, item_type = self.get_item_info(row.path[0])
            if item_type == "item":
                if item_read == 1:
                    self.update_history(uri, False, row.path[0])
                if self.mark_enc_and_item == True and enc_read == 1:
                    self.update_history(enc_uri, False, row.path[0])
        self.conn.commit()

    # mark all feed items in the item list as unread
    def mark_all_unread(self, widget):
        for row in self.item_list_store:
            uri, item_read, enc_uri, enc_read, item_type = self.get_item_info(row.path[0])
            if item_type == "item":
                if item_read == 2:
                    self.remove_history(uri, row.path[0])
                if self.mark_enc_and_item == True and enc_read == 2:
                    self.remove_history(enc_uri, row.path[0])
        self.conn.commit()

    # mark the selected feed item in the item list as read
    def mark_read(self, widget):
        uri, item_read, enc_uri, enc_read, item_type = self.get_item_info(self.current_row)
        if item_read == 1:
            self.update_history(uri, True, self.current_row)
        if self.mark_enc_and_item == True and enc_read == 1:
            self.update_history(enc_uri, True, self.current_row)

    # mark the selected feed item in the item list as unread
    def mark_unread(self, widget):
        uri, item_read, enc_uri, enc_read, item_type = self.get_item_info(self.current_row)
        if item_read == 2:
            self.remove_history(uri, self.current_row)
        if self.mark_enc_and_item == True and enc_read == 2:
            self.remove_history(enc_uri, self.current_row)
        self.conn.commit()

    # mark all feed enclosures in the item list as read
    def mark_all_enc_read(self, widget):
        for row in self.item_list_store:
            uri, item_read, enc_uri, enc_read, item_type = self.get_item_info(row.path[0])
            if item_type == "item" and enc_read == 1:
                self.update_history(enc_uri, False, row.path[0])
        self.conn.commit()

    # mark all feed enclosures in the item list as unread
    def mark_all_enc_unread(self, widget):
        for row in self.item_list_store:
            uri, item_read, enc_uri, enc_read, item_type = self.get_item_info(row.path[0])
            if item_type == "item" and enc_read == 2:
                    self.remove_history(enc_uri, row.path[0])
        self.conn.commit()

    # mark the selected feed enclosure in the item list as read
    def mark_enc_read(self, widget):
        uri, item_read, enc_uri, enc_read, item_type = self.get_item_info(self.current_row)
        if enc_read == 1:
            self.update_history(enc_uri, True, self.current_row)

    # mark the selected feed enclosure in the item list as unread
    def mark_enc_unread(self, widget):
        uri, item_read, enc_uri, enc_read, item_type = self.get_item_info(self.current_row)
        if enc_read == 2:
            self.remove_history(enc_uri, self.current_row)
            self.conn.commit()

    def get_item_info(self, row):
        uri = self.item_list_store[row][self.ils_item_uri]
        item_read = self.item_list_store[row][self.ils_item_read]
        enc_uri = self.item_list_store[row][self.ils_enc_uri]
        enc_read = self.item_list_store[row][self.ils_enc_read]
        item_type = self.item_list_store[row][self.ils_item_type]
        return [uri, item_read, enc_uri, enc_read, item_type]

    # add folder dialog stuff -------------------------------------------------------------------------
    def add_folder_show(self, widget):
        self.add_folder_title.grab_focus()
        self.add_folder_dialog.show()

    def add_folder_delete(self, widget, event):
        self.add_folder_dialog.hide()
        return True;

    def add_folder_response(self, widget, response):
        if response == 2:
            self.add_folder_dialog.hide()
        elif response == 1:
            title = self.add_folder_title.get_text()
            self.add_folder_title.set_text("")
            if len(title) > 0:
                self.add_folder_dialog.hide()
                recid = self.add_folder_feed(title)
                self.feed_tree_store.append(self.current_tree_iter, [title, "", "Normal " + self.feed_tree_font_size, recid, "folder", 0, None, 0, self.pixbuf_none, self.pixbuf_folder])


    # add feed dialog stuff -----------------------------------------------------------------------------
    def add_feed_show(self, widget):
        self.add_feed_title.grab_focus()
        self.add_feed_dialog.show()

    def add_feed_delete(self, widget, event):
        self.add_feed_dialog.hide()
        return True;

    def add_feed_response(self, widget, response):
        if response == 2:
            self.add_feed_dialog.hide()
        elif response == 1:
            title = self.add_feed_title.get_text()
            uri = self.add_feed_uri.get_text()
            if len(title) > 0 and len(uri) > 0:
                self.add_feed_title.set_text("")
                self.add_feed_uri.set_text("")
                self.add_feed_dialog.hide()
                recid = self.add_folder_feed(title)
                self.feed_tree_store.append(self.current_tree_iter, [title, uri, "Bold " + self.feed_tree_font_size, recid, "feed", 0, None, 0, self.pixbuf_none, self.pixbuf_feed])

    # properties clicked on the feed tree context menu -------------------------------------------------------
    def properties_show(self, widget):
        selection = self.feed_tree.get_selection()
        model, iter = selection.get_selected()
        type = self.feed_tree_store.get_value(iter, self.fts_type)
        if type == "folder":
            self.folder_title_prop.grab_focus()
            self.folder_title_prop.set_text(self.feed_tree_store.get_value(iter, self.fts_feed_title))
            self.folder_properties_dialog.show()
        else:
            self.feed_title_prop.grab_focus()
            self.feed_title_prop.set_text(self.feed_tree_store.get_value(iter, self.fts_feed_title))
            self.feed_url_prop.set_text(self.feed_tree_store.get_value(iter, self.fts_feed_uri))
            self.feed_hitcount_prop.set_text(str(self.feed_tree_store.get_value(iter, self.fts_hitcount)))
            if self.feed_tree_store.get_value(iter, self.fts_lasthitdate):
                date = time.asctime(time.localtime(int(self.feed_tree_store.get_value(iter, self.fts_lasthitdate))))
            else:
                date = "Never"
            self.feed_lastread_prop.set_text(date)
            if self.feed_tree_store.get_value(iter, self.fts_wl_status) == 0:
                wl_status = "No"
            self.feed_watched_prop.set_text(wl_status)
            self.feed_properties_dialog.show()

    # folder properties dialog stuff ------------------------------------------------------------------
    def folder_properties_delete(self, widget, event):
        self.folder_properties_dialog.hide()
        return True;

    def folder_properties_response(self, widget, response):
        if response == 2:
            self.folder_properties_dialog.hide()
        elif response == 1:
            title = self.folder_title_prop.get_text()
            if len(title) > 0:
                self.feed_properties_dialog.hide()
                selection = self.feed_tree.get_selection()
                model, iter = selection.get_selected()
                path = model.get_path(iter)
                self.feed_tree_store[path][self.fts_feed_title] = title
                self.folder_properties_dialog.hide()

    # found feeds dialog stuff ------------------------------------------------------------------
    def found_feeds_delete(self, widget, event):
        self.found_feeds_dialog.hide()
        return True;

    def found_feeds_response(self, widget, response):
        if response == 1:
            self.found_feeds_dialog.hide()

    # feed properties dialog stuff -------------------------------------------------------------------
    def feed_properties_delete(self, widget, event):
        self.feed_properties_dialog.hide()
        return True;

    def feed_properties_response(self, widget, response):
        if response == 3:
            self.feed_properties_dialog.hide()
        elif response == 2:
            uri = self.feed_url_prop.get_text()
            uri = "http://www.feedvalidator.org/check.cgi?url=" + uri
            id = self.browsertabs.get_current_page()
            nbPage = self.browsertabs.get_nth_page(id)
            nbPage.goto_page(uri)
            self.feed_properties_dialog.hide()
        elif response == 1:
            title = self.feed_title_prop.get_text()
            uri = self.feed_url_prop.get_text()
            if len(title) > 0 and len(uri) > 0:
                self.feed_properties_dialog.hide()
                selection = self.feed_tree.get_selection()
                model, iter = selection.get_selected()
                path = model.get_path(iter)
                self.feed_tree_store[path][self.fts_feed_title] = title
                self.feed_tree_store[path][self.fts_feed_uri] = uri

    # confirm dialog stuff ----------------------------------------------------------------------------------
    def confirm_dialog_show(self, widget):
        selection = self.feed_tree.get_selection()
        model, iter = selection.get_selected()
        type = self.feed_tree_store.get_value(iter, self.fts_type)
        self.no_button.grab_focus()
        if type == "folder":
            self.confirm_dialog.set_property("text", "Are you sure want to delete this folder?")
            self.confirm_dialog.set_property("secondary-text", "Any feeds in the folder will also be deleted.")
        else:
            self.confirm_dialog.set_property("text", "Are you sure want to delete this feed?")
            self.confirm_dialog.set_property("secondary-text", "")
        self.confirm_dialog.show()

    def confirm_dialog_delete(self, widget, event):
        self.confirm_dialog.hide()
        return True;

    def confirm_dialog_response(self, widget, response):
        if response == 2:
            self.confirm_dialog.hide()
        elif response == 1:
            selection = self.feed_tree.get_selection()
            model, iter = selection.get_selected()
            self.delete_feed_items(iter)
            self.conn.commit()
            model.remove(iter)
            self.confirm_dialog.hide()

    # recursively remove feed items and feeds from the database, for the feed or folder that was deleted
    def delete_feed_items(self, iter):
        treeid = self.feed_tree_store.get_value(iter, self.fts_feed_id)
        if self.feed_tree_store.get_value(iter, self.fts_type) == "feed":
            self.cursor.execute("DELETE FROM items WHERE feedid = ?", [treeid])
        self.cursor.execute("DELETE FROM feeds WHERE feedid = ?", [treeid])
        if self.feed_tree_store.iter_has_child(iter):
            self.delete_feed_kids(self.feed_tree_store.iter_children(iter))

    def delete_feed_kids(self, iter):
        while iter:
            treeid = self.feed_tree_store.get_value(iter, self.fts_feed_id)
            if self.feed_tree_store.get_value(iter, self.fts_type) == "feed":
                self.cursor.execute("DELETE FROM items WHERE feedid = ?", [treeid])
            self.cursor.execute("DELETE FROM feeds WHERE feedid = ?", [treeid])
            if self.feed_tree_store.iter_has_child(iter):
                self.delete_feed_kids(self.feed_tree_store.iter_children(iter))
            iter = self.feed_tree_store.iter_next(iter)

    # message dialog stuff ----------------------------------------------------------------------------------
    def message_dialog_response(self, widget, response):
        if response == -5:
            self.message_dialog.hide()

    def message_dialog_delete(self, widget, event):
        self.message_dialog.hide()
        return True;

    # write folders and feeds, from the feed tree, to the database
    def save_tree(self, iter, level):
        if level == 0:
            self.tree_order = 0
        while iter:
            tpath = self.feed_tree_store.get_path(iter)
            expanded = self.feed_tree.row_expanded(tpath)
            treeid = self.feed_tree_store.get_value(iter, self.fts_feed_id)
            title = self.feed_tree_store.get_value(iter, self.fts_feed_title)
            uri = self.feed_tree_store.get_value(iter, self.fts_feed_uri)
            if level == 0:
                path = None
            else:
                parent = self.feed_tree_store.iter_parent(iter)
                path = str(self.feed_tree_store.get_path(parent))
                path = path.replace("(", "")
                path = path.replace(",)", "")
                path = path.replace(")", "")
                path = path.replace(", ", ":")
            treetype = self.feed_tree_store.get_value(iter, self.fts_type)
            
            hitcount = self.feed_tree_store.get_value(iter, self.fts_hitcount)
            lasthitdate = self.feed_tree_store.get_value(iter, self.fts_lasthitdate)
            wlstatus = self.feed_tree_store.get_value(iter, self.fts_wl_status)

            self.tree_order += 1
            
            self.cursor.execute("UPDATE feeds SET treeorder = ?, name = ?, uri = ?, treepath = ?, type = ?, hitcount = ?, lasthitdate = ?, wlstatus = ?, expanded = ? WHERE feedid = ?", (self.tree_order, title, uri, path, treetype, hitcount, lasthitdate, wlstatus, expanded, treeid))
            
            if self.feed_tree_store.iter_has_child(iter):
                child_level = level + 1
                self.save_tree(self.feed_tree_store.iter_children(iter), child_level)
            iter = self.feed_tree_store.iter_next(iter)
        if not iter and level == 0:
            self.conn.commit()

    # mouse moves over a feed item in the item list
    def item_hover(self, widget, event):
        path = self.item_list.get_path_at_pos(int(event.x), int(event.y))
        if path:
            if path[0][0] != self.current_row:
                row = path[0][0]
                self.item_list_store[self.current_row][self.ils_pixbuf_tick] = self.pixbuf_none
                self.item_list_store[row][self.ils_pixbuf_tick] = self.pixbuf_tick
                self.current_row = row
                self.show_item_summary(path)

    # something was dropped on the item list. try to parse it as a feed
    def il_drag_data_received(self, widget, context, x, y, selection, info, etime):
        soup = BeautifulSoup(''.join(selection.data))
        if soup("a")[0]["href"]:
            context.finish(gtk.TRUE, gtk.TRUE, etime)
            self.read_feed(soup("a")[0]["href"], '0', 0)

    def ft_drag_drop(self, treeview, context, x, y, etime):
        try:
            path, pos = treeview.get_dest_row_at_pos(x, y)
            if context.action == gtk.gdk.ACTION_MOVE:
                if self.feed_tree_store[path][self.fts_type] == "folder":
                    return False
                else:
                    if pos == gtk.TREE_VIEW_DROP_INTO_OR_BEFORE or pos == gtk.TREE_VIEW_DROP_INTO_OR_AFTER:
                        return True
                    else:
                        return False
        except:
            pass

    def ft_drag_data_received(self, treeview, context, x, y, selection, info, etime):
        if context.targets[0] != "GTK_TREE_MODEL_ROW":
            try:
                path, pos = treeview.get_dest_row_at_pos(x, y)
                model, iter_to_copy = treeview.get_selection().get_selected()
                target_iter = model.get_iter(path)
                soup = BeautifulSoup(''.join(selection.data))
                if soup("a")[0]["href"] and soup("a")[0].contents:
                    uri = soup("a")[0]["href"]
                    title = "".join(soup("a")[0].contents)
                    recid = self.add_folder_feed(title)
                    if self.feed_tree_store[path][self.fts_type] == "folder":
                        model.append(target_iter, [title, uri, "Bold " + self.feed_tree_font_size, recid, "feed", 0, None, 0, self.pixbuf_none, self.pixbuf_feed])
                    else:
                        if pos == gtk.TREE_VIEW_DROP_BEFORE:
                            model.insert_before(None, target_iter, [title, uri, "Bold " + self.feed_tree_font_size, recid, "feed", 0, None, 0, self.pixbuf_none, self.pixbuf_feed])
                        elif pos == gtk.TREE_VIEW_DROP_AFTER:
                            model.insert_after(None, target_iter, [title, uri, "Bold " + self.feed_tree_font_size, recid, "feed", 0, None, 0, self.pixbuf_none, self.pixbuf_feed])
            except:
                pass

    # updates the unread status of a feed/enclosure item, and removes the URI from the history database
    def remove_history(self, uri, row):
        if self.item_list_store[row][self.ils_item_uri] == uri:
            self.item_list_store[row][self.ils_item_read] = 1
        elif self.item_list_store[row][self.ils_enc_uri] == uri:
            self.item_list_store[row][self.ils_enc_read] = 1
        self.item_list_store[row][self.ils_pixbuf] = self.get_indicator(self.item_list_store[row][self.ils_item_read], self.item_list_store[row][self.ils_enc_read])
        self.cursor.execute("DELETE FROM history WHERE uri = ?", [uri])
        
    # Updates the read status of a feed/enclosure item, and updates or adds the URI to the history database
    def update_history(self, uri, commit, row = None):
        if (not row):
            for row in self.item_list_store:
                if row[self.ils_item_type] == "item" and row[self.ils_item_uri] == uri:
                    row[self.ils_item_read] = 2
                elif row[self.ils_item_type] == "item" and row[self.ils_enc_uri] == uri:
                    row[self.ils_enc_read] = 2
                row[self.ils_pixbuf] = self.get_indicator(row[self.ils_item_read], row[self.ils_enc_read])
        else:
            if self.item_list_store[row][self.ils_item_type] == "item" and self.item_list_store[row][self.ils_item_uri] == uri:
                self.item_list_store[row][self.ils_item_read] = 2
            elif self.item_list_store[row][self.ils_item_type] == "item" and self.item_list_store[row][self.ils_enc_uri] == uri:
                self.item_list_store[row][self.ils_enc_read] = 2
            self.item_list_store[row][self.ils_pixbuf] = self.get_indicator(self.item_list_store[row][self.ils_item_read], self.item_list_store[row][self.ils_enc_read])

        now = int(time.time())
        self.cursor.execute("SELECT COUNT(uri) FROM history WHERE uri = ?", [uri])
        row = self.cursor.fetchone()
        if row[0] == 0:
            self.cursor.execute("INSERT INTO history(uri, hitcount, lasthitdate) VALUES (?, ?, ?)", (uri, 1, now))
        else:
            self.cursor.execute("UPDATE history SET hitcount = hitcount + 1, lasthitdate = ? WHERE uri = ?", (now, uri))

        if commit == True:
            self.conn.commit()

    # works out which image to display next to an item in the item list
    def get_indicator(self, read, enc_read):
        if read == 0 and enc_read == 0:
			pixbuf = self.pixbuf_none
		elif (read == 2 and enc_read == 2) or (read == 2 and enc_read == 0) or (read == 0 and enc_read == 2):
			pixbuf = self.pixbuf_read
		elif (read == 1 and enc_read == 1) or (read == 1 and enc_read == 0) or (read == 0 and enc_read == 1):
			pixbuf = self.pixbuf_unread
		else:
			pixbuf = self.pixbuf_partread
        return pixbuf

    # keyboard is used to navigate up and down the item list
    def item_key_released(self, widget, event):
        if event.keyval == 65364 or event.keyval == 65362 or event.keyval == 65433 or event.keyval == 65431:
            selection = self.item_list.get_selection()
            (model, iter) = selection.get_selected()
            if iter:
                path = model.get_path(iter)
                if path[0] != self.current_row:
                    self.item_list_store[self.current_row][self.ils_pixbuf_tick] = self.pixbuf_none
                    self.item_list_store[path[0]][self.ils_pixbuf_tick] = self.pixbuf_tick
                    self.current_row = path[0]
                    self.show_item_summary(path)
                    uri = self.item_list_store[path[0]][self.ils_item_uri]
                    if len(uri) > 0:
                        self.update_history(uri, True, path[0])
                        id = self.browsertabs.get_current_page()
                        nbPage = self.browsertabs.get_nth_page(id)
                        nbPage.goto_page(uri)

    # if a link is clicked in the msg window, stop the naviagtion and send the main browser to the URI
    def nav_request_policy_decision(self, view, frame, net_req, nav_act, pol_dec):
        uri = net_req.get_uri()
        self.update_history(uri, True)
        id = self.browsertabs.get_current_page()
        nbPage = self.browsertabs.get_nth_page(id)
        nbPage.goto_page(uri)
        pol_dec.ignore()
        return True

    # show feed item summary at the bottom of the sidebar
    def show_item_summary(self, path):
            html = "<div style='font-size: 12px;'><div style='color: darkred; font-weight: bold; padding-bottom: 15px;'>" + self.item_list_store[path[0]][self.ils_item_title] + "</div><div style='padding-bottom: 15px;'>" + self.item_list_store[path[0]][self.ils_item_summary] + "</div><div style='font-style: italic;'>" + self.item_list_store[path[0]][self.ils_item_date] + "</div></div>"
            self.msg.load_string(html, "text/html", "utf-8", "")

    # find feeds in the current HTML page
    def find_feeds(self, widget):
        id = self.browsertabs.get_current_page()
        nbPage = self.browsertabs.get_nth_page(id)
        head, title = nbPage.get_head()
        if title:
            soup = BeautifulSoup(''.join(head))
            rss = soup.findAll('link', type = "application/rss+xml")
            atom = soup.findAll('link', type = "application/atom+xml")
            if len(rss) == 0 and len(atom) == 0:
                self.message_dialog.set_property("text", "No feeds found")
                self.message_dialog.set_property("secondary-text", "The page titled '" + title + "', doesn't contain any feeds")
                self.message_dialog.show()
            else:
                self.found_dialog_msg.set_text("Feeds found on the page titled '" + title + "' are listed below")
                html = "<div style='font-size: 12px;'>"
                if len(rss) > 0:
                    html += "<div style='font-size: 16px; font-weight: bold; padding-bottom: 5px;'>RSS Feeds</div>"
                    for i in range(len(rss)):
                        html += "<a href='" + rss[i]["href"] + "'>" + rss[i]["title"] + "</a><br/>"
                    html += "<br/>"
                if len(atom) > 0:
                    html += "<div style='font-size: 16px; font-weight: bold; padding-bottom: 5px;'>Atom Feeds</div>"
                    for i in range(len(atom)):
                        html += "<a href='" + atom[i]["href"] + "'>" + atom[i]["title"] + "</a><br/>"
                html += "</div>"
                self.found_list.load_string(html, "text/html", "utf-8", "")
                self.found_feeds_dialog.show()

class notebookPage:

    def __init__(self, notebook, uri, parent):

        self.notebook = notebook
        self.parent = parent

        # ---- create all the stuff for the notebook page

        # create a scrolled window and a browser (an instance of webkit). Add the browser to the scrolled window
        scrolled_browser = gtk.ScrolledWindow()
        self.webview = webkit.WebView()
        self.webview.set_settings(self.parent.websettings)
        self.webview.connect("notify::load-status", self.load_status)
        self.webview.connect("notify::title", self.title_changed)
        self.webview.connect("navigation-policy-decision-requested", self.nav_request_policy_decision)
        self.webview.connect("new-window-policy-decision-requested", self.on_new_window_policy_decision_requested)
        #self.webview.connect("navigation-requested", self.on_navigation_requested)
        scrolled_browser.add(self.webview)

        # create an entry that allows the user to enter a URI. The entry will also display the URI of the current web page
        self.toolbar_uri = gtk.Entry()
        self.toolbar_uri.connect("activate", self.toolbar_uri_activate)

        # create a vbox to hold the various objects on the notebook page
        self.page_vbox = pageVbox(self.webview);
        self.page_vbox.connect("key-press-event", self.key_press_cb)
        self.page_vbox.set_spacing(3)

        # create a toolbar to put above the browser
        browser_toolbar = gtk.Toolbar()
        browser_toolbar.set_style(gtk.TOOLBAR_ICONS)
        browser_toolbar.set_icon_size(gtk.ICON_SIZE_MENU)
        browser_toolbar.set_show_arrow(False)

        # create a back button and add it to the toolbar
        self.back_button = gtk.ToolButton(gtk.STOCK_GO_BACK)
        self.back_button.set_sensitive(False)
        self.back_button.connect("clicked", self.go_back)
        self.back_button.set_tooltip_text("Go back")
        browser_toolbar.add(self.back_button)

        # create a forward button and add it to the toolbar
        self.forward_button = gtk.ToolButton(gtk.STOCK_GO_FORWARD)
        self.forward_button.set_sensitive(False)
        self.forward_button.connect("clicked", self.go_forward)
        self.forward_button.set_tooltip_text("Go forward")
        browser_toolbar.add(self.forward_button)

        # create a refresh button and add it to the toolbar
        self.refresh_button = gtk.ToolButton(gtk.STOCK_REFRESH)
        self.refresh_button.connect("clicked", self.refresh_clicked)
        browser_toolbar.add(self.refresh_button)

        # create a home button and add it to the toolbar
        home_button = gtk.ToolButton(gtk.STOCK_HOME)
        home_button.connect("clicked", self.go_home)
        home_button.set_tooltip_text("Go to the Wizz RSS home page")
        browser_toolbar.add(home_button)

        # create a preview button and add it to the toolbar
        preview_button = gtk.ToolButton(gtk.STOCK_ABOUT)
        preview_button.connect("clicked", self.preview_details)
        preview_button.set_tooltip_text("Preview the current feed")
        browser_toolbar.add(preview_button)

        # create a tool item (an entry object can't be added directly to a toolbar) to hold the URI entry and add the item to the toolbar
        toolbar_uri_item = gtk.ToolItem()
        toolbar_uri_item.set_expand(True)
        toolbar_uri_item.add(self.toolbar_uri)
        browser_toolbar.add(toolbar_uri_item)

        self.toolbar_search = gtk.Entry()
        self.toolbar_search.set_icon_from_stock(gtk.ENTRY_ICON_PRIMARY, gtk.STOCK_FIND);
        self.toolbar_search.set_tooltip_text("Search the current web page")
        self.toolbar_search.connect("activate", self.toolbar_search_activate)

        toolbar_search_item = gtk.ToolItem()
        toolbar_search_item.add(self.toolbar_search)
        browser_toolbar.add(toolbar_search_item)

        # add the toolbar and the browser to the notebook page
        self.page_vbox.pack_start(browser_toolbar, False);
        self.page_vbox.pack_start(scrolled_browser);
        self.page_vbox.show_all()

        # ---- create all the stuff for the notebook page tab

        # create a throbber image to be used for showing browser activity on the notebook page
        self.throbber = gtk.Image()

        # create a label for the notebook page tab
        self.tab_label = gtk.Label("New Tab")
        PLIST = pango.AttrList()
        FONT = pango.AttrSize(10000, 0, -1)
        PLIST.insert(FONT)
        self.tab_label.set_property("attributes", PLIST)
        self.tab_label.set_max_width_chars(12)
        self.tab_label.set_ellipsize(pango.ELLIPSIZE_END)

        # create a close image to be used for closing the tab
        tab_close_image = gtk.Image()
        tab_close_image.set_from_stock(gtk.STOCK_CLOSE, gtk.ICON_SIZE_MENU)

        # create an event box to hold the close image. Buttons look really yuk in the tab, so we used an event box
        self.close_button = gtk.EventBox()
        self.close_button.set_visible_window(False)
        self.close_button.add(tab_close_image)
        self.close_button.connect("button-press-event", self.close_tab)

        # create a hbox to hold all the tab stuff and add the throbber, the tab label and the close button
        tab_hbox = gtk.HBox()
        tab_hbox.set_spacing(3)
        tab_hbox.pack_start(self.throbber)
        tab_hbox.pack_start(self.tab_label)
        tab_hbox.pack_start(self.close_button)
        tab_hbox.show_all()

        # add the page to the notebook and make the page current
        pagenum = self.notebook.append_page(self.page_vbox, tab_hbox)
        self.notebook.set_current_page(pagenum)

        # get the browser to naviagate to the URI passed in the constructor
        self.webview.open(uri);

    def key_press_cb(self, widget, event):
        if event.state & gtk.gdk.CONTROL_MASK:
            if event.keyval == 61:
                self.webview.zoom_in()
            elif event.keyval == 45:
                self.webview.zoom_out()
            elif event.keyval == 48:
                self.webview.set_zoom_level(1.0)

    # tab close button click handler. Make sure that at least one notebook page remains
    def close_tab(self, widget, event):
        if self.notebook.get_n_pages() > 1:
            pageid = self.notebook.page_num(self.page_vbox)
            self.notebook.remove_page(pageid)

    # browser title changed handler. When the title changes, set the notebook page tab to display the title, and set a tooltip if the title is too long
    def title_changed(self, view, frame):
        title = view.get_main_frame().get_title()
        if title != None:
            self.tab_label.set_text(title);
            if len(title) > 15:
                self.tab_label.set_tooltip_text(title)
            else:
                self.tab_label.set_tooltip_text("")

    # user has entered a URI. Tell the browser to navigate to the URI
    def toolbar_uri_activate(self, widget):
        uri = widget.get_text()
        result = urlparse(uri)
        if result[0] == "":
            uri = "http://" + uri
        self.webview.open(uri)

    def toolbar_search_activate(self, widget):
        self.webview.search_text(self.toolbar_search.get_text(), False, True, True)

    def nav_request_policy_decision(self, view, frame, net_req, nav_act, pol_dec):
        uri = net_req.get_uri()
        if nav_act.get_button() == 2:
            self.parent.open_new_tab(None, uri)
            pol_dec.ignore()
            return True
        else:
            if uri == "about:feed":
                self.parent.preview_details(self)
                pol_dec.ignore()
                return True
            else:
                self.parent.update_history(uri, True)

    def on_new_window_policy_decision_requested(self, view, frame, net_req, nav_act, pol_dec):
        uri = net_req.get_uri()
        self.parent.open_new_tab(None, uri)
        pol_dec.ignore()
        return True

    def on_navigation_requested(self, widget, frame, net_req):
        print "on_navigation"
        return False

    def preview_details(self, widget):
        self.parent.preview_details(self)

    # browser (webkit) load status handler
    def load_status(self, view, load_status):
        status = view.get_load_status()
        if status == webkit.LOAD_PROVISIONAL: # load started
            try:
                self.refresh_button.set_stock_id(gtk.STOCK_STOP)
                self.refresh_button.set_tooltip_text("Stop loading this page")
                self.throbber.set_from_file(get_media_file("throbber.gif", True))
                self.check_buttons()
            except:
                pass
        elif status == webkit.LOAD_COMMITTED: # got dom... I think
            try:
                uri = view.get_main_frame().get_uri()
                self.toolbar_uri.set_text(uri)
                if len(uri) > 0:
                    self.parent.update_history(uri, True)
            except:
                pass
        elif status == webkit.LOAD_FINISHED: # load finished
            try:
                self.refresh_button.set_stock_id(gtk.STOCK_REFRESH)
                self.refresh_button.set_tooltip_text("Reload this page")
                self.throbber.set_from_stock(gtk.STOCK_FILE, gtk.ICON_SIZE_MENU) # TODO: put a favicon here, when we work out how to get it out of webkit
                self.check_buttons()
            except:
                pass
        elif status == webkit.LOAD_FAILED: # something went wrong. Also fired by stop_loading()
            try:
                self.refresh_button.set_stock_id(gtk.STOCK_REFRESH)
                self.refresh_button.set_tooltip_text("Reload this page")
                self.throbber.set_from_stock(gtk.STOCK_FILE, gtk.ICON_SIZE_MENU)
                self.check_buttons()
            except:
                pass

    # toolbar refresh / stop button click handler
    def refresh_clicked(self, widget):
        if self.refresh_button.get_stock_id() == gtk.STOCK_REFRESH:
            self.webview.reload() # if the button was clicked while displaying the refresh icon
        else:
            self.webview.stop_loading() # if the button was clicked while displaying the stop icon

    # toolbar back button click handler
    def go_back(self, widget):
        self.webview.go_back()

    # toolbar forward button click handler
    def go_forward(self, widget):
        self.webview.go_forward()

    # toolbar home button click handler
    def go_home(self, widget):
        self.toolbar_uri.set_text("")
        self.webview.open("http://wizzrss.com/ubuntu.php")

    # check if the back and forward toolbar buttons should be enabled or disabled
    def check_buttons(self):
        if self.webview.can_go_back() == True:
            self.back_button.set_sensitive(True)
        else:
            self.back_button.set_sensitive(False)

        if self.webview.can_go_forward() == True:
            self.forward_button.set_sensitive(True)
        else:
            self.forward_button.set_sensitive(False)

class pageVbox(gtk.VBox):

    def __init__(self, webview):
        gtk.VBox.__init__(self)
        self.webview = webview

    def goto_page(self, uri):
        self.webview.open(uri)

    def show_html(self, html):
        self.webview.load_string(html, "text/html", "utf-8", "")

    def get_head(self):
        frame = self.webview.get_main_frame()
        ctx = jswebkit.JSContext(frame.get_global_context())
        text = ctx.EvaluateScript("document.head.innerHTML")
        return [text, frame.get_title()]
