import os
import sys
import glob


class BaseService(object):
    """
    Define a Base Service class used by each service.
    All services must implement at least one of method get_image or set_image.
    """
    # A lower case ascii string
    name = ''

    # An unicode string used on preferences dialog
    title = ''

    # A description used on preferences dialog. Optional
    description = ''

    # If this service provide a widget for authentication
    use_auth = False

    # Specified an icon file path. The icon size must be 24x24 pixels.
    icon = ''

    # A list of methods supported. Posible values are 'set' and/or 'set'.
    provides = []

    # Only used for freedesktop plugin. Optional.
    is_default = False

    def get_image(self):
        """
        Must be implemented only if the service provides a method to get 
        a profile image. This method it's optional.
        Return a file path to the image downloaded.
        """
        pass

    def set_image(self, pixbuf):
        """
        Must be implemented only if the service provides a method to set
        a new profile image. This method it's optional.
        Set image using a GdkPixbuf object.
        """
        pass


def get_installed_services():
    """
    Return a list of classes of installed services.
    """
    services = []
    plugins_dir = os.path.dirname(__file__)
    dir_list = glob.glob('%s/*' % plugins_dir)
    for filename in dir_list:
        if os.path.isdir(filename):
            module_name = 'ubatar.plugins.' + os.path.basename(filename)
            __import__(module_name)
            module = sys.modules[module_name]
            for attr in dir(module):
                if not 'BaseService' in attr and 'Service' in attr:
                    cls = getattr(module, attr)
                    if cls.is_default:
                        services.insert(0, cls)
                    else:
                        services.append(cls)
    return services
