# -*- Mode: Python; coding: utf-8; indent-tabs-mode: nil; tab-width: 4 -*-
### BEGIN LICENSE
# Copyright (C) 2012 Pablo SEMINARIO <pabluk@gmail.com>
# This program is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License version 3, as published 
# by the Free Software Foundation.
# 
# This program is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranties of 
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
# PURPOSE.  See the GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License along 
# with this program.  If not, see <http://www.gnu.org/licenses/>.
### END LICENSE

import os
import gettext
from gettext import gettext as _
gettext.textdomain('ubatar')

from gi.repository import GLib, Gtk, GdkPixbuf, Notify  # pylint: disable=E0611
import logging
logger = logging.getLogger('ubatar')

from ubatar_lib import Window
from ubatar_lib.utils import pixbuf_normalize, get_ribbons_files
from ubatar_lib.keyring import SecretServiceKeyring
from ubatar_lib.ubatarconfig import get_data_file
from ubatar.plugins import get_installed_services
from ubatar.AboutUbatarDialog import AboutUbatarDialog
from ubatar.PreferencesUbatarDialog import PreferencesUbatarDialog


# See ubatar_lib.Window.py for more details about how this class works
class UbatarWindow(Window):
    __gtype_name__ = "UbatarWindow"
    
    def finish_initializing(self, builder): # pylint: disable=E1002
        """Set up the main window"""
        super(UbatarWindow, self).finish_initializing(builder)

        self.AboutDialog = AboutUbatarDialog
        self.PreferencesDialog = PreferencesUbatarDialog

        # Code for other initialization actions should be added here.
        # Initialize the style for the main toolbar
        context = self.ui.toolbar1.get_style_context()
        context.add_class(Gtk.STYLE_CLASS_PRIMARY_TOOLBAR)

        self.ui.filechooserdialog1.add_buttons(
            Gtk.STOCK_CANCEL, Gtk.ResponseType.CANCEL,
            Gtk.STOCK_OPEN, Gtk.ResponseType.OK)

        self.img_default_size = 600
        self.img_thumb_size = 128
        self.ribbons = get_ribbons_files()

        self.pixbuf_selected = None
        last_image_path = self.get_last_image_path()
        if os.path.exists(last_image_path):
            self.update_image_from_file(last_image_path)

        self.services = get_installed_services()
        self.show_icon_for_services()

        self.default_download_service = None
        self.load_download_menu()

    def on_toolbtn_add_clicked(self, filechooser):
        response = filechooser.run()
        if response == Gtk.ResponseType.OK:
            self.image_selected = filechooser.get_filename()
            self.update_image_from_file(self.image_selected)
        filechooser.hide()

    def on_toolbtn_sync_clicked(self, image):
        pixbuf = image.get_pixbuf()
        pixbuf.savev(self.get_last_image_path(), 'png', [], [])
        
        for service in self.services:
            enabled = self.settings.get_boolean('%s-enable' % service.name)
            if enabled:
                if service.use_auth:
                    login = self.settings.get_string('%s-login' % service.name)
                    keyring = SecretServiceKeyring()
                    secret = keyring.get_password(service.name, login)
                    instance = service(login, secret)
                else:
                    instance = service()
                try:
                    instance.set_image(pixbuf)
                except:
                    # TODO: Use a msg box to show errors.
                    logger.debug(u'Error syncing image profile on %s' % service.name)
        self.show_notification()

    def update_image_from_file(self, image_path):
        """Update main image widget using image_path."""
        self.pixbuf_selected = GdkPixbuf.Pixbuf.new_from_file(image_path)
        self.pixbuf_selected = pixbuf_normalize(self.pixbuf_selected, self.img_default_size)
        self.update_image_from_pixbuf(self.pixbuf_selected)
        self.load_ribbons(image_path)

    def update_image_from_pixbuf(self, pixbuf):
        """Update main image widget using a GdkPixbuf object."""
        self.ui.image1.set_from_pixbuf(pixbuf)
        self.resize(self.img_default_size, self.img_default_size)

    def on_preferences_changed(self, settings, key, data=None):
        enabled_services = ['%s-enable' % s.name for s in self.services]
        if key in enabled_services:
            service_name = '-'.join(key.split('-')[:-1])
            img_widget_name = 'img_%s' % service_name
            for img_widget in self.ui.service_icons_box.get_children():    
                if img_widget.get_name() == img_widget_name:
                    img_widget.set_visible(settings.get_boolean(key))
            self.default_download_service = None
            self.load_download_menu()

    def show_icon_for_services(self):
        """Set icon for the enabled services."""
        for service in self.services:
            enabled = self.settings.get_boolean('%s-enable' % service.name)
            icon = get_data_file('media', service.icon)
            img = Gtk.Image.new_from_file(icon)
            img.set_name('img_%s' % service.name)
            img.set_tooltip_text(service.title)
            img.set_margin_top(0)
            img.set_margin_bottom(5)
            img.set_margin_left(10)
            img.set_margin_right(0)
            img.set_visible(enabled)
            self.ui.service_icons_box.pack_start(img, False, True, 0)

    def get_last_image_path(self):
        """
        Get the last image's path.
        If directory doesn't exists it will be created.
        """
        user_cache_dir = GLib.get_user_cache_dir()
        ubatar_cache_dir = os.path.join(user_cache_dir, 'ubatar')
        if not os.path.exists(ubatar_cache_dir):
            os.mkdir(ubatar_cache_dir)
        return os.path.join(ubatar_cache_dir, 'ubatar.png')

    def show_notification(self):
        Notify.init('ubatar')
        notification = Notify.Notification.new(_('Ubatar synchronization'),
            _('Your new profile picture was synchronized with all the enabled services.'),
            self.get_last_image_path())
        notification.show()

    def load_ribbons(self, image_path):
        self.ui.iconview.set_pixbuf_column(0)
        self.ui.liststore.clear()
        for img in self.ribbons.values():
            current_preview = GdkPixbuf.Pixbuf.new_from_file(image_path)
            current_preview = pixbuf_normalize(current_preview, 128)
            pixbuf = GdkPixbuf.Pixbuf.new_from_file_at_size(img, 128, 128)
            pixbuf = GdkPixbuf.Pixbuf.add_alpha(pixbuf, False, 0, 0, 0)
            pixbuf.composite(current_preview, 0, 0, 128, 128, 0, 0, 1, 1, GdkPixbuf.InterpType.BILINEAR, 255)
            self.ui.liststore.append([current_preview])
        self.ui.iconview.connect('selection-changed', self.on_ribbon_selection_changed)
        
    def on_ribbon_selection_changed(self, iconview):
        items = self.ui.iconview.get_selected_items()
        if items:
            item = int(items[0].to_string())
            ribbon_path = self.ribbons[item]
            self.apply_ribbon_to_img(ribbon_path)

    def apply_ribbon_to_img(self, img_path):
        current = self.pixbuf_selected.copy()
        pixbuf = GdkPixbuf.Pixbuf.new_from_file_at_size(img_path, self.img_default_size, self.img_default_size)
        pixbuf = GdkPixbuf.Pixbuf.add_alpha(pixbuf, False, 0, 0, 0)
        pixbuf.composite(current, 0, 0, self.img_default_size, self.img_default_size, 0, 0, 1, 1, GdkPixbuf.InterpType.BILINEAR, 255)
        self.update_image_from_pixbuf(current)

    def on_toolbtn_ribbons_toggled(self, widget, data=None):
        if widget.get_active():
            self.ui.iconview.show()
        else:
            self.ui.iconview.hide()
            self.resize(self.img_default_size, self.img_default_size)

    def load_download_menu(self):
        menu = Gtk.Menu()
        group = None
        for service in self.services:
            enabled = self.settings.get_boolean('%s-enable' % service.name)
            if enabled and not service.is_default \
                and 'get' in service.provides:
                menu_item = Gtk.RadioMenuItem(group=group, label=service.title)
                menu_item.set_name(service.name)
                menu_item.connect('activate', self.on_download_service_activate)
                group = menu_item
                menu.add(menu_item)
                if not self.default_download_service:
                    self.default_download_service = service
        if group:
            menu.show_all()
            self.ui.toolbtn_download.set_menu(menu)
            self.ui.toolbtn_download.set_sensitive(True)
            self.ui.toolbtn_download.set_tooltip_text(_(u'Download profile picture from service'))
        else:
            self.ui.toolbtn_download.set_sensitive(False)
            self.ui.toolbtn_download.set_tooltip_text(_(u'You need enable at least one service'))

    def on_download_service_activate(self, widget, data=None):
        for service in self.services:
            if widget.get_active() and service.name == widget.get_name():
                self.default_download_service = service

    def on_toolbtn_download_clicked(self, widget, data=None):
        service = self.default_download_service
        if service.use_auth:
            login = self.settings.get_string('%s-login' % service.name)
            keyring = SecretServiceKeyring()
            secret = keyring.get_password(service.name, login)
            instance = service(login, secret)
        else:
            instance = service()
        try:
            self.image_selected = instance.get_image()
            self.update_image_from_file(self.image_selected)
        except:
            # TODO: Use a msg box to show errors.
            logger.debug(u'Error fetching image from %s' % service.name)
