# -*- Mode: Python; coding: utf-8; indent-tabs-mode: nil; tab-width: 4 -*-
### BEGIN LICENSE
# Copyright (C) 2012 Marcos Costales https://launchpad.net/~costales
# This program is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License version 3, as published 
# by the Free Software Foundation.
# 
# This program is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranties of 
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
# PURPOSE.  See the GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License along 
# with this program.  If not, see <http://www.gnu.org/licenses/>.
### END LICENSE

import gettext
from gettext import gettext as _
gettext.textdomain('pyenglish')

from gi.repository import Gtk, Gdk
import logging
logger = logging.getLogger('pyenglish')

from pyenglish_lib import Window
from pyenglish.AboutPyenglishDialog import AboutPyenglishDialog

from Sentences import Sentences
import commands
import os
import webbrowser
from multiprocessing import Process

def speak_sentence(sentence):
    os.system("espeak -ven -s136 ?" + sentence)
    
    
# See pyenglish_lib.Window.py for more details about how this class works
class PyenglishWindow(Window):
    __gtype_name__ = "PyenglishWindow"
    TIME_STATUS_BAR = 8

    def finish_initializing(self, builder): # pylint: disable=E1002
        """Set up the main window"""
        super(PyenglishWindow, self).finish_initializing(builder)

        self.AboutDialog = AboutPyenglishDialog

        # Background colors
        self.ui.eventbox_subject.modify_bg(Gtk.StateType.NORMAL, Gdk.color_parse('#c7c8ff'))
        self.ui.eventbox_verb.modify_bg(Gtk.StateType.NORMAL, Gdk.color_parse('#ceffd1'))
        self.ui.eventbox_verb_tense.modify_bg(Gtk.StateType.NORMAL, Gdk.color_parse('#ffc0c0'))
        self.ui.eventbox_option.modify_bg(Gtk.StateType.NORMAL, Gdk.color_parse('#fffbbd'))
        self.ui.eventbox_object.modify_bg(Gtk.StateType.NORMAL, Gdk.color_parse('#ffd4f7'))
        
        # Default XML
        path = os.path.join('data', 'xml', 'default.xml')
        if not os.path.exists(path):
            path = os.path.join('/opt', 'extras.ubuntu.com', 'pyenglish', 'share', 'pyenglish', 'xml', 'default.xml')
        if not os.path.exists(path):
            path = os.path.join('/usr', 'share', 'pyenglish', 'xml', 'default.xml')
        
        # Set initial sentences
        self.extra_lives = 3
        self.sentences = Sentences()
        if self.sentences.load_file(path):
            self._set_window_status('new_sentence')
            self._set_new_sentence()
        else:
            self._set_window_status('error_initial_file')
            self.popup_error()
    
    def _set_window_status(self, status):
        if status == 'error_initial_file':
            self.ui.entry_sentence.set_sensitive(False)
            self.ui.btn_lives.set_sensitive(False)
            self.ui.btn_try_next.set_sensitive(False)
        elif status == 'new_sentence':
            self.ui.pyenglish_window.set_title(_('Practice Your ') + _(self.sentences.get_current_lang()))
            self.ui.btn_try_next.set_sensitive(True)
            self.ui.entry_sentence.set_text('')
            self.ui.img_state.show()
            self.ui.btn_play.hide()
            self.ui.img_state.set_from_stock('gtk-edit', 2)
            self.ui.entry_sentence.set_sensitive(True)
            self.ui.entry_sentence.set_tooltip_text('')
            self.ui.lbl_statusbar.set_text(_('Write the right sentence in the entry box'))
            if self.extra_lives > 0:
                self.ui.btn_lives.set_sensitive(True)
            self.ui.entry_sentence.grab_focus()
        elif status == 'complete_ok':
            self.ui.lbl_statusbar.set_text('')
            self.ui.entry_sentence.set_sensitive(False)
            self.ui.btn_lives.set_sensitive(False)
            self.ui.btn_play.set_sensitive(True)
            self.ui.lbl_statusbar.set_text(_('Well done!! :) Try the next one'))
            self.ui.img_state.hide()
            self.ui.btn_play.show()
            self.ui.btn_try_next.grab_focus()
        elif status == 'partial_ok':
            self.ui.lbl_statusbar.set_text('')
            self.ui.img_state.set_from_stock('gtk-yes', 2)
        elif status == 'partial_no_ok':
            self.ui.lbl_statusbar.set_text('')
            self.ui.img_state.set_from_stock('gtk-no', 2)
        elif status == 'empty_entry':
            self.ui.img_state.set_from_stock('gtk-edit', 2)
        elif status == 'set_lbl_lives':
            if self.extra_lives == 0:
                self.ui.btn_lives.set_sensitive(False)
                self.ui.btn_lives.set_label(_('No Lives'))
            elif self.extra_lives == 1:
                self.ui.btn_lives.set_label(_('1 Live'))
            elif self.extra_lives == 2:
                self.ui.btn_lives.set_label(_('2 Lives'))
            elif self.extra_lives == 3:
                self.ui.btn_lives.set_label(_('3 Lives'))
        elif status == 'used_live':
            self.ui.btn_lives.set_sensitive(False)
            self.ui.entry_sentence.set_sensitive(False)
            self.ui.entry_sentence.set_text(self.sentences.get_current_solution())
            self.ui.entry_sentence.set_tooltip_text(self.sentences.get_current_solution())
            self.ui.lbl_statusbar.set_text(_('That was the sentence!'))

    def _set_new_sentence(self):
        """Set initial random sentence"""
        self.sentences.set_sentence()
        self.ui.lbl_subject.set_text(self.sentences.get_current_subject())
        self.ui.lbl_verb.set_text(self.sentences.get_current_verb())
        self.ui.lbl_tense.set_text(self.sentences.get_current_tense())
        self.ui.lbl_option.set_text(self.sentences.get_current_option())
        self.ui.lbl_object.set_text(self.sentences.get_current_object())
        self._set_window_status('new_sentence')
        
    def popup_error(self):
        dlg_error = Gtk.MessageDialog(None, 
                    Gtk.DialogFlags.MODAL | Gtk.DialogFlags.DESTROY_WITH_PARENT,
                    Gtk.MessageType.WARNING, Gtk.ButtonsType.CLOSE, 
                    "Error loading the sentences. Try to load another file")
        dlg_error.set_title("Error")
        dlg_error.run()
        dlg_error.destroy()
        return
        
    # GUI Events
    def on_entry_sentence_changed(self, widget):
        """Check sentence is completed"""
        if self.ui.entry_sentence.get_text() == '':                              # empty
            self._set_window_status('empty_entry')
        elif self.sentences.check_is_ok(self.ui.entry_sentence.get_text()):      # Check complete sentence!
            self._set_window_status('complete_ok')
        elif self.sentences.check_is_partial(self.ui.entry_sentence.get_text()): # Check start sentence is right
            self._set_window_status('partial_ok')
        else:                                                                    # Wrong sentence
            self._set_window_status('partial_no_ok')
    
    def on_btn_try_next_clicked(self, widget):
        self._set_new_sentence()
    
    def on_mnu_open_activate(self, widget):
        """Open sentences template"""
        dialog = Gtk.FileChooserDialog(_("Please choose a file"), self,
            Gtk.FileChooserAction.OPEN,
            (Gtk.STOCK_CANCEL, Gtk.ResponseType.CANCEL,
             Gtk.STOCK_SAVE, Gtk.ResponseType.OK))
        filter_xml = Gtk.FileFilter()
        filter_xml.set_name(_("XML files"))
        filter_xml.add_pattern("*.xml")
        dialog.add_filter(filter_xml)
        response = dialog.run()

        if response == Gtk.ResponseType.OK:
            dialog.hide()
            if self.sentences.load_file(dialog.get_filename()):
                self.extra_lives = 3
                self._set_window_status('set_lbl_lives')
                self._set_window_status('new_sentence')
                self._set_new_sentence()
            else:
                self.popup_error()
                
        dialog.destroy()

    def on_btn_lives_activate(self, widget):
        """Extra lives"""
        self.extra_lives = self.extra_lives - 1
        self._set_window_status('set_lbl_lives')
        self._set_window_status('used_live')
    
    def on_mnu_repository_activate(self, widget):
        """Launch browser with Sentences Repository"""
        webbrowser.open_new("http://pyenglish.tuxfamily.org/index.html#templates")
    
    def on_mnu_bug_activate(self, widget):
        """Report bug"""
        webbrowser.open_new("https://bugs.launchpad.net/pyenglish")
    
    def on_btn_play_clicked(self, widget):
        """Play sentence"""
        sentence = ','.join(self.ui.entry_sentence.get_text().split()).decode('utf8')
        sentence = sentence.replace("'", r"\'")
        self.ui.btn_play.set_sensitive(False)
        speak = Process(target=speak_sentence, args=(sentence,))
        speak.start()
    
