# -*- Mode: Python; coding: utf-8; indent-tabs-mode: nil; tab-width: 4 -*-
##
# Project: ppasc - PPA Software Center
# Author: Marco Scarpetta <marcoscarpetta@mailoo.org>
# Copyright: 2012 Marco Scarpetta
# License: GPL-3
# This program is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License version 3, as published 
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
# FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
# more details.
#
# On Debian GNU/Linux systems, the full text of the GNU General Public License
# can be found in the file /usr/share/common-licenses/GPL-3.
##

import gettext
from gettext import gettext as _
gettext.textdomain('ppasc')

import cairo
from gi.repository import Gtk,GObject,Gdk
from ppasc_lib import LpSearch, Menu
import thread
import sys

GObject.threads_init()

class MainWin():
   def __init__(self):
      self.window = Gtk.Window()
      self.window.connect('delete-event',Gtk.main_quit)
      self.window.set_title('PPA Software Center')
      image = cairo.ImageSurface.create_from_png('/opt/extras.ubuntu.com/ppasc/share/ppasc/media/64.png')
      icona = Gdk.pixbuf_get_from_surface(image,0,0,64,64)
      self.window.set_icon(icona)
      self.window.maximize()
      
      self.stop_search = False
      self.running = False
      self.tab = 0
      self.busy = False
      
      self.installing = Gtk.Label()
      self.progbar = Gtk.ProgressBar()
      self.progbar.set_property('margin',20)
      
      self.vbox = Gtk.Grid()
      #self.vbox.set_row_spacing(20)
      
      self.notebook = Gtk.Notebook()
      self.s_grid = Gtk.Grid()
      self.s_grid.set_column_homogeneous(False)
      self.i_grid = Gtk.Grid()
      #self.i_grid.set_column_spacing(30)
      #self.i_grid.set_row_spacing(30)
      
      self.b1=Gtk.Button()
      self.b1.set_image(Gtk.Image.new_from_stock(Gtk.STOCK_FIND,Gtk.IconSize.LARGE_TOOLBAR))
      self.b1.set_margin_bottom(20)
      self.b1.connect('pressed',self.change,0)
      self.b1.set_sensitive(False)
      
      self.b2=Gtk.Button()
      self.b2.set_image(Gtk.Image.new_from_stock(Gtk.STOCK_INFO,Gtk.IconSize.LARGE_TOOLBAR))
      self.b2.set_margin_bottom(20)
      self.b2.connect('pressed',self.change,1)
      
      menu=Menu.get_menu(self)
      
      self.vbox.attach(menu,1,1,2,1)
      self.vbox.attach(self.b1,1,2,1,1)
      self.vbox.attach(self.b2,2,2,1,1)
   
      self.vbox.attach(self.s_grid,1,3,2,1)
      #self.notebook.append_page(self.s_grid,Gtk.Label(_('Search')))
      #self.notebook.append_page(self.i_grid,Gtk.Label(_('Info')))
      
      self.entry = Gtk.Entry()
      self.entry.set_property("activates-default",True)
      self.button = Gtk.Button.new_from_stock(Gtk.STOCK_FIND)
      self.button.connect('pressed',self.search)
      self.button.connect('activate',self.search)
      self.stop_b = Gtk.Button.new_from_stock(Gtk.STOCK_STOP)
      self.stop_b.connect('pressed',self.stop)
      self.spinner = Gtk.Spinner()
      self.exact = Gtk.Switch()
      
      self.packages = Gtk.ListStore(str,str,str,str,str,str,str)
      tree_view = Gtk.TreeView.new_with_model(self.packages)
      cell = Gtk.CellRendererText()
      column = Gtk.TreeViewColumn(_('Package name'), cell, text = 0)
      column.set_sort_column_id(0)
      tree_view.append_column(column)
      cell = Gtk.CellRendererText()
      column = Gtk.TreeViewColumn(_('Version'), cell, text = 1)
      column.set_sort_column_id(1)
      tree_view.append_column(column)
      cell = Gtk.CellRendererText()
      column = Gtk.TreeViewColumn(_('PPA'), cell, text = 2)
      column.set_sort_column_id(2)
      tree_view.append_column(column)
      cell = Gtk.CellRendererText()
      column = Gtk.TreeViewColumn(_('PPA status'), cell, text = 3)
      column.set_sort_column_id(3)
      tree_view.append_column(column)
      cell = Gtk.CellRendererText()
      column = Gtk.TreeViewColumn(_('Local version'), cell, text = 4)
      column.set_sort_column_id(4)
      tree_view.append_column(column)
      tree_view.connect("row-activated",self.selected)
      scrolledwindow = Gtk.ScrolledWindow()
      scrolledwindow.set_hexpand(True)
      scrolledwindow.set_vexpand(True)
      scrolledwindow.add(tree_view)
      
      self.s_grid.attach(self.entry,1,1,2,1)
      self.s_grid.attach(self.button,3,1,1,1)
      self.s_grid.attach(self.stop_b,4,1,1,1)
      self.s_grid.attach(self.spinner,5,1,1,1)
      self.s_grid.attach(Gtk.Label(_('Exact match:')),1,2,1,1)
      self.s_grid.attach(self.exact,2,2,1,1)
      self.s_grid.attach(scrolledwindow,1,3,5,1)
      
      self.info_label = Gtk.Label(_('No package selected!'))
      self.info_label.set_selectable(True)
      self.info_label.set_line_wrap(True)
      self.info_label.set_halign(Gtk.Align.START)
      self.info_label.set_valign(Gtk.Align.START)
      self.pname = ''
      self.ppa = ''
      self.local = ''
      self.add_ppa = True
      self.info_install = Gtk.Label(_('No package selected!'))
      self.info_install.set_line_wrap(True)
      self.info_install.set_halign(Gtk.Align.START)
      self.info_install.set_property('margin',20)
      self.b_copy = Gtk.Button.new_with_label(_('Copy to clipboard'))
      self.b_copy.connect('pressed', self.copy)
      self.b_copy.set_halign(Gtk.Align.END)
      self.b_copy.set_valign(Gtk.Align.CENTER)
      self.b_copy.set_sensitive(False)
      self.b_copy.set_property('margin',20)
      
      scrolledwindow1 = Gtk.ScrolledWindow()
      scrolledwindow1.set_hexpand(True)
      scrolledwindow1.set_vexpand(True)
      scrolledwindow1.add_with_viewport(self.info_label)
      scrolledwindow1.set_property('margin',20)
      scrolledwindow1.set_border_width(2)
      
      self.i_grid.attach(self.b_copy,2,2,1,1)
      self.i_grid.attach(self.info_install,1,2,1,1)
      self.i_grid.attach(scrolledwindow1,1,1,2,1)
      
      #for widget in self.i_grid.get_children():
      #   widget.set_halign(Gtk.Align.START)
      #   widget.set_valign(Gtk.Align.START)
      
      #for widget in self.s_grid.get_children():
      #   widget.set_property('margin',20)

      #self.spinner.set_halign(Gtk.Align.FILL)
      #scrolledwindow.set_halign(Gtk.Align.FILL)
      #self.entry.set_halign(Gtk.Align.FILL)
      
      self.window.add(self.vbox)
      
      self.button.set_can_default(True)
      self.button.grab_default()
   
   def change(self, widget, tab):
      if tab == 1 and self.tab == 0:
         self.tab=1
         self.vbox.remove(self.s_grid)
         self.vbox.attach(self.i_grid,1,3,2,1)
         self.b1.set_sensitive(True)
         self.b2.set_sensitive(False)
      elif tab == 0 and self.tab == 1:
         self.tab=0
         self.vbox.remove(self.i_grid)
         self.vbox.attach(self.s_grid,1,3,2,1)
         self.b1.set_sensitive(False)
         self.b2.set_sensitive(True)
         self.button.set_can_default(True)
         self.button.grab_default()
      self.window.show_all()
      
   def search(self, widget):
      if self.running == True:
          self.show_dialog(_('A search is already in progress!'),
            [_('OK')],
            _('Stop the current search and start a new one.'))
      elif self.entry.get_text() != '':
         self.running = True
         self.clean_packages()
         string = self.entry.get_text()
         exact = self.exact.get_active()
         self.spinner.start()
         thread.start_new_thread(LpSearch.get_packages,(string,exact,self.add,self.end,self.stopped))
   
   def stop(self,widget):
      self.stop_search = True
      
   def stopped(self):
      return self.stop_search
   
   def add(self, ppa, status, lversion, package, version, description, website):
      if description == None:
         description = 'Empty'
      self.packages.append([package,version,ppa,status,lversion,description,website])
   
   def end(self):       
      self.spinner.stop()
      self.stop_search = False
      self.running = False
   
   def clean_packages(self):
      treeiter = self.packages.get_iter_first()
      ntree = treeiter
      while ntree:
		ntree=self.packages.iter_next(treeiter)
		self.packages.remove(treeiter)
		treeiter=ntree
        
   def selected(self,widget, path, view_column):
      text = ''
      infos = [
         _('Package name:'),
         _('Package version:'),
         _('PPA:'),
         _('PPA Status:'),
         _('Local version:'),
         _('PPA description:'),
         _('PPA Url:')]
      n=0
      while n<6:
         info=self.packages[path][n].replace('&','&amp;')
         info=info.replace('<','&lt;')
         info=info.replace('>','&gt;')
         text=text+'<b>'+infos[n]+'</b>\n\n'+info+'\n\n'
         n=n+1
      text=text+'<b>'+infos[6]+'</b>\n\n<a href="'+self.packages[path][6]+'">'+self.packages[path][6]+'</a>\n\n'
      self.info_label.set_markup(text)
      if self.packages[path][4] == _('Major'):
         text = '<b>This package can\'t be installed on your system, because the local version is already more up to date than the one in the PPA.</b>'
         self.b_copy.set_sensitive(False)
      else:
         if self.packages[path][3] == _('Not present'):
            text = _('<b>To install this package run:</b>\n\n')+'sudo add-apt-repository ppa:%s &amp;&amp; ' %self.packages[path][2]
         else:
            text = _('<b>To install this package run:</b>\n\n')
         text = text+'sudo apt-get update &amp;&amp; sudo apt-get install %s' %self.packages[path][0]
         self.b_copy.set_sensitive(True)
      self.info_install.set_markup(text)
      self.change(None,1)
   
   def copy(self,widget):
      text = self.info_install.get_text().split('\n')[2]
      c = Gtk.Clipboard.get(Gdk.SELECTION_CLIPBOARD)
      c.set_text(text,len(text))
      c.store()
   
   def show_dialog(self, title, buttons, lab, response=None):
      if response == None: response = self.destroy_dialog
      dialog = Gtk.Dialog(title,
         self.window,
         Gtk.DialogFlags.MODAL)
      n=0
      for button in buttons:
         dialog.add_button(button,n)
         n=n+1 
      label=Gtk.Label(lab)
      label.set_property('margin',20)
      dialog.get_content_area().add(label)
      dialog.connect('response', response)
      dialog.show_all()
   
   def destroy_dialog(self,dialog,ide):
      dialog.destroy()
   
   def show(self):
      self.window.show_all()

def main():
   main_win = MainWin()
   main_win.show()
   Gtk.main()
