// FIXME: Remove signal handlers

public extern const string VERSION;
public extern const string GETTEXT_PACKAGE;
public extern const string LOCALE_DIR;

public class Picsaw : Gtk.Application
{
    private Gtk.Window window;
    private View view;
    private int window_width = 800;
    private int window_height = 600;
    private bool window_is_maximized = false;

    private string get_cache_dir ()
    {
        return Path.build_filename (Environment.get_user_cache_dir (), "picsaw");
    }

    public Picsaw ()
    {
        // FIXME: What domain?
        Object (application_id: "net.launchpad.picsaw");
    }

    public override void startup ()
    {
        base.startup ();
        window = new Gtk.ApplicationWindow (this);
        window.configure_event.connect ((event) =>
        {
            if (!window_is_maximized)
            {
                window_width = event.width;
                window_height = event.height;
            }
            return false;
        });
        window.window_state_event.connect ((event) =>
        {
            window_is_maximized = (event.new_window_state & Gdk.WindowState.MAXIMIZED) != 0;
            return false;
        });

        var filename = Path.build_filename (get_cache_dir (), "state");
        var state = new KeyFile ();
        var n_pieces = 15;
        try
        {
            state.load_from_file (filename, KeyFileFlags.NONE);
            if (state.has_group ("window"))
            {
                window_width = state.get_integer ("window", "width");
                window_height = state.get_integer ("window", "height");
                window_is_maximized = state.get_boolean ("window", "is-maximized");
            }
            if (state.has_group ("puzzle"))
                n_pieces = state.get_integer ("puzzle", "n-pieces");
        }
        catch (Error e)
        {
        }

        window.set_size_request (400, 300);
        window.set_default_size (window_width, window_height);
        if (window_is_maximized)
            window.maximize ();

        Puzzle? puzzle = null;
        string? puzzle_filename = null;
        try
        {
            if (state.has_group ("puzzle") && state.has_key ("puzzle", "filename"))
            {
                puzzle = new Puzzle.from_key_file (state);
                puzzle_filename = state.get_string ("puzzle", "filename");
            }
        }
        catch (Error e)
        {
            puzzle = null;
            puzzle_filename = null;
            if (!(e is FileError.NOENT))
                stderr.printf ("Failed to restore game from %s: %s\n", filename, e.message);
        }

        var action = new SimpleAction ("new-puzzle", null);
        action.activate.connect ( () => { view.home (); } );
        add_action (action);

        action = new SimpleAction ("help", null);
        action.activate.connect (help_cb);
        add_action (action);

        action = new SimpleAction ("about", null);
        action.activate.connect (about_cb);
        add_action (action);

        action = new SimpleAction ("quit", null);
        action.activate.connect (quit_cb);
        add_action (action);

        add_accelerator ("<Primary>n", "app.new-puzzle", null);
        add_accelerator ("F1", "app.help", null);
        add_accelerator ("<Primary>q", "app.quit", null);
        add_accelerator ("<Primary>w", "app.quit", null);

        var menu = new Menu ();
        var section = new Menu ();
        menu.append_section (null, section);
        section.append (_("_New Puzzle"), "app.new-puzzle");
        section.append (_("_Help"), "app.help");
        section.append (_("_About"), "app.about");
        section.append (_("_Quit"), "app.quit");
        app_menu = menu;

        view = new View (puzzle, puzzle_filename);
        view.confirm_home.connect (confirm_home_cb);
        view.n_pieces = n_pieces;
        view.show ();
        window.add (view);
    }

    private void confirm_home_cb ()
    {
        var dialog = new Gtk.MessageDialog (window, Gtk.DialogFlags.MODAL, Gtk.MessageType.WARNING, Gtk.ButtonsType.NONE, "%s",
                                            /* Dialog title shown when the user starts a new game while already playing */
                                            _("Puzzle in Progress"));
        dialog.format_secondary_text (/* Dialog message shown when the user starts a new game while already playing */
                                      _("If you discard this puzzle all progress will be permanently lost."));
        dialog.add_button (/* Dialog button to keep playing the current puzzle */
                           _("Keep Playing"), Gtk.ResponseType.REJECT);
        dialog.add_button (/* Dialog button to discard current puzzle and choose a new one */
                           _("Discard Puzzle"), Gtk.ResponseType.ACCEPT);
        dialog.present ();
        dialog.response.connect ( (id) =>
        {
            if (id == Gtk.ResponseType.ACCEPT)
                view.home (false);
            dialog.destroy ();
        } );
    }

    private void help_cb ()
    {
        try
        {
            Gtk.show_uri (window.get_screen (), "help:picsaw", Gtk.get_current_event_time ());
        }
        catch (Error e)
        {
        }
    }

    private void about_cb ()
    {
        string[] authors = { "Robert Ancell", null };
        Gtk.show_about_dialog (window,
                               "version", VERSION,
                               "comments", _("A jigsaw puzzle game that uses your own photos to make puzzles"),
                               "copyright", "Copyright \xc2\xa9 2012 Robert Ancell",
                               "license", """This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.""",
                               "authors", authors,
                               "translator-credits", _("translator-credits"),
                               "logo-icon-name", "picsaw",
                               "website", "http://launchpad.net/picsaw");
    }

    private void quit_cb ()
    {
        window.destroy ();
    }

    public override void activate ()
    {
        window.present ();
    }

    public override void shutdown ()
    {
        base.shutdown ();
        DirUtils.create_with_parents (get_cache_dir (), 0755);

        var filename = Path.build_filename (get_cache_dir (), "state");
        var state = new KeyFile ();
        state.set_integer ("window", "width", window_width);
        state.set_integer ("window", "height", window_height);
        state.set_boolean ("window", "is-maximized", window_is_maximized);
        state.set_integer ("puzzle", "n-pieces", view.n_pieces);
        if (view.puzzle != null && view.puzzle.puzzle.is_shuffled && !view.puzzle.puzzle.is_solved)
        {
            view.puzzle.puzzle.save (state);
            state.set_string ("puzzle", "filename", view.puzzle.filename);
        }
        try
        {
            FileUtils.set_contents (filename, state.to_data ());
        }
        catch (Error e)
        {
        }
    }

    public static int main (string[] args)
    {
        Intl.setlocale (LocaleCategory.ALL, "");
        Intl.bindtextdomain (GETTEXT_PACKAGE, LOCALE_DIR);
        Intl.bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
        Intl.textdomain (GETTEXT_PACKAGE);

        Environment.set_application_name (/* Title of the application */
                                          _("Picsaw"));
        Gtk.Window.set_default_icon_name ("picsaw");

        try
        {
            var error = GtkClutter.init_with_args (ref args, "", new OptionEntry[0], null);
            if (error != Clutter.InitError.SUCCESS)
                return Posix.EXIT_FAILURE;
        }
        catch (Error e)
        {
            stderr.printf ("Failed to initialize Clutter: %s\n", e.message);
            return Posix.EXIT_FAILURE;
        }

        var app = new Picsaw ();
        return app.run ();
    }
}

public class View : GtkClutter.Embed
{
    private List<string> pictures;
    private int picture_index;
    private PuzzleActor puzzles[4];
    public PuzzleActor? puzzle = null;
    private Clutter.Text? error_actor = null;
    private Clutter.Actor puzzle_layer;
    private Clutter.CairoTexture prev_actor;
    private Clutter.CairoTexture next_actor;
    private Clutter.Actor size_actor;
    private Clutter.Actor start_actor;
    private Clutter.CairoTexture home_actor;
    private Clutter.Actor done_actor;
    private Clutter.Text title_text_actor;
    private Clutter.Text text_actor;
    private bool sliding = false;
    public int n_pieces = 30;

    private const float home_x_positions[4] = { 0.25f, 0.75f, 0.75f, 0.25f };
    private const float home_y_positions[4] = { 0.25f, 0.25f, 0.75f, 0.75f };
    private const float hidden_x_positions[4] = { -0.25f, 1.25f, 1.25f, -0.25f };
    private const float hidden_y_positions[4] = { -0.25f, -0.25f, 1.25f, 1.25f };

    public signal void confirm_home ();

    public View (Puzzle? puzzle = null, string? filename = null)
    {
        Clutter.Color color = { 0x2e, 0x34, 0x36, 0xff };
        get_stage ().set_background_color (color);

        pictures = new List<string> ();
        find_pictures (Path.build_filename (Environment.get_user_special_dir (UserDirectory.PICTURES), "Jigsaw Puzzles"));
        if (pictures.length () == 0)
            find_pictures (Path.build_filename (Environment.get_user_special_dir (UserDirectory.PICTURES)));
        if (pictures.length () == 0)
            find_pictures ("/usr/share/backgrounds");

        /* Puzzles on the bottom layer */
        puzzle_layer = new Clutter.Actor ();
        get_stage ().add_child (puzzle_layer);

        /* Load in in progress puzzle */
        if (puzzle != null)
        {
            try
            {
                var picture = new Gdk.Pixbuf.from_file (filename);
                puzzles[0] = this.puzzle = add_puzzle (puzzle, filename, picture);

                /* Put this filename as the first in the list */
                for (unowned List<string> i = pictures; i != null; i = i.next)
                {
                    if (i.data == filename)
                    {
                        pictures.remove_link (i);
                        break;
                    }
                }
                pictures.prepend (filename);
                picture_index++;
                if (picture_index == pictures.length ())
                    picture_index = 0;
            }
            catch (Error e)
            {
                puzzle = null;
            }
        }

        /* Create four random puzzles */
        if (pictures.length () > 0)
            create_puzzles (puzzle != null);
        else
        {
            error_actor = new Clutter.Text ();
            get_stage ().add_child (error_actor);
            // FIXME: Use GTK+ to get the font name
            error_actor.font_name = "ubuntu 20";
            Clutter.Color text_color = { 0xff, 0xff, 0xff, 0xff };
            error_actor.color = text_color;
            /* Text shown when no pictures available */
            error_actor.text = _("You have no pictures.\nGo take some photos!");
            error_actor.set_anchor_point (error_actor.width * 0.5f, error_actor.height * 0.5f);
        }

        size_allocate.connect (size_allocate_cb);

        get_stage ().reactive = true;
        get_stage ().motion_event.connect (motion_cb);
        get_stage ().button_release_event.connect (button_release_cb);

        /* Icons to scroll images */
        prev_actor = new Clutter.CairoTexture (80, 80);
        get_stage ().add_child (prev_actor);
        prev_actor.set_anchor_point (prev_actor.width * 0.5f, prev_actor.height * 0.5f);
        prev_actor.draw.connect (draw_prev_icon_cb);
        prev_actor.invalidate ();
        prev_actor.reactive = true;
        prev_actor.button_press_event.connect (prev_cb);
        next_actor = new Clutter.CairoTexture (80, 80);
        get_stage ().add_child (next_actor);
        next_actor.set_anchor_point (next_actor.width * 0.5f, next_actor.height * 0.5f);
        next_actor.draw.connect (draw_next_icon_cb);
        next_actor.invalidate ();
        next_actor.reactive = true;
        next_actor.button_press_event.connect (next_cb);

        /* Add size controls */
        size_actor = new Clutter.Actor ();
        size_actor.hide ();
        get_stage ().add_child (size_actor);

        /* Icon to select small pieces */
        var small_actor = new Clutter.CairoTexture (20, 20);
        size_actor.add_child (small_actor);
        small_actor.draw.connect (draw_size_icon_cb);
        small_actor.invalidate ();
        small_actor.set_position (-35, 0);
        small_actor.set_anchor_point (small_actor.width * 0.5f, small_actor.height * 0.5f);
        small_actor.reactive = true;
        small_actor.button_press_event.connect (small_cb);

        /* Icon to select medium pieces */
        var medium_actor = new Clutter.CairoTexture (30, 30);
        size_actor.add_child (medium_actor);
        medium_actor.draw.connect (draw_size_icon_cb);
        medium_actor.invalidate ();
        medium_actor.set_position (0, 0);
        medium_actor.set_anchor_point (medium_actor.width * 0.5f, medium_actor.height * 0.5f);
        medium_actor.reactive = true;
        medium_actor.button_press_event.connect (medium_cb);

        /* Icon to select large pieces */
        var large_actor = new Clutter.CairoTexture (40, 40);
        size_actor.add_child (large_actor);
        large_actor.draw.connect (draw_size_icon_cb);
        large_actor.invalidate ();
        large_actor.set_position (45, 0);
        large_actor.set_anchor_point (large_actor.width * 0.5f, large_actor.height * 0.5f);
        large_actor.reactive = true;
        large_actor.button_press_event.connect (large_cb);

        /* Start button */
        start_actor = new Clutter.Actor ();
        start_actor.set_background_color ({ 0xff, 0xff, 0xff, 0xff });
        start_actor.hide ();
        start_actor.reactive = true;
        start_actor.button_press_event.connect (start_cb);
        get_stage ().add_child (start_actor);

        var start_text = new Clutter.Text ();
        start_actor.add_child (start_text);
        // FIXME: Use GTK+ to get the font name
        start_text.font_name = "ubuntu 20";
        Clutter.Color text_color = { 0x2e, 0x34, 0x36, 0xff };
        start_text.color = text_color;
        /* Text on button that shuffles (i.e. starts) puzzle */
        start_text.text = _("Shuffle");
        start_text.set_anchor_point (start_text.width * 0.5f, start_text.height * 0.5f);

        start_actor.set_size (start_text.width + 15, start_text.height + 15);
        start_actor.set_anchor_point (start_actor.width * 0.5f, start_actor.height * 0.5f);
        start_text.set_position (start_actor.width * 0.5f, start_actor.height * 0.5f);

        /* Home button */
        home_actor = new Clutter.CairoTexture (41, 31);
        if (puzzle == null)
            home_actor.hide ();
        get_stage ().add_child (home_actor);
        home_actor.draw.connect (draw_home_icon_cb);
        home_actor.invalidate ();
        home_actor.reactive = true;
        home_actor.button_press_event.connect ( () => { home (); return true; } );

        /* Done message */
        done_actor = new Clutter.Actor ();
        get_stage ().add_child (done_actor);
        done_actor.hide ();
        title_text_actor = new Clutter.Text ();
        // FIXME: Use GTK+ to get the font name
        title_text_actor.font_name = "ubuntu 40";
        text_color = { 0xff, 0xff, 0xff, 0xff };
        title_text_actor.color = text_color;
        done_actor.add_child (title_text_actor);
        text_actor = new Clutter.Text ();
        text_actor.font_name = "ubuntu 20";
        text_actor.color = text_color;
        done_actor.add_child (text_actor);
    }

    private void size_allocate_cb ()
    {
        var stage_width = (int) get_stage ().width;
        var stage_height = (int) get_stage ().height;

        if (puzzle == null)
        {
            prev_actor.set_position (prev_actor.width * 0.75f, stage_height * 0.5f);
            next_actor.set_position (stage_width - next_actor.width * 0.75f, stage_height * 0.5f);
        }
        else
        {
            prev_actor.set_position (-stage_width * 0.5f, stage_height * 0.5f);
            next_actor.set_position (stage_width * 1.5f, stage_height * 0.5f);
        }
        size_actor.set_position (stage_width * 0.5f, stage_height * 0.125f);
        start_actor.set_position (stage_width * 0.5f, stage_height * 0.875f);
        home_actor.set_position (10, 10);
        done_actor.set_position (stage_width * 0.5f, stage_height * 0.15f);

        if (error_actor != null)
        {
            error_actor.set_position (stage_width * 0.5f, stage_height * 0.5f);
            return;
        }

        /* Resize puzzles */
        for (var i = 0; i < puzzles.length; i++)
        {
            var border = 0;
            var max_width = (int) ((stage_width - border * 3) * 0.5f);
            var max_height = (int) ((stage_height - border * 3) * 0.5f);
            var max_aspect = (float) max_width / max_height;
            var surface_aspect = (float) puzzles[i].pixbuf.width / puzzles[i].pixbuf.height;
            var width = max_width;
            var height = max_height;
            if (surface_aspect > max_aspect)
                height = (int) (width / surface_aspect);
            else
                width = (int) (height * surface_aspect);

            puzzles[i].puzzle.resize (stage_width, stage_height, width, height);

            if (puzzle == null)
            {
                float home_x, home_y;
                get_home_position (i, out home_x, out home_y);
                puzzles[i].set_position (home_x, home_y);
            }
            else if (puzzles[i] == puzzle)
            {
                float center_x, center_y;
                get_center_position (out center_x, out center_y);
                puzzles[i].set_position (center_x, center_y);
            }
            else
                puzzles[i].set_position (stage_width * hidden_x_positions [i], stage_height * hidden_y_positions[i]);
        }
    }

    private void get_home_position (int i, out float x, out float y)
    {
        var stage_width = (int) get_stage ().width;
        var stage_height = (int) get_stage ().height;

        var x_offset = (stage_width * 0.5f - puzzles[i].width) * 0.5f;
        var y_offset = (stage_height * 0.5f - puzzles[i].height) * 0.5f;
        if (i == 1 || i == 2)
            x_offset = -x_offset;
        if (i == 2 || i == 3)
            y_offset = -y_offset;
        x = stage_width * home_x_positions[i] + x_offset;
        y = stage_height * home_y_positions[i] + y_offset;
    }

    private void get_center_position (out float x, out float y)
    {
        var stage_width = (int) get_stage ().width;
        var stage_height = (int) get_stage ().height;

        x = stage_width * 0.5f;
        y = stage_height * 0.5f;

        /* Align puzzle on pixel boundary */
        if (puzzle.puzzle.width % 2 != stage_width % 2)
            x -= 0.5f;
        if (puzzle.puzzle.height % 2 != stage_height % 2)
            y -= 0.5f;
    }

    private bool draw_prev_icon_cb (Clutter.CairoTexture actor, Cairo.Context c)
    {
        var r = actor.width * 0.5f;
        c.arc (actor.width * 0.5f, actor.height * 0.5f, r, 0, 2 * Math.PI);
        var arrow_r = r * 0.75f;
        c.translate (actor.width * 0.5f, actor.height * 0.5f);
        c.move_to (-arrow_r, 0);
        c.line_to (arrow_r * Math.cos (Math.PI / 3), arrow_r * Math.sin (Math.PI / 3));
        c.line_to (arrow_r * Math.cos (-Math.PI / 3), arrow_r * Math.sin (-Math.PI / 3));
        c.set_source_rgb (1.0, 1.0, 1.0);
        c.fill ();
        return true;
    }

    private bool draw_next_icon_cb (Clutter.CairoTexture actor, Cairo.Context c)
    {
        var r = actor.width * 0.5f;
        c.arc (actor.width * 0.5f, actor.height * 0.5f, r, 0, 2 * Math.PI);
        var arrow_r = r * 0.75f;
        c.translate (actor.width * 0.5f, actor.height * 0.5f);
        c.move_to (arrow_r, 0);
        c.line_to (-arrow_r * Math.cos (-Math.PI / 3), arrow_r * Math.sin (-Math.PI / 3));
        c.line_to (-arrow_r * Math.cos (Math.PI / 3), arrow_r * Math.sin (Math.PI / 3));
        c.set_source_rgb (1.0, 1.0, 1.0);
        c.fill ();
        return true;
    }

    private bool draw_size_icon_cb (Clutter.CairoTexture actor, Cairo.Context c)
    {
        var plug_size = (int) (actor.width * 0.2);
        var size = (int) actor.width - plug_size;
        c.move_to (plug_size, plug_size);
        draw_puzzle_piece (c, size, size, plug_size, EdgeType.OUT, EdgeType.IN, EdgeType.IN, EdgeType.OUT);
        c.set_source_rgb (1.0, 1.0, 1.0);
        c.fill ();
        return true;
    }

    private bool draw_home_icon_cb (Clutter.CairoTexture actor, Cairo.Context c)
    {
        var width = (int) actor.width;
        var height = (int) actor.height;
        var w = (width - 1) * 0.5;
        var h = (height - 1) * 0.5;
        c.rectangle (0, 0, w, h);
        c.rectangle (w + 1, 0, w, h);
        c.rectangle (w + 1, h + 1, w, h);
        c.rectangle (0, h + 1, w, h);
        c.set_source_rgb (1.0, 1.0, 1.0);
        c.fill ();
        return true;
    }

    private void find_pictures (string directory)
    {
        Dir dir;
        try
        {
            dir = Dir.open (directory);
        }
        catch (FileError e)
        {
            return;
        }
        while (true)
        {
            var filename = dir.read_name ();
            if (filename == null)
                break;
            var full_filename = Path.build_filename (directory, filename);

            /* See what sort of file this is */
            Posix.Stat info;
            if (Posix.stat (full_filename, out info) != 0)
                continue;

            /* Recurse into directories */
            if (Posix.S_ISDIR (info.st_mode))
                find_pictures (full_filename);

            /* Records files in a random order */
            if (Posix.S_ISREG (info.st_mode))
                pictures.insert (full_filename, Random.int_range (0, (int32) pictures.length () + 1));
        }
    }

    private void create_puzzles (bool have_first = false)
    {
        if (!have_first)
            puzzles[0] = create_puzzle ();
        puzzles[1] = create_puzzle ();
        puzzles[2] = create_puzzle ();
        puzzles[3] = create_puzzle ();
    }

    private PuzzleActor? create_puzzle ()
    {
        /* Load next picture and scale to fit size */
        string filename;
        var pixbuf = load_next_picture (out filename);
        return add_puzzle (new Puzzle (), filename, pixbuf);
    }

    private Gdk.Pixbuf? load_next_picture (out string filename)
    {
        var n_tried = 0;
        while (true)
        {
            filename = pictures.nth_data (picture_index);
            picture_index++;
            if (picture_index == pictures.length ())
                picture_index = 0;

            /* Stop if we've looped all the way around and not found anything */
            if (n_tried == pictures.length ())
                return null;

            try
            {
                return new Gdk.Pixbuf.from_file (filename);
            }
            catch (Error e)
            {
            }
            n_tried++;
        }
    }

    private PuzzleActor add_puzzle (Puzzle puzzle, string filename, Gdk.Pixbuf pixbuf)
    {
        var actor = new PuzzleActor (puzzle, filename, pixbuf);
        actor.selected.connect (select_cb);
        actor.combined.connect (() => { home (false); } );
        actor.done.connect (done_cb);
        puzzle_layer.add_child (actor);
        return actor;
    }

    private void select_cb (PuzzleActor actor)
    {
        if (actor == puzzle)
            return;
        puzzle = actor;

        var stage_width = get_stage ().width;
        var stage_height = get_stage ().height;

        var timeline = new Clutter.Timeline (250);
        timeline.completed.connect (selected_cb);

        /* Slide the unused puzzles offscreen */
        for (var i = 0; i < puzzles.length; i++)
        {
            if (puzzles[i] != puzzle)
                puzzles[i].animate_with_timeline (Clutter.AnimationMode.EASE_IN_QUINT, timeline,
                                                  "x", stage_width * hidden_x_positions[i],
                                                  "y", stage_height * hidden_y_positions[i]);
        }

        /* Slide the controls offscreen */
        prev_actor.reactive = false;
        prev_actor.animate_with_timeline (Clutter.AnimationMode.EASE_IN_QUINT, timeline, "x", -stage_width * 0.5f);
        next_actor.reactive = false;
        next_actor.animate_with_timeline (Clutter.AnimationMode.EASE_IN_QUINT, timeline, "x", stage_width * 1.5f);

        /* Move the main piece to the middle */
        float center_x, center_y;
        get_center_position (out center_x, out center_y);
        puzzle.animate_with_timeline (Clutter.AnimationMode.EASE_IN_OUT_QUINT, timeline, "x", center_x, "y", center_y);

        size_actor.opacity = 0;
        size_actor.show ();
        size_actor.animate (Clutter.AnimationMode.EASE_OUT_SINE, 300, "opacity", 255);
        start_actor.opacity = 0;
        start_actor.show ();
        start_actor.animate (Clutter.AnimationMode.EASE_OUT_SINE, 300, "opacity", 255);
        home_actor.opacity = 0;
        home_actor.show ();
        home_actor.animate (Clutter.AnimationMode.EASE_IN_SINE, 150, "opacity", 255);
    }

    private void done_cb (PuzzleActor actor)
    {
        /* Text shown when the puzzle is completed */
        title_text_actor.text = _("Well done!");
        title_text_actor.set_anchor_point (title_text_actor.width * 0.5f, 0);

        var d = actor.puzzle.duration;
        var n_hours = d / 3600;
        var n_minutes = (d - n_hours * 3600) / 60;
        var n_seconds = d - n_hours * 3600 - n_minutes * 60;
        var label = "";
        if (d < 60)
            label = ngettext (/* Text shown when puzzle is completed in less than a minute */
                              "Puzzle solved in %d second", "Puzzle solved in %d seconds", n_seconds).printf (n_seconds);
        else
        {
            /* Special non-displayed string to indicate if this language supports showing the duration using string fragments.
             * Translate to "support-time-fragments=false" if this the time fragment strings can't be translated in your language.
             */
            if (_("support-time-fragments=true") == "support-time-fragments=true")
            {
                var hour_text = ngettext (/* Hour text fragment */ "%d hour", "%d hours", n_hours).printf (n_hours);
                var minute_text = ngettext (/* Minute text fragment */ "%d minute", "%d minutes", n_minutes).printf (n_minutes);
                var second_text = ngettext (/* Second text fragment */ "%d second", "%d seconds", n_seconds).printf (n_seconds);
                if (n_hours > 0)
                    /* Text to show duration of puzzle (only used if support-time-fragments is set to "true").
                     * %1$s is the text for the number of hours played.
                     * %2$s is the text for the number of minutes played.
                     * %3$s is the text for the number of seconds played.
                     */
                    label = _("Puzzle solved in %1$s, %2$s and %3$s").printf (hour_text, minute_text, second_text);
                else
                    /* Text to show duration of puzzle (only used if support-time-fragments is set to "true").
                     * %1$s is the text for the number of minutes played.
                     * %2$s is the text for the number of seconds played.
                     */
                    label = _("Puzzle solved in %1$s and %2$s").printf (minute_text, second_text);
            }
            else
            {
                if (n_hours > 0)
                    /* Text to show duration of puzzle (only used if support-time-fragments is set to "false").
                     * %1$d is the number of hours played.
                     * %2$d is the number of minutes played.
                     * %3$d is the number of seconds played.
                     */
                    label = _("Puzzle solved in %1$dh%2$dm%3$ds").printf (n_hours, n_minutes, n_seconds);
                else
                    /* Text to show duration of puzzle (only used if support-time-fragments is set to "false").
                     * %2$d is the number of minutes played.
                     * %3$d is the number of seconds played.
                     */
                    label = _("Puzzle solved in %1$dm%2$ds").printf (n_minutes, n_seconds);
            }
        }

        text_actor.text = label;
        text_actor.set_anchor_point (text_actor.width * 0.5f, 0);

        var height = title_text_actor.height * 1.25f + text_actor.height;
        title_text_actor.set_position (0, -height * 0.5f);
        text_actor.set_position (0, title_text_actor.height * 1.25f - height * 0.5f);

        done_actor.opacity = 0;
        done_actor.show ();
        done_actor.animate (Clutter.AnimationMode.EASE_OUT_SINE, 1000, "opacity", 255);
    }

    private bool prev_cb ()
    {
        if (sliding)
            return true;

        picture_index -= 8;
        while (picture_index < 0)
            picture_index += (int) pictures.length ();
        slide (-1.0f);

        return true;
    }

    private bool next_cb ()
    {
        if (sliding)
            return true;

        slide (1.0f);

        return true;
    }

    private void slide (float offset)
    {
        if (pictures.length () == 0)
            return;

        sliding = true;

        var stage_width = (int) get_stage ().width;
        var stage_height = (int) get_stage ().height;

        /* Slide the unused puzzles offscreen */
        var timeline = new PuzzleTimeline ();
        timeline.completed.connect (() => { sliding = false; });
        timeline.duration = 500;
        for (var i = 0; i < puzzles.length; i++)
        {
            puzzles[i].reactive = false;
            timeline.puzzles[i] = puzzles[i];
            puzzles[i].animate_with_timeline (Clutter.AnimationMode.EASE_IN_OUT_SINE, timeline, "x", stage_width * (home_x_positions[i] - offset));
        }

        /* Bring in new puzzles */
        create_puzzles ();
        for (var i = 0; i < puzzles.length; i++)
        {
            // FIXME: Copied
            var border = 0;
            var max_width = (int) ((stage_width - border * 3) * 0.5f);
            var max_height = (int) ((stage_height - border * 3) * 0.5f);
            var max_aspect = (float) max_width / max_height;
            var surface_aspect = (float) puzzles[i].pixbuf.width / puzzles[i].pixbuf.height;
            var width = max_width;
            var height = max_height;
            if (surface_aspect > max_aspect)
                height = (int) (width / surface_aspect);
            else
                width = (int) (height * surface_aspect);

            puzzles[i].puzzle.resize (stage_width, stage_height, width, height);

            float home_x, home_y;
            get_home_position (i, out home_x, out home_y);
            puzzles[i].set_position (home_x + stage_width * offset, home_y);
            puzzles[i].animate_with_timeline (Clutter.AnimationMode.EASE_IN_OUT_SINE, timeline, "x", home_x);
        }
    }

    private bool small_cb ()
    {
        n_pieces = 60;
        puzzle.set_n_pieces (n_pieces);
        return true;
    }

    private bool medium_cb ()
    {
        n_pieces = 30;
        puzzle.set_n_pieces (n_pieces);
        return true;
    }

    private bool large_cb ()
    {
        n_pieces = 15;
        puzzle.set_n_pieces (n_pieces);
        return true;
    }

    private bool start_cb ()
    {
        size_actor.hide ();
        size_actor.animate (Clutter.AnimationMode.EASE_IN_SINE, 150, "opacity", 0);
        start_actor.hide ();
        start_actor.animate (Clutter.AnimationMode.EASE_IN_SINE, 150, "opacity", 0);
        puzzle.start ();
        return true;
    }

    public void home (bool check_saved = true)
    {
        if (check_saved && puzzle != null && puzzle.puzzle.is_shuffled && !puzzle.puzzle.is_solved)
        {
            confirm_home ();
            return;
        }

        size_actor.hide ();
        size_actor.animate (Clutter.AnimationMode.EASE_IN_SINE, 150, "opacity", 0);
        start_actor.hide ();
        start_actor.animate (Clutter.AnimationMode.EASE_IN_SINE, 150, "opacity", 0);
        home_actor.hide ();
        home_actor.animate (Clutter.AnimationMode.EASE_IN_SINE, 150, "opacity", 0);
        done_actor.animate (Clutter.AnimationMode.EASE_IN_SINE, 150, "opacity", 0);

        var stage_width = get_stage ().width;

        var timeline = new Clutter.Timeline (250);
        if (puzzle != null && !puzzle.is_combined)
            puzzle.combine ();
        else
        {
            puzzle = null;
            for (var i = 0; i < puzzles.length; i++)
            {
                float home_x, home_y;
                get_home_position (i, out home_x, out home_y);
                puzzles[i].animate_with_timeline (Clutter.AnimationMode.EASE_OUT_QUINT, timeline, "x", home_x, "y", home_y);
            }
        }
        prev_actor.reactive = true;
        prev_actor.animate_with_timeline (Clutter.AnimationMode.EASE_OUT_QUINT, timeline, "x", prev_actor.width * 0.75f);
        next_actor.reactive = true;
        next_actor.animate_with_timeline (Clutter.AnimationMode.EASE_OUT_QUINT, timeline, "x", stage_width - next_actor.width * 0.75f);
    }

    private void selected_cb ()
    {
        /* Split the puzzles */
        puzzle.set_n_pieces (n_pieces);
    }

    private bool motion_cb (Clutter.Actor actor, Clutter.MotionEvent event)
    {
        if (puzzle == null)
            return false;

        puzzle.motion (event);
        return true;
    }

    private bool button_release_cb (Clutter.Actor actor, Clutter.ButtonEvent event)
    {
        if (puzzle == null)
            return false;

        if (event.button == 1)
            puzzle.deselect ();

        return true;
    }
}

private class PuzzleTimeline : Clutter.Timeline
{
    public PuzzleActor puzzles[4];

    protected override void completed ()
    {
        foreach (var puzzle in puzzles)
            puzzle.destroy ();
    }
}

public class PuzzleActor : Clutter.Actor
{
    public Puzzle puzzle;
    public string filename;
    public Gdk.Pixbuf pixbuf;
    private Clutter.CairoTexture texture;
    public Clutter.Timeline explode_timeline;
    public Clutter.Timeline implode_timeline;
    public Clutter.Timeline combine_timeline;
    private bool exploded = false;
    private int n_pieces;
    public bool is_combined;
    public Clutter.Actor pieces;
    public PuzzlePieceActor? selected_piece = null;
    public float select_x;
    public float select_y;
    public float start_x;
    public float start_y;
    private HashTable<PuzzlePiece, PuzzlePieceActor> piece_actors;

    public signal void selected ();
    public signal void combined ();
    public signal void done ();

    public PuzzleActor (Puzzle puzzle, string filename, Gdk.Pixbuf pixbuf)
    {
        piece_actors = new HashTable<PuzzlePiece, PuzzlePieceActor> (direct_hash, direct_equal);

        this.puzzle = puzzle;
        this.filename = filename;
        this.pixbuf = pixbuf;

        puzzle.resized.connect (resized_cb);
        puzzle.splitted.connect (splitted_cb);
        puzzle.shuffled.connect (shuffled_cb);
        puzzle.group_merged.connect (group_merged_cb);

        texture = new Clutter.CairoTexture (0, 0);
        texture.reactive = true;
        texture.button_press_event.connect (button_press_event_cb);
        texture.draw.connect (draw_cb);
        add_child (texture);

        pieces = new Clutter.Actor ();
        add_child (pieces);

        if (puzzle.groups.length () > 0)
        {
            texture.opacity = 64;
            create_pieces ();
        }
    }

    public void set_n_pieces (int n_pieces)
    {
        this.n_pieces = n_pieces;
        if (exploded)
        {
            implode_timeline = new Clutter.Timeline (150);
            implode_timeline.completed.connect (implode_completed_cb);
            puzzle.implode ();
        }
        else
            implode_completed_cb ();
    }

    public void start ()
    {
        puzzle.shuffle ();
    }

    private bool button_press_event_cb (Clutter.ButtonEvent event)
    {
        selected ();
        return false;
    }

    private void resized_cb ()
    {
        texture.set_size (puzzle.width, puzzle.height);
        texture.set_anchor_point (texture.width * 0.5f, texture.height * 0.5f);
        texture.set_surface_size (puzzle.width, puzzle.height);
        texture.invalidate ();
    }

    private void splitted_cb ()
    {
        texture.opacity = 0;
        create_pieces ();
    }

    private void create_pieces ()
    {
        piece_actors.remove_all ();
        pieces.remove_all_children ();
        foreach (var group in puzzle.groups)
        {
            foreach (var piece in group.pieces)
            {
                var actor = new PuzzlePieceActor (this, piece);
                piece_actors.insert (piece, actor);
                actor.target.reactive = true;
                actor.target.button_press_event.connect (button_press_cb);
                pieces.add_child (actor);
            }
        }
    }

    private void shuffled_cb ()
    {
        texture.animate (Clutter.AnimationMode.EASE_IN_OUT_SINE, 100, "opacity", 64);
    }

    private void group_merged_cb (PuzzleGroup updated_group, PuzzleGroup removed_group)
    {
        /* Redraw merged pieces so their edges match */
        foreach (var p in updated_group.pieces)
        {
            var a = piece_actors.lookup (p);
            a.texture.invalidate ();
        }
    }

    private bool button_press_cb (Clutter.Actor actor, Clutter.ButtonEvent event)
    {
        var piece = actor.get_parent () as PuzzlePieceActor;

        if (!puzzle.is_shuffled || puzzle.is_solved)
            return false;

        // FIXME: Only if started

        /* Raise group to the top */
        // FIXME: Also raise merged groups
        foreach (var p in piece.piece.group.pieces)
        {
            var a = piece_actors.lookup (p);
            pieces.set_child_above_sibling (a, null);
        }

        /* Select this piece for moving */
        if (event.button == 1)
        {
            selected_piece = piece;
            select_x = event.x;
            select_y = event.y;
            start_x = piece.x;
            start_y = piece.y;
        }

        /* Rotate on right click */
        if (event.button == 3)
            piece.piece.rotate ();

        return true;
    }

    public void motion (Clutter.MotionEvent event)
    {
        if (selected_piece == null)
            return;

        var x_step = (int) (event.x - select_x + 0.5);
        var y_step = (int) (event.y - select_y + 0.5);
        selected_piece.piece.move (start_x + x_step, start_y + y_step, MoveType.PLACE);
    }

    public void deselect ()
    {
        if (selected_piece == null)
            return;

        puzzle.join (selected_piece.piece.group);
        selected_piece = null;

        if (puzzle.is_solved)
        {
            combine_timeline = new Clutter.Timeline (150);
            puzzle.combine ();
            done ();
        }
    }

    private void implode_completed_cb ()
    {
        is_combined = false;
        puzzle.split (n_pieces);
        explode_timeline = new Clutter.Timeline (150);
        exploded = true;
        puzzle.explode ();
    }

    public void combine ()
    {
        combine_timeline = new Clutter.Timeline (150);
        combine_timeline.completed.connect (combine_completed_cb);
        puzzle.combine ();
    }

    private void combine_completed_cb ()
    {
        pieces.remove_all_children ();
        texture.opacity = 255;
        is_combined = true;
        combined ();
    }

    private Cairo.Pattern? puzzle_pattern = null;
    public Cairo.Pattern get_puzzle_pattern (Cairo.Context c)
    {
        if (puzzle_pattern != null)
            return puzzle_pattern;

        var surface = new Cairo.Surface.similar (c.get_target (), Cairo.Content.COLOR, pixbuf.width, pixbuf.height);
        var c2 = new Cairo.Context (surface);
        Gdk.cairo_set_source_pixbuf (c2, pixbuf, 0, 0);
        c2.paint ();
        puzzle_pattern = new Cairo.Pattern.for_surface (surface);
        return puzzle_pattern;
    }

    private bool draw_cb (Cairo.Context c)
    {
        c.set_source (get_puzzle_pattern (c));
        var pattern = c.get_source ();
        pattern.set_filter (Cairo.Filter.BEST);
        var matrix = Cairo.Matrix.identity ();
        matrix.scale ((double) pixbuf.width / puzzle.width, (double) pixbuf.height / puzzle.height);
        pattern.set_matrix (matrix);
        c.paint ();
        return true;
    }
}

private void draw_puzzle_piece (Cairo.Context c,
                                float width, float height, float plug_size,
                                EdgeType n_edge_type, EdgeType e_edge_type, EdgeType s_edge_type, EdgeType w_edge_type)
{
    draw_puzzle_edge (c, width, 0, plug_size, 0, n_edge_type);
    draw_puzzle_edge (c, 0, height, 0, plug_size, e_edge_type);
    draw_puzzle_edge (c, -width, 0, -plug_size, 0, s_edge_type);
    draw_puzzle_edge (c, 0, -height, 0, -plug_size, w_edge_type);
}

private void draw_puzzle_edge (Cairo.Context c, float edge_x, float edge_y, float plug_x, float plug_y, EdgeType type)
{
    /* Flat edges are easy... */
    if (type == EdgeType.FLAT)
    {
        c.rel_line_to (edge_x, edge_y);
        return;
    }

    /*
     * Plugs/sockets are like this:
     *
     *          d
     *        ,-o-.
     *       /     \
     *    c o       o e
     *       \     /
     * o-----o     o-----o
     * a     b     f     g
     *
     * We have the vector a-g (edge_x, edge_y) and c-e (plug_x, plug_y)
     *
     * c, d, and e are the midpoints of the plug square.
     *
     * Points b-c, c-d, d-e, e-f are cubic bezier curves with control points set to approximate circles
     */

    /* Get perpindicular vector to (plug_x, plug_y) */
    var plug_xp = plug_y;
    var plug_yp = -plug_x;

    /* Sockets are just the inverse of plugs */
    if (type == EdgeType.IN)
    {
        plug_xp = -plug_xp;
        plug_yp = -plug_yp;
    }

    /* b-f is 80% of the width of the plug */
    var bfx = plug_x * 0.8f;
    var bfy = plug_y * 0.8f;

    /* c-b along a-g vector */
    var cbx = (plug_x - bfx) * 0.5f;
    var cby = (plug_y - bfy) * 0.5f;

    /* Vectors from previous point to new point */
    var bx = (edge_x - bfx) * 0.5f;
    var by = (edge_y - bfy) * 0.5f;
    var cx = -cbx + plug_xp * 0.5f;
    var cy = -cby + plug_yp * 0.5f;
    var dx = (plug_x + plug_xp) * 0.5f;
    var dy = (plug_y + plug_yp) * 0.5f;
    var ex = (plug_x - plug_xp) * 0.5f;
    var ey = (plug_y - plug_yp) * 0.5f;
    var fx = -cbx - plug_xp * 0.5f;
    var fy = -cby - plug_yp * 0.5f;
    var gx = bx;
    var gy = by;

    /* Control points for circle appropimation */
    var z = 0.55228475f;
    var zx = plug_x * 0.5f * z;
    var zy = plug_y * 0.5f * z;
    var zxp = plug_xp * 0.5f * z;
    var zyp = plug_yp * 0.5f * z;

    /* Draw the lines */
    c.rel_line_to (bx, by);
    c.rel_curve_to (zxp, zyp, cx - zxp, cy - zyp, cx, cy);
    c.rel_curve_to (zxp, zyp, dx - zx, dy - zy, dx, dy);
    c.rel_curve_to (zx, zy, ex + zxp, ey + zyp, ex, ey);
    c.rel_curve_to (-zxp, -zyp, fx + zxp, fy + zyp, fx, fy);
    c.rel_line_to (gx, gy);
}

public class PuzzlePieceActor : Clutter.Actor
{
    public PuzzleActor parent;
    public PuzzlePiece piece;
    public Clutter.CairoTexture texture;
    public Clutter.Actor target;

    public PuzzlePieceActor (PuzzleActor parent, PuzzlePiece piece)
    {
        this.parent = parent;
        this.piece = piece;

        texture = new Clutter.CairoTexture (0, 0);
        texture.draw.connect (draw_cb);
        add_child (texture);

        /* Create a reactive selection area for the core of the piece */
        target = new Clutter.Actor ();
        add_child (target);

        piece_resized_cb ();
        piece.resized.connect (piece_resized_cb);
        piece.moved.connect (piece_moved_cb);
    }

    private void piece_moved_cb (MoveType type)
    {
        switch (type)
        {
        case MoveType.SNAP:
            animate (Clutter.AnimationMode.EASE_IN_OUT_QUINT, 100, "x", piece.x, "y", piece.y);
            break;
        case MoveType.SHUFFLE:
            animate (Clutter.AnimationMode.EASE_IN_OUT_QUINT, 600, "x", piece.x, "y", piece.y);
            break;
        case MoveType.EXPLODE:
            animate_with_timeline (Clutter.AnimationMode.EASE_IN_OUT_QUINT, parent.explode_timeline, "x", piece.x, "y", piece.y);
            break;
        case MoveType.IMPLODE:
            animate_with_timeline (Clutter.AnimationMode.EASE_IN_OUT_QUINT, parent.implode_timeline, "x", piece.x, "y", piece.y);
            break;
        case MoveType.COMBINE:
            animate_with_timeline (Clutter.AnimationMode.EASE_IN_OUT_QUINT, parent.combine_timeline, "x", piece.x, "y", piece.y);
            break;
        case MoveType.PLACE:
        default:
            set_position (piece.x, piece.y);
            break;
        }
    }

    private void piece_resized_cb ()
    {
        set_position (piece.x, piece.y);
        set_size (piece.width + piece.plug_size * 2, piece.height + piece.plug_size * 2);
        texture.set_surface_size ((int) width, (int) height);
        texture.set_size (piece.width + piece.plug_size * 2, piece.height + piece.plug_size * 2);
        texture.set_anchor_point (width / 2, height / 2);
        texture.invalidate ();
        target.set_size (piece.width, piece.height);
        target.set_anchor_point (target.width / 2, target.height / 2);
    }

    private bool draw_cb (Cairo.Context c)
    {
        /* Draw flat edges to adjacent pieces */
        var n_edge_type = piece.n_edge_type;
        var e_edge_type = piece.e_edge_type;
        var s_edge_type = piece.s_edge_type;
        var w_edge_type = piece.w_edge_type;
        foreach (var p in piece.group.pieces)
        {
            if (p.row == piece.row)
            {
                if (p.column == piece.column - 1)
                    w_edge_type = EdgeType.FLAT;
                else if (p.column == piece.column + 1)
                    e_edge_type = EdgeType.FLAT;
            }
            if (p.column == piece.column)
            {
                if (p.row == piece.row - 1)
                    n_edge_type = EdgeType.FLAT;
                else if (p.row == piece.row + 1)
                    s_edge_type = EdgeType.FLAT;
            }
        }

        /* Draw the piece */
        c.set_source (parent.get_puzzle_pattern (c));
        var pattern = c.get_source ();
        pattern.set_filter (Cairo.Filter.BEST);
        var matrix = Cairo.Matrix.identity ();
        matrix.scale ((double) parent.pixbuf.width / parent.puzzle.width, (double) parent.pixbuf.height / parent.puzzle.height);
        matrix.translate (-(piece.plug_size - piece.x_offset), -(piece.plug_size - piece.y_offset));
        pattern.set_matrix (matrix);
        c.set_source (pattern);
        c.move_to (piece.plug_size, piece.plug_size);
        draw_puzzle_piece (c, piece.width, piece.height, piece.plug_size, piece.n_edge_type, piece.e_edge_type, piece.s_edge_type, piece.w_edge_type);
        c.fill ();
        return true;
    }
}

public errordomain PuzzleError
{
    LOAD_ERROR
}

public class Puzzle
{
    /* How long this puzzle has been solved in */
    private Timer timer;

    /* Number of seconds elapsed before timer was started */
    private int existing_duration = 0;

    /* Number of seconds this puzzle has been played for */
    public int duration { get { return existing_duration + (int) (timer.elapsed () + 0.5d); } }

    private int table_width;
    private int table_height;

    /* Width of the puzzle */
    public int width;

    /* Height of the puzzle in pixels */
    public int height;

    /* Number of rows */
    public int n_rows { get { return pieces == null ? 0 : pieces.length[1]; } }

    /* Number of columns */
    public int n_columns { get { return pieces == null ? 0 : pieces.length[0]; } }

    /* Size of an internal piece */
    public int piece_size { get { return n_rows == 0 || n_columns == 0 ? 0 : int.min (width / n_columns, height / n_rows); } }

    /* Groups of pieces */
    public List<PuzzleGroup> groups;

    /* Pieces */
    public PuzzlePiece[,] pieces;

    public bool is_shuffled = false;

    public bool is_solved { get { return groups.length () == 1; } }

    public signal void resized ();
    public signal void splitted ();
    public signal void shuffled ();
    public signal void group_merged (PuzzleGroup updated_group, PuzzleGroup removed_group);
    public signal void solved ();

    public Puzzle ()
    {
        timer = new Timer ();
    }

    public Puzzle.from_key_file (KeyFile file) throws Error
    {
        timer = new Timer ();

        existing_duration = file.get_integer ("puzzle", "duration");
        table_width = file.get_integer ("puzzle", "table-width");
        table_height = file.get_integer ("puzzle", "table-height");
        width = file.get_integer ("puzzle", "width");
        height = file.get_integer ("puzzle", "height");
        var n_rows = file.get_integer ("puzzle", "n-rows");
        var n_columns = file.get_integer ("puzzle", "n-columns");
        if (width <= 0 || height <= 0 || table_width <= width || table_height < height || n_rows <= 0 || n_columns <= 0)
            throw new PuzzleError.LOAD_ERROR ("Corrupt puzzle data");

        var lines = file.get_string ("puzzle", "pieces").split (":");
        if (lines.length < n_rows * n_columns * 6)
            throw new PuzzleError.LOAD_ERROR ("Corrupt puzzle data");

        groups = new List<PuzzleGroup> ();
        var group_table = new PuzzleGroup[n_rows * n_columns];
        pieces = new PuzzlePiece[n_columns, n_rows];
        var row = 0;
        var column = 0;
        var n = 0;
        for (var i = 0; i < n_rows * n_columns; i++)
        {
            var group_index = int.parse (lines[n++]);
            if (group_index < 0 || group_index >= group_table.length)
                throw new PuzzleError.LOAD_ERROR ("Corrupt puzzle data");
            var group = group_table[group_index];
            if (group == null)
            {
                group_table[group_index] = group = new PuzzleGroup (this);
                groups.append (group);
            }
            var e_edge_type = edge_type_parse (lines[n++]);
            var s_edge_type = edge_type_parse (lines[n++]);
            var x = (float) double.parse (lines[n++]);
            var y = (float) double.parse (lines[n++]);
            n++; // FIXME: rotation

            int x_offset, y_offset, piece_width, piece_height;
            calculate_piece_size (row, column, out x_offset, out y_offset, out piece_width, out piece_height);
            var piece_x = -width * 0.5f + x_offset + piece_width * 0.5f;
            var piece_y = -height * 0.5f + y_offset + piece_height * 0.5f;

            var piece = new PuzzlePiece (group, piece_width, piece_height, piece_size / 4, x, y);
            group.pieces.append (piece);
            pieces[column, row] = piece;
            piece.orig_x = piece_x;
            piece.orig_y = piece_y;
            piece.row = row;
            piece.column = column;
            piece.x_offset = x_offset;
            piece.y_offset = y_offset;
            piece.e_edge_type = e_edge_type;
            piece.s_edge_type = s_edge_type;
            if (row != 0)
                piece.n_edge_type = opposite_edge_type (pieces[column, row - 1].s_edge_type);
            if (column != 0)
                piece.w_edge_type = opposite_edge_type (pieces[column - 1, row].e_edge_type);

            column++;
            if (column == n_columns)
            {
                column = 0;
                row++;
            }
        }

        is_shuffled = true;
    }

    public void resize (int table_width, int table_height, int width, int height)
    {
        var old_table_width = this.table_width;
        var old_table_height = this.table_height;

        this.table_width = table_width;
        this.table_height = table_height;
        this.width = width;
        this.height = height;

        /* Resize and move pieces */
        if (pieces != null)
        {
            foreach (var group in groups)
            {
                var first_piece = group.pieces.nth_data (0);
                foreach (var piece in group.pieces)
                {
                    int x_offset, y_offset, piece_width, piece_height;
                    calculate_piece_size (piece.row, piece.column, out x_offset, out y_offset, out piece_width, out piece_height);
                    piece.x_offset = x_offset;
                    piece.y_offset = y_offset;
                    piece.orig_x = -width * 0.5f + x_offset + piece_width * 0.5f;
                    piece.orig_y = -height * 0.5f + y_offset + piece_height * 0.5f;

                    /* Scale the location of the new piece or place it beside the already moved piece */
                    if (piece == first_piece)
                    {
                        var new_x = piece.x * table_width / old_table_width;
                        var new_y = piece.y * table_height / old_table_height;
                        piece.x = piece.orig_x + (int) (new_x - piece.orig_x);
                        piece.y = piece.orig_y + (int) (new_y - piece.orig_y);
                    }
                    else
                    {
                        var dx = first_piece.x - first_piece.orig_x;
                        var dy = first_piece.y - first_piece.orig_y;
                        piece.x = piece.orig_x + dx;
                        piece.y = piece.orig_y + dy;
                    }

                    piece.resize (piece_width, piece_height, piece_size / 4);
                }
            }
        }

        resized ();
    }

    public void split (int n_pieces)
    {
        /* Work out the piece size that gets us nearest to the requested number of pieces */
        /* Divide the area by the number of pieces to get the area of one piece,
         * work out the size of an edge of that square piece and round down */
        var piece_size = (int) Math.sqrt (width * height / (float) n_pieces);
        var plug_size = piece_size / 4;
        var n_columns = width / piece_size;
        var n_rows = height / piece_size;

        groups = new List<PuzzleGroup> ();
        pieces = new PuzzlePiece[n_columns, n_rows];
        for (var x = 0; x < n_columns; x++)
        {
            for (var y = 0; y < n_rows; y++)
            {
                var group = new PuzzleGroup (this);
                groups.append (group);

                int x_offset, y_offset, piece_width, piece_height;
                calculate_piece_size (y, x, out x_offset, out y_offset, out piece_width, out piece_height);

                var piece_x = -width * 0.5f + x_offset + piece_width * 0.5f;
                var piece_y = -height * 0.5f + y_offset + piece_height * 0.5f;
                var piece = new PuzzlePiece (group, piece_width, piece_height, plug_size, piece_x, piece_y);
                piece.row = y;
                piece.column = x;
                piece.x_offset = x_offset;
                piece.y_offset = y_offset;

                pieces[x, y] = piece;
                if (x < n_columns - 1)
                    piece.e_edge_type = random_edge_type ();
                if (y < n_rows - 1)
                    piece.s_edge_type = random_edge_type ();
                if (x > 0)
                    piece.w_edge_type = opposite_edge_type (pieces[x - 1, y].e_edge_type);
                if (y > 0)
                    piece.n_edge_type = opposite_edge_type (pieces[x, y - 1].s_edge_type);

                group.pieces.append (piece);
            }
        }

        splitted ();
    }

    private void calculate_piece_size (int row, int column, out int x_offset, out int y_offset, out int piece_width, out int piece_height)
    {
        var left_extra = (width - (n_columns * piece_size)) / 2;
        var right_extra = width - (n_columns * piece_size) - left_extra;
        var top_extra = (height - (n_rows * piece_size)) / 2;
        var bottom_extra = height - (n_rows * piece_size) - top_extra;
    
        x_offset = column * piece_size;
        piece_width = piece_size;
        if (column == 0)
            piece_width += left_extra;
        else
            x_offset += left_extra;
        if (column == n_columns - 1)
            piece_width += right_extra;
        y_offset = row * piece_size;
        piece_height = piece_size;
        if (row == 0)
            piece_height += top_extra;
        else
            y_offset += top_extra;
        if (row == n_rows - 1)
            piece_height += bottom_extra;
    }

    public void shuffle ()
    {
        var new_groups = new List<PuzzleGroup> ();
        foreach (var group in groups)
        {
            foreach (var piece in group.pieces)
            {
                var w = (table_width - piece.width) * 0.5f;
                var h = (table_height - piece.height) * 0.5f;
                var x = (float) Random.double_range (-w, w);
                var y = (float) Random.double_range (-h, h);
                //var angle = 90d * rotation;
                //if (Random.boolean ())
                //    angle = angle - 360d;

                piece.group = new PuzzleGroup (this);
                piece.group.pieces.append (piece);
                new_groups.append (piece.group);

                var dx = (int) (x - piece.orig_x);
                var dy = (int) (y - piece.orig_y);
                piece.move (piece.orig_x + dx, piece.orig_y + dy, MoveType.SHUFFLE);
            }
        }
        groups = (owned) new_groups;

        /* Start the timer */
        timer.start ();

        is_shuffled = true;
        shuffled ();
    }

    public bool join (PuzzleGroup group)
    {
        var is_joined = false;
        while (true)
        {
            var new_group = join_group (group);
            if (new_group == null)
                return is_joined;
            is_joined = true;
            group = new_group;

            if (groups.length () == 1)
            {
                timer.stop ();
                solved ();
            }
        }
    }

    private PuzzleGroup? join_group (PuzzleGroup group)
    {
        foreach (var piece in group.pieces)
        {
            foreach (var g in groups)
            {
                if (g == group)
                    continue;

                foreach (var p in g.pieces)
                {
                    float dx, dy;
                    if (pieces_match (p, piece, out dx, out dy))
                    {
                        /* Move to final location */
                        if (dx != 0 || dy != 0)
                            group.move (dx, dy, MoveType.SNAP);

                        foreach (var gp in group.pieces)
                        {
                            p.group.pieces.append (gp);
                            gp.group = p.group;
                        }
                        groups.remove (group);

                        group_merged (g, group);
                        return p.group;
                    }
                }
            }
        }

        return null;
    }

    private bool pieces_match (PuzzlePiece p, PuzzlePiece piece, out float dx, out float dy)
    {
        var max_distance = 10.0f;

        /* Left edge */
        dx = p.x + (p.width + piece.width) * 0.5f - piece.x;
        dy = p.y - piece.y;
        var d = Math.sqrtf (dx * dx + dy * dy);
        if (d <= max_distance && p.row == piece.row && p.column == piece.column - 1)
            return true;

        /* Right edge */
        dx = p.x - (p.width + piece.width) * 0.5f - piece.x;
        dy = p.y - piece.y;
        d = Math.sqrtf (dx * dx + dy * dy);
        if (d <= max_distance && p.row == piece.row && p.column == piece.column + 1)
            return true;

        /* Top edge */
        dx = p.x - piece.x;
        dy = p.y + (p.height + piece.height) * 0.5f - piece.y;
        d = Math.sqrtf (dx * dx + dy * dy);
        if (d <= max_distance && p.row == piece.row - 1 && p.column == piece.column)
            return true;

        /* Bottom edge */
        dx = p.x - piece.x;
        dy = p.y - (p.height + piece.height) * 0.5f - piece.y;
        d = Math.sqrtf (dx * dx + dy * dy);
        if (d <= max_distance && p.row == piece.row + 1 && p.column == piece.column)
            return true;

        return false;
    }

    public void explode ()
    {
        var plug_size = piece_size / 4;
        var exploded_width = (float) width + (n_columns - 1) * plug_size * 2;
        var exploded_height = (float) height + (n_rows - 1) * plug_size * 2;
        foreach (var group in groups)
            foreach (var piece in group.pieces)
            {
                var dx = (int) (piece.orig_x * exploded_width / width - piece.orig_x);
                var dy = (int) (piece.orig_y * exploded_height / height - piece.orig_y);
                piece.move (piece.orig_x + dx, piece.orig_y + dy, MoveType.EXPLODE);
            }
    }

    public void implode ()
    {
        foreach (var group in groups)
            foreach (var piece in group.pieces)
                piece.move (piece.orig_x, piece.orig_y, MoveType.IMPLODE);
    }

    public void combine ()
    {
        foreach (var group in groups)
            foreach (var piece in group.pieces)
                piece.move (piece.orig_x, piece.orig_y, MoveType.COMBINE);
        is_shuffled = false;
    }

    public void save (KeyFile file)
    {
        file.set_integer ("puzzle", "duration", duration);
        file.set_integer ("puzzle", "table-width", table_width);
        file.set_integer ("puzzle", "table-height", table_height);
        file.set_integer ("puzzle", "width", width);
        file.set_integer ("puzzle", "height", height);
        file.set_integer ("puzzle", "n-rows", n_rows);
        file.set_integer ("puzzle", "n-columns", n_columns);

        /* Puzzle pieces */
        var data = "";
        for (var row = 0; row < n_rows; row++)
        {
            for (var column = 0; column < n_columns; column++)
            {
                var piece = pieces[column, row];
                if (data != "")
                    data += ":";
                data += "%d:%s:%s:%s:%s:%.0f".printf (groups.index (piece.group), edge_names[piece.e_edge_type], edge_names[piece.s_edge_type], ((double) piece.x).to_string (), ((double) piece.y).to_string (), 0);
            }
        }
        file.set_string ("puzzle", "pieces", data);
    }
}

public enum MoveType
{
    PLACE,
    SHUFFLE,
    SNAP,
    EXPLODE,
    IMPLODE,
    COMBINE
}

public class PuzzleGroup
{
    public Puzzle puzzle;
    public List<PuzzlePiece> pieces;

    public signal void moved (MoveType type);
    public signal void rotated ();

    public PuzzleGroup (Puzzle puzzle)
    {
        this.puzzle = puzzle;
        pieces = new List<PuzzlePiece> ();
    }

    public void move (float x_step, float y_step, MoveType type)
    {
        foreach (var piece in pieces)
        {
            piece.x += x_step;
            piece.y += y_step;
            piece.moved (type);
        }
        moved (type);
    }

    public void rotate ()
    {
        foreach (var piece in pieces)
            piece.rotated ();
        rotated ();
    }
}

public enum EdgeType
{
    FLAT,
    OUT,
    IN
}

public const string[] edge_names = { "flat", "out", "in" };

private EdgeType edge_type_parse (string name)
{
    EdgeType type = EdgeType.FLAT;
    foreach (var n in edge_names)
    {
        if (n == name)
            break;
        type += 1;
    }

    return type;
}

private EdgeType random_edge_type ()
{
    if (Random.boolean ())
        return EdgeType.IN;
    else
        return EdgeType.OUT;
}

private EdgeType opposite_edge_type (EdgeType type)
{
    switch (type)
    {
    case EdgeType.OUT:
        return EdgeType.IN;
    case EdgeType.IN:
        return EdgeType.OUT;
    default:
        return type;
    }
}

public class PuzzlePiece
{
    /* Group this piece is in */
    public PuzzleGroup group;

    /* Row location in solved puzzle */
    public int row;

    /* Column location in solved puzzle */
    public int column;

    /* Width of piece in pixels */
    public int width;

    /* Height of piece in pixels */
    public int height;

    /* Radius of plug/socket in pixels */
    public int plug_size;

    /* Pixel location from the top edge of the solved puzzle */
    public int x_offset;

    /* Pixel location from the left edge of the solved puzzle */
    public int y_offset;

    /* Location in solved puzzle */
    public float orig_x;
    public float orig_y;

    /* Current location */
    public float x;
    public float y;

    /* Edge pattern */
    public EdgeType n_edge_type;
    public EdgeType e_edge_type;
    public EdgeType s_edge_type;
    public EdgeType w_edge_type;

    public signal void moved (MoveType type);
    public signal void rotated ();
    public signal void resized ();

    public PuzzlePiece (PuzzleGroup group, int width, int height, int plug_size, float x, float y)
    {
        this.group = group;
        this.width = width;
        this.plug_size = plug_size;
        this.height = height;
        orig_x = x;
        orig_y = y;
        this.x = x;
        this.y = y;
        n_edge_type = EdgeType.FLAT;
        e_edge_type = EdgeType.FLAT;
        s_edge_type = EdgeType.FLAT;
        w_edge_type = EdgeType.FLAT;
    }

    public void move (float x, float y, MoveType type)
    {
        this.group.move (x - this.x, y - this.y, type);
    }

    public void rotate ()
    {
        this.group.rotate ();
    }

    public void resize (int width, int height, int plug_size)
    {
        this.width = width;
        this.height = height;
        this.plug_size = plug_size;
        resized ();
    }
}
