/*
 *      nm.c
 *      
 *      Copyright 2012 Alex <alex@linuxonly.ru>
 *      
 *      This program is free software: you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation; either version 3 of the License, or
 *      (at your option) any later version.
 *      
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *      
 *      You should have received a copy of the GNU General Public License
 *      along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdlib.h>
#include <glib.h>
#include <gio/gio.h>
#include <glib/gprintf.h>
#include <net/if.h>
#include <string.h>

#include "nm.h"

static gboolean nm_parse_proc_file(nm_data_t data);
static void nm_clear_device_info(nm_data_t data);


static gboolean nm_parse_proc_file(nm_data_t data)
{
	FILE *devstats;
	gchar statstr[1024];
	gchar *delim;
	gint start, i; 
	gchar *segment;
	gint64 *rows[16] = {&data->rxbytes, &data->rxpackets, &data->rxerrs, &data->rxdrop, &data->rxfifo, &data->rxframe, &data->rxcompressed, &data->rxmulticast,
						&data->txbytes, &data->txpackets, &data->txerrs, &data->txdrop, &data->txfifo, &data->txcolls, &data->txcarrier, &data->txcompressed};
	
	if (data == NULL) return FALSE;
	
	devstats = fopen("/proc/net/dev", "r");
	
	if (devstats != NULL) {
		while (fgets(statstr, sizeof(statstr), devstats) != NULL) {
			delim = strchr(statstr, ':');
			if (delim != NULL) {
				for (start = 0; start<delim-statstr; start++) {
					if (statstr[start] != ' ') break;
				}
				
				if (strncmp(statstr+start, data->interface, strlen(data->interface)) == 0) {
					for (i=0; i<16; i++) {
						if (i == 0) {
							segment = strtok(delim+1, " ");
						} else {
							segment = strtok(NULL, " ");
						}
						if (segment != NULL) {
							*(rows[i]) = strtoull(segment, NULL, 10);
						}
					}
				}
			}
		}
		fclose(devstats);
	}
	
	return TRUE;
}

static void nm_clear_device_info(nm_data_t data)
{
	if (data == NULL) return;
	
	memset(data->interface, 0, IFNAMSIZ);
	data->enabled = FALSE;
	data->rxbytes = 0;
	data->rxpackets = 0;
	data->rxerrs = 0;
	data->rxdrop = 0;
	data->rxfifo = 0;
	data->rxframe = 0;
	data->rxcompressed = 0;
	data->rxmulticast = 0;
	data->txbytes = 0;
	data->txpackets = 0;
	data->txerrs = 0;
	data->txdrop = 0;
	data->txfifo = 0;
	data->txcolls = 0;
	data->txcarrier = 0;
	data->txcompressed = 0;
}

gboolean nm_update_device_connection_info(nm_data_t data, guint devid)
{
	GError *error;
	GVariant *devicesvar;
	GVariantIter deviter, deviter2;
	GVariant *devnode, *devnode2;
	GVariant *devvalue;
	const gchar *devpath;
	gsize strsize;
	gchar *srcdevpath;
	GDBusProxy *devproxy;
	GVariant *info;
	const gchar *udevpath;
	const gchar *interface;
	guint type;
	guint state;
	gboolean found;
	
	if (data == NULL) return FALSE;
	
	error = NULL;
	
	srcdevpath = g_strdup_printf("/org/freedesktop/ModemManager/Modems/%u", devid);
	
	devicesvar = g_dbus_proxy_call_sync(data->nmproxy, "GetDevices", NULL, 0, -1, NULL, &error);
	
	if (error != NULL) {
		g_warning("Dbus proxy error: %s", error->message);
		g_error_free(error);
		g_free(srcdevpath);
		return FALSE;
	}
	
	found = FALSE;
	
	g_variant_iter_init(&deviter, devicesvar);
	
	while ((devnode = g_variant_iter_next_value(&deviter)) != NULL) {
		g_variant_iter_init(&deviter2, devnode);
		while ((devnode2 = g_variant_iter_next_value(&deviter2)) != NULL) {
			strsize = 256;
			devpath = g_variant_get_string(devnode2, &strsize);
			
			error = NULL;
			devproxy = g_dbus_proxy_new_sync(data->connection,
											G_DBUS_PROXY_FLAGS_NONE,
											NULL,
											"org.freedesktop.NetworkManager",
											devpath,
											"org.freedesktop.NetworkManager.Device",
											NULL,
											&error);
	
			if (error != NULL) {
				g_warning("Dbus proxy error: %s", error->message);
				g_error_free(error);
			} else {
				info = g_dbus_proxy_get_cached_property(devproxy, "Udi");
				strsize = 256;
				udevpath = g_variant_get_string(info, &strsize);
				g_variant_unref(info);
				
				info = g_dbus_proxy_get_cached_property(devproxy, "DeviceType");
				type = g_variant_get_uint32(info);
				g_variant_unref(info);
				
				if ((type == 8) && (g_str_equal(srcdevpath, udevpath))) {
					found = TRUE;
					
					info = g_dbus_proxy_get_cached_property(devproxy, "State");
					state = g_variant_get_uint32(info);
					g_variant_unref(info);
					
					if (state == 100) {
						data->enabled = TRUE;
						info = g_dbus_proxy_get_cached_property(devproxy, "IpInterface");
						strsize = 256;
						interface = g_variant_get_string(info, &strsize);
						memset(data->interface, 0, IFNAMSIZ);
						strncpy(data->interface, interface, IFNAMSIZ);
						g_variant_unref(info);
						nm_parse_proc_file(data);
					} else {
						data->enabled = FALSE;
						memset(data->interface, 0, IFNAMSIZ);
					} 
				}
				g_object_unref(G_OBJECT(devproxy));
			}
			g_variant_unref(devnode2);
		}
		g_variant_unref(devnode);
    }
	
	g_free(srcdevpath);
	
	return found;
}

gboolean nm_get_device_enabled(nm_data_t data)
{
	if (data == NULL) return FALSE;
	
	return data->enabled;
}

gchar *nm_get_device_interface(nm_data_t data)
{
	if (data == NULL) return NULL;
	if (data->interface[0] == '\0') return NULL;
	
	return data->interface;
}

gboolean nm_get_device_traffic(nm_data_t data, guint64 *rxbytes, guint64 *txbytes)
{
	if (data == NULL) return FALSE;
	
	if (data->enabled) {
		if (rxbytes != NULL) *rxbytes = data->rxbytes;
		if (txbytes != NULL) *txbytes = data->txbytes;
	} else {
		if (rxbytes != NULL) *rxbytes = 0;
		if (txbytes != NULL) *txbytes = 0; 
	}
	
	return TRUE;
}

void nm_close(nm_data_t data)
{
	if (data == NULL) return;
	
	if (data->nmproxy != NULL) {
		g_object_unref(G_OBJECT(data->nmproxy));
	}
	
	if (data->connection != NULL) {
		g_object_unref(G_OBJECT(data->connection));
	}
	
	g_free(data);
}

nm_data_t nm_open(void)
{
	nm_data_t data;
	GError *error;
		
	data = g_new(struct _nm_data, 1);
	
	data->connection = g_bus_get_sync(G_BUS_TYPE_SYSTEM, NULL, NULL);
	
	error = NULL;
	
	data->nmproxy = g_dbus_proxy_new_sync(data->connection,
											G_DBUS_PROXY_FLAGS_NONE,
											NULL,
											"org.freedesktop.NetworkManager",
											"/org/freedesktop/NetworkManager",
											"org.freedesktop.NetworkManager",
											NULL,
											&error);
	
	if (error != NULL) {
		g_warning("Dbus proxy error: %s", error->message);
		g_error_free(error);
		g_object_unref(data->connection);
		g_free(data);
		return NULL;
	} else {
		nm_clear_device_info(data);
	}
	
	return data;
}
