# -*- Mode: Python; coding: utf-8; indent-tabs-mode: nil; tab-width: 4 -*-
### BEGIN LICENSE
# Copyright (C) 2012 Simon Vermeersch simonvermeersch@gmail.com
# This program is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License version 3, as published 
# by the Free Software Foundation.
# 
# This program is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranties of 
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
# PURPOSE.  See the GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License along 
# with this program.  If not, see <http://www.gnu.org/licenses/>.
### END LICENSE
import os
import logging
import json
import datetime
import uuid
import glob
import weakref
import collections

logger = logging.getLogger('journey')

class Journey(object):
    """Holds a single journey with all of its attributes."""

    def __init__(self, storage, path):
        """Constructor. Storage is a reference to our storage class and path
        is a filename used to save to and load from."""
        self.storage = storage
        self.path = path

        self.text = ""
        self.title = "New journey"
        self.date = datetime.date.today()

        self.load()

    def set_text(self, text):
        """Sets the content of the journey."""
        self.text = text

    def set_title(self, title):
        """Sets the title of the journey."""
        self.title = title

    def set_date(self, date):
        """Sets the date of the journey."""
        self.date = date

    def load(self):
        """Loads a journey from the file we specified at the constructor. If
        the file does not exist, then this will not do anything."""
        if os.path.exists(self.path):
            logger.debug("Loading a journey from %s." % (self.path, ))
            f = open(self.path, "r")
            d = json.load(f)
            f.close()

            self.set_text(d["text"])
            self.set_title(d["title"])
            self.set_date(datetime.datetime \
                    .strptime(d["date"], "%Y-%m-%d").date())

    def save(self):
        """Saves the journey back to our file."""
        logger.debug("Saving a journey to %s." % (self.path, ))
        d = {"text": self.text,
             "date": self.date.isoformat(),
             "title": self.title}
        f = open(self.path, "w")
        json.dump(d, f)
        f.close()

    def delete(self):
        """Deletes a journey."""
        logger.debug("Deleting journey: %s." % (self.path, ))
        os.unlink(self.path)
        self.storage.all_journeys.remove(self)

    def month(self):
        """Returns the month this journey is for."""
        return (self.date.year, self.date.month)

class Storage(object):
    """This class will handle all our storage needs to save and load our
    journeys and also provides methods for finding journeys."""

    def __init__(self, journeys_dir):
        """Constructor. Takes a directory as an argument where journeys will
        be read from and stored to. If that directory does not exist, it will
        be created."""
        self.directory = journeys_dir
        self.all_journeys = []

        logger.debug("Using %s as storage directory." % (journeys_dir, ))
        if not os.path.exists:
            os.makedirs(self.directory)

        self.load_all_journeys()

    def load_all_journeys(self):
        """Loads all the journeys stored into the journey directory."""
        logger.debug("Loading all journeys.")
        files = glob.glob(os.path.join(self.directory, "*.journey"))
        for f in files:
            j = Journey(self, f)
            self.all_journeys.append(j)

    def create_new_journey(self):
        """Creates a new empty journey."""
        fname = uuid.uuid4().hex + ".journey"
        path = os.path.join(self.directory, fname)
        journey = Journey(self, path)
        journey.save()
        self.all_journeys.append(journey)
        return journey

def initialize(directory = "~/.journey"):
    """Creates a new global Storage instance that uses the users home directory
    by default to store journeys in"""
    homedir = os.path.expanduser(directory)

    if not os.path.exists(homedir):
        os.mkdir(homedir)

    global _storage
    _storage = Storage(homedir)

def journeys():
    """Returns a list of all our journeys."""
    return _storage.all_journeys

def get_journeys_order_by_date():
    """Returns a list of all the journeys ordered by date (descending)"""
    s = sorted(_storage.all_journeys, key = lambda j: j.date, reverse = True)
    return s

def create_new_journey():
    return _storage.create_new_journey()

def get_journeys_in_month(year, month):
    """Returns a list of all the journeys that happened in a specific month."""
    s = [j for j in _storage.all_journeys \
            if j.date.month == month and \
               j.date.year == year]
    return s

def get_journeys_for_date(date):
    """Returns all the journeys for one specific date."""
    s = [j for j in _storage.all_journeys \
            if j.date == date]
    return s
