# -*- Mode: Python; coding: utf-8; indent-tabs-mode: nil; tab-width: 4 -*-
### BEGIN LICENSE
# Copyright (C) 2012 Simon Vermeersch simonvermeersch@gmail.com
# This program is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License version 3, as published 
# by the Free Software Foundation.
# 
# This program is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranties of 
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
# PURPOSE.  See the GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License along 
# with this program.  If not, see <http://www.gnu.org/licenses/>.
### END LICENSE
import datetime
import gettext
from gettext import gettext as _
gettext.textdomain('journey')

from gi.repository import Gtk, Gdk # pylint: disable=E0611
import logging
logger = logging.getLogger('journey')

from journey_lib import Window
from journey.AboutJourneyDialog import AboutJourneyDialog
from journey.PreferencesJourneyDialog import PreferencesJourneyDialog
from journey.JourneyViewWidget import JourneyViewWidget
from journey.CalendarViewWidget import CalendarViewWidget
from journey import Storage

# See journey_lib.Window.py for more details about how this class works
class JourneyWindow(Window):
    __gtype_name__ = "JourneyWindow"
    
    def finish_initializing(self, builder): # pylint: disable=E1002
        """Set up the main window"""
        super(JourneyWindow, self).finish_initializing(builder)

        self.AboutDialog = AboutJourneyDialog
        self.PreferencesDialog = PreferencesJourneyDialog

        # Code for other initialization actions should be added here.

        #When this is true we don't run our callbacks
        self.refreshing = False

        self.current_view = None
        self.recreate_treeview()

        css = """GtkEventBox {
                background-image: -gtk-gradient (linear, 0 0, 0 1,
                                                 from(alpha(skyblue, 0.7)),
                                                 color-stop(0.7, white));
            }

            GtkTextView {
                background-color: rgba(255, 255, 255, 0.3);
                background-image: none;
            }

            GtkAlignment GtkCalendar {
                background-color: rgba(255, 255, 255, 0);
                background-image: none;
            }

            GtkViewport {
                background-color: rgba(255, 255, 255, 0);
                background-image: none;
            }

        """
        css_provider = Gtk.CssProvider()
        css_provider.load_from_data(css)

        screen = Gdk.Screen.get_default()
        style_context = Gtk.StyleContext()
        style_context.add_provider_for_screen(screen, css_provider, \
                Gtk.STYLE_PROVIDER_PRIORITY_USER)

        context = self.ui.toolbar1.get_style_context()
        context.add_class(Gtk.STYLE_CLASS_PRIMARY_TOOLBAR)

        current_date = datetime.date.today()
        self.show_calendar_view(current_date.year, current_date.month)


    def treeview_selection_changed_cb(self, parent):
        """Callback for when the selection changes on the treeview."""
        if self.refreshing:
            return
        store, it = parent.get_selected()
        text = store.get_value(it, 0)
        journey = store.get_value(it, 1)
        logger.debug("Selection changed to %s." % (text, ))
        if journey is not None:
            jvw = JourneyViewWidget()
            jvw.set_journey(journey)
            jvw.connect("title-changed", self.current_journey_title_changed_cb)
            jvw.connect("date-changed", self.current_journey_date_changed_cb)
            jvw.connect("deleted", self.journeydeleted_cb)
            self.set_view(jvw)
        else:
            year = store.get_value(it, 2)
            month = store.get_value(it, 3)
            self.show_calendar_view(year, month)

    def show_calendar_view(self, year, month):
        """Shows the calendar view for the selected year and month."""
        calview = CalendarViewWidget()
        calview.select_month(year, month)
        calview.connect("new-journey", self.calendar_view_new_cb)
        calview.connect("selected-journey", self.calendar_view_selected_cb)
        self.set_view(calview)

    def calendar_view_new_cb(self, parent, date):
        """Callback when the user clicks the new button in the calendar view."""
        logger.debug("Creating a new journey for " + str(date))
        new_journey = Storage.create_new_journey()
        new_journey.date = date
        new_journey.save()
        self.recreate_treeview(new_journey)
        self.treeview_selection_changed_cb(self.ui.treeview_selection)

    def calendar_view_selected_cb(self, parent, journey):
        """Callback for when the user clicks on the title of a journey in the
        calendar view."""
        logger.debug("Selecting journey: " + journey.title)
        self.recreate_treeview(journey)
        self.treeview_selection_changed_cb(self.ui.treeview_selection)


    def current_journey_title_changed_cb(self, parent):
        """Callback for when the user changes the title of the journey. We
        should update our treeview."""
        if self.refreshing:
            return
        logger.debug("Title changed")

        #Since we can only select one journey, the currently selected one is
        #always the one we need to change. How handy :)
        store, it = self.ui.treeview_selection.get_selected()
        journey = store.get_value(it, 1)
        newtitle = journey.title
        store.set_value(it, 0, newtitle)

    def current_journey_date_changed_cb(self, parent):
        """Gets called when the user changed the date of the selected
        journey."""

        #XXX Let's be lazy here and just recreate the whole tree, will be much
        #easier. (We lose what branches the user has expanded though in the
        #tree)
        store, it = self.ui.treeview_selection.get_selected()
        journey = store.get_value(it, 1)
        self.recreate_treeview(journey)

    def newjourneyact_activate_cb(self, parent):
        """Callback for the new button."""
        logger.debug("New journey")
        new_journey = Storage.create_new_journey()
        self.recreate_treeview(new_journey)
        self.treeview_selection_changed_cb(self.ui.treeview_selection)

    def deletejourneyact_activate_cb(self, parent):
        """Callback for the delete button."""
        logger.debug("Delete journey")
        store, it = self.ui.treeview_selection.get_selected()
        journey = store.get_value(it, 1)
        journey.delete()
        self.recreate_treeview()
        current_date = datetime.date.today()
        self.show_calendar_view(current_date.year, current_date.month)

    def journeydeleted_cb(self, parent):
        """Called when the user deletes a journey in the journey view widget."""
        self.recreate_treeview()
        current_date = datetime.date.today()
        self.show_calendar_view(current_date.year, current_date.month)

    def set_view(self, widget_to_show):
        """Replaces the current widget on the right-side of our paned view and
        replaces it with a new one."""
        if self.current_view:
            self.ui.paned1.remove(self.current_view)
        self.current_view = widget_to_show
        widget_to_show.show()
        self.ui.paned1.add2(widget_to_show)

    def recreate_treeview(self, selected = None):
        """Clears the model of our treeview and recreates it from scratch.
        The optional parameter is the currently selected journey."""
        logger.debug("Recreating the treestore from scratch.")
        self.refreshing = True
        store = self.ui.treestore
        store.clear()

        current_month = (0, 0)
        parent_iter = None
        selection = None
        for j in Storage.get_journeys_order_by_date():
            if j.month() != current_month:
                current_month = j.month()
                parent_iter = store.append(None, [j.date.strftime("%B %Y"), \
                        None, current_month[0], current_month[1]])
            it = store.append(parent_iter, [j.title, j, current_month[0], \
                    current_month[1]])

            if j == selected:
                selection = store.get_path(it)

        if selection:
            self.ui.treeview1.expand_to_path(selection)
            self.ui.treeview_selection.select_path(selection)
        self.refreshing = False
