# -*- Mode: Python; coding: utf-8; indent-tabs-mode: nil; tab-width: 4 -*-
### BEGIN LICENSE
# Copyright (C) 2012 Simon Vermeersch simonvermeersch@gmail.com
# This program is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License version 3, as published 
# by the Free Software Foundation.
# 
# This program is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranties of 
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
# PURPOSE.  See the GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License along 
# with this program.  If not, see <http://www.gnu.org/licenses/>.
### END LICENSE

import datetime

from gi.repository import Gtk, Gdk, GObject # pylint: disable=E0611
import logging
logger = logging.getLogger('journey')

from journey_lib.helpers import get_builder

class JourneyViewWidget(Gtk.EventBox):
    __gtype_name__ = "JourneyViewWidget"
    __gsignals__ = {
        'title-changed' : (GObject.SIGNAL_RUN_LAST, GObject.TYPE_NONE, \
                           ()),
        'date-changed' : (GObject.SIGNAL_RUN_LAST, GObject.TYPE_NONE, \
                           ()),
        'deleted' :  (GObject.SIGNAL_RUN_LAST, GObject.TYPE_NONE, ()),
    }

    def __new__(cls):
        """Special static method that's automatically called by Python when
        constructing a new instance of this class.

        Returns a fully instantiated JourneyViewWidget object.
        """
        builder = get_builder('JourneyViewWidget')
        new_object = builder.get_object("journey_view_widget")
        new_object.finish_initializing(builder)
        return new_object

    def finish_initializing(self, builder):
        """Called while initializing this instance in __new__

        finish_initalizing should be called after parsing the ui definition
        and creating a JourneyViewWidget object with it in order to finish
        initializing the start of the new JourneyViewWidget instance.

        Put your initialization code in here and leave __init__ undefined.
        """
        # Get a reference to the builder and set up the signals.
        self.builder = builder
        self.ui = builder.get_ui(self)

        self.refreshing = False
        self.save_timeout = None

    def set_journey(self, journey):
        """Sets the journey to display."""
        if self.refreshing:
            return False

        self.refreshing = True
        self.journey = journey
        self.ui.titleentry.set_text(journey.title)
        self.ui.datebutton.set_label(journey.date.strftime("%x"))
        self.ui.textbuffer.set_text(journey.text)
        self.refreshing = False

    def titleentry_changed_cb(self, parent):
        """Callback for when the user changes the text on the title entry
        box."""
        if self.refreshing:
            return False

        newtitle = parent.get_text()
        self.journey.title = newtitle
        self.emit("title-changed")
        self.queue_save()

    def find_parent_window(self):
        """Returns the original parent GtkWindow of our control."""
        #XXX Somehow I doubt this is how something like that should be done,
        #but it works for now.
        parent_window = self.get_parent()
        while True:
            try:
                parent_window = parent_window.get_parent()
                if isinstance(parent_window, Gtk.Window):
                    break
            except AttributeError:
                parent_window = None
                break

    def datebutton_clicked_cb(self, parent):
        """Callback when the date button is clicked."""
        logger.debug("Date button clicked")

        parent_window = self.find_parent_window()
        current_date = self.journey.date

        dlg = Gtk.Dialog("Pick a date", parent_window, 0,
            (Gtk.STOCK_CANCEL, Gtk.ResponseType.CANCEL,
             Gtk.STOCK_OK, Gtk.ResponseType.OK))

        cal = Gtk.Calendar()
        cal.set_display_options(Gtk.CalendarDisplayOptions.SHOW_HEADING | \
            Gtk.CalendarDisplayOptions.SHOW_DAY_NAMES | \
            Gtk.CalendarDisplayOptions.SHOW_WEEK_NUMBERS)
        cal.select_month(current_date.month-1, current_date.year)
        cal.select_day(current_date.day)


        dlg.get_content_area().add(cal)
        cal.show()

        if dlg.run() == Gtk.ResponseType.OK:
            logger.debug("New date selected")
            (year, month, day) = cal.get_date()
            new_date = datetime.date(year, month+1, day)
            self.journey.date = new_date
            self.ui.datebutton.set_label(new_date.strftime("%x"))
            self.emit("date-changed")
            self.queue_save()

        dlg.destroy()

    def deletebtn_clicked_cb(self, parent):
        """User clicked on the delete button."""
        logger.debug("Delete button clicked")
        parent_window = self.find_parent_window()
        dlg = Gtk.MessageDialog(parent_window, \
                Gtk.DialogFlags.DESTROY_WITH_PARENT, 
                Gtk.MessageType.WARNING,
                Gtk.ButtonsType.YES_NO, \
                "Are you sure you want to delete this journey?")
        if dlg.run() == Gtk.ResponseType.YES:
            self.journey.delete()
            self.emit("deleted")
        dlg.destroy()

    def queue_save(self):
        """Triggers a timeout in 250ms. If there was a previous timeout
        already then that one gets cancelled.
        The reason we do not save immediately is to not cause unneeded saves
        as the user is typing something."""
        if self.save_timeout:
            logger.debug("Cancelling previous save timeout.")
            GObject.source_remove(self.save_timeout)
        logger.debug("Starting new save timeout.")
        self.save_timeout = GObject.timeout_add(250, self.save_timeout_cb)

    def save_timeout_cb(self):
        """Callback when the save timeout hit."""
        logger.debug("Save timeout hit: saving journey.")
        self.save_timeout = None
        self.journey.save()
        return False

    def textbuffer_changed_cb(self, parent):
        """Called when we change the text."""
        if self.refreshing:
            return False
        logger.debug("Journey text changed.""")
        start = parent.get_start_iter()
        end = parent.get_end_iter()
        new_text = parent.get_text(start, end, False)
        self.journey.text = new_text
        self.queue_save()

