# -*- Mode: Python; coding: utf-8; indent-tabs-mode: nil; tab-width: 4 -*-
### BEGIN LICENSE
# Copyright (C) 2012 Simon Vermeersch simonvermeersch@gmail.com
# This program is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License version 3, as published 
# by the Free Software Foundation.
# 
# This program is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranties of 
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
# PURPOSE.  See the GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License along 
# with this program.  If not, see <http://www.gnu.org/licenses/>.
### END LICENSE

import datetime

from gi.repository import Gtk, Gdk, GObject # pylint: disable=E0611
import logging
logger = logging.getLogger('journey')

from journey_lib.helpers import get_builder
from journey import Storage

class CalendarViewWidget(Gtk.EventBox):
    __gtype_name__ = "CalendarViewWidget"
    __gsignals__ = {
        'new-journey' : (GObject.SIGNAL_RUN_LAST, GObject.TYPE_NONE, \
                           (object,)),
        'selected-journey' : (GObject.SIGNAL_RUN_LAST, GObject.TYPE_NONE, \
                           (object,)),
    }

    def __new__(cls):
        """Special static method that's automatically called by Python when
        constructing a new instance of this class.

        Returns a fully instantiated JourneyViewWidget object.
        """
        builder = get_builder('CalendarViewWidget')
        new_object = builder.get_object("calendar_view_widget")
        new_object.finish_initializing(builder)
        return new_object

    def finish_initializing(self, builder):
        """Called while initializing this instance in __new__

        finish_initalizing should be called after parsing the ui definition
        and creating a JourneyViewWidget object with it in order to finish
        initializing the start of the new JourneyViewWidget instance.

        Put your initialization code in here and leave __init__ undefined.
        """
        # Get a reference to the builder and set up the signals.
        self.builder = builder
        self.ui = builder.get_ui(self)

        self.journey_buttons = []


    def select_month(self, year, month):
        """Selects a month in the calendar."""
        self.ui.calendar.select_month(month-1, year)

    def calendar_month_changed_cb(self, calendar):
        """Callback when the months get changed in the calendar: we have to
        update our various marks here."""
        logger.debug("Calendar month changed.")

        (year, month, day) = self.ui.calendar.get_date()
        month += 1 #GtkCalendar starts counting from 0
        self.ui.calendar.clear_marks()

        journeys = Storage.get_journeys_in_month(year, month)
        for j in journeys:
            self.ui.calendar.mark_day(j.date.day)

        #force to refresh the day too
        self.calendar_day_selected_cb(None)

    def calendar_day_selected_cb(self, calendar):
        """Called when the user selects another date on the calendar."""
        logger.debug("Calendar day changed.")

        (year, month, day) = self.ui.calendar.get_date()
        date = datetime.date(year, month+1, day)

        for btn in self.journey_buttons:
            btn.destroy()

        self.journey_buttons = []
        journeys = Storage.get_journeys_for_date(date)
        for journey in journeys:
            linkbutton = Gtk.Button(journey.title)
            linkbutton.set_relief(Gtk.ReliefStyle.NONE)
            linkbutton.show()
            linkbutton.connect("clicked", self.journeybtn_clicked_cb, \
                journey)
            self.ui.box2.pack_start(linkbutton, False, True, 0)
            self.journey_buttons.append(linkbutton)

    def newjourneybtn_clicked_cb(self, parent):
        """Callback for when the user clicks on the new button."""
        logger.debug("New journey clicked")
        (year, month, day) = self.ui.calendar.get_date()
        date = datetime.date(year, month+1, day)

        self.emit("new-journey", date)

    def journeybtn_clicked_cb(self, parent, journey):
        """Callback for when the user clicks on a journey button."""
        logger.debug("Journey button clicked")
        self.emit("selected-journey", journey)

