# -*- Mode: Python; coding: utf-8; indent-tabs-mode: nil; tab-width: 4 -*-
### BEGIN LICENSE
# Copyright (C) 2012 <Michal Predotka> <mivoligo@gmail.com>
# This program is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License version 3, as published 
# by the Free Software Foundation.
# 
# This program is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranties of 
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
# PURPOSE.  See the GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License along 
# with this program.  If not, see <http://www.gnu.org/licenses/>.
### END LICENSE

import gettext
from gettext import gettext as _
gettext.textdomain('easy-stopwatch')
from easy_stopwatch_lib.helpers import get_media_file
from easy_stopwatch_lib.helpers import get_help_uri
from easy_stopwatch_lib.helpers import show_uri
from gi.repository import Gtk # pylint: disable=E0611
from gi.repository import Gdk
from gi.repository import Pango
from gi.repository import GObject
from gi.repository import Notify
GObject.threads_init()
import time
import datetime
import logging
logger = logging.getLogger('easy_stopwatch')
from easy_stopwatch_lib import Window
from easy_stopwatch.AboutEasyStopwatchDialog import AboutEasyStopwatchDialog
from easy_stopwatch.PreferencesEasyStopwatchDialog import PreferencesEasyStopwatchDialog



# See easy_stopwatch_lib.Window.py for more details about how this class works
class EasyStopwatchWindow(Window):
    __gtype_name__ = "EasyStopwatchWindow"
    
    def finish_initializing(self, builder): # pylint: disable=E1002
        """Set up the main window"""
        super(EasyStopwatchWindow, self).finish_initializing(builder)

        self.AboutDialog = AboutEasyStopwatchDialog
        self.PreferencesDialog = PreferencesEasyStopwatchDialog

        # Code for other initialization actions should be added here.
        
        # Dark toolbar
        self.toolbar = self.builder.get_object("toolbar")
        context = self.toolbar.get_style_context()
        context.add_class(Gtk.STYLE_CLASS_PRIMARY_TOOLBAR)


        # Time stuff
        self.timeentry = self.builder.get_object("timeentry")
        self.timeentry.set_text("00:00:00.0")
        self.elapsedtime = None
        self.seconds = 0.0
        self.minutes = 0
        self.hours = 0

        # Styling via CSS file
        screen = Gdk.Screen.get_default()
        css_provider = Gtk.CssProvider()
        css_provider.load_from_path(get_media_file("gtkstyle.css"))
        style_context = Gtk.StyleContext()
        style_context.add_provider_for_screen(screen, css_provider, Gtk.STYLE_PROVIDER_PRIORITY_USER)
        self.timeentry.get_style_context().add_class("timeentry")
        self.startpausebutton = self.builder.get_object("startpausebutton")
        self.startpausebutton.get_style_context().add_class("start")
        self.laptimes = self.builder.get_object("laptimeview")
        self.laptimes.get_style_context().add_class("laptimes")
        self.scrolledw1 = self.builder.get_object("scrolledwindow1")
        self.scrolledw1.get_style_context().add_class("scrolledw1")
        self.timeentrybutton = self.builder.get_object("timeentrybutton")
        self.timeentrybutton.get_style_context().add_class("timeentrybutton")

    # Counting time
    def update_timeentry(self):
        if self.seconds > 60:
            self.minutes += 1
            self.seconds = 0
            if self.minutes > 60:
                self.hours += 1
                self.minutes = 0
        else:
            self.seconds += 0.1
        self.timeentry.set_text("%02d:%02d:%04.1f" % (self.hours, self.minutes, self.seconds) )
        return True

    # "Start-Pause" button clicked
    def on_startpausebutton_clicked(self, widget):
        self.label = self.builder.get_object("startpausebuttonlabel")
        self.whattext = self.label.get_text()
        timeentry = self.builder.get_object("timeentry")
        if self.whattext == "Start":
            # Change label on start-pause button to "Pause"
            self.label.set_text("Pause")
            # Add button classes for styling
            self.startpausebutton.get_style_context().remove_class("start")
            self.startpausebutton.get_style_context().add_class("pause")
            # Start stopwatch
            self.elapsedtime = GObject.timeout_add(100, self.update_timeentry)
        else:
            # Change label on start-pause button to "Start"
            self.label.set_text("Start")
            # Add button classes for styling
            self.startpausebutton.get_style_context().remove_class("pause")
            self.startpausebutton.get_style_context().add_class("start")            
            # Pause stopwatch
            GObject.source_remove(self.elapsedtime)
            self.elapsedtime = None
    def on_mnu_start_pause_activate(self, widget):
        self.on_startpausebutton_clicked(self)
    # "Timeentry"/reset button clicked
    def on_timeentrybutton_clicked(self, widget):
        self.label = self.builder.get_object("startpausebuttonlabel")
        self.whattext = self.label.get_text()
        if self.whattext == "Start":
            # Reset to zero if stopwatch is paused
            self.seconds = 0
            self.minutes = 0
            self.hours = 0
            self.timeentry.set_text("00:00:00.0")
        else:
            # Reset to zero if stopwatch is running
            GObject.source_remove(self.elapsedtime)
            self.elapsedtime = None
            self.seconds = 0
            self.minutes = 0
            self.hours = 0
            self.timeentry.set_text("00:00:00.0")
            self.label.set_text("Start")
    def on_mnu_reset_activate(self, widget):
        self.on_timeentrybutton_clicked(self)
    # "Show lap times" button clicked
    def on_showlapbutton_toggled(self, widget):
        self.mnushowlaps = self.builder.get_object("mnu_show_laps")
        self.showlapbutton = self.builder.get_object("showlapbutton")
        lapbox = self.builder.get_object("lapbox")
        if self.showlapbutton.get_active() == True:
            # Show laptime box
            lapbox.show()
            # Check menu item "mnu_show_laps"
            self.mnushowlaps.set_active(True)
        else:
            # Hide laptime box
            lapbox.hide()
            # Uncheck menu item "mnu_show_laps"
            self.mnushowlaps.set_active(False)
    def on_mnu_show_laps_toggled(self, widget):
        self.showlapbutton = self.builder.get_object("showlapbutton")
        self.mnushowlaps = self.builder.get_object("mnu_show_laps")
        if self.mnushowlaps.get_active() == True:
            self.showlapbutton.set_active(True)
        else:
            self.showlapbutton.set_active(False)


    # "Lap" button clicked
    def on_lapbutton_clicked(self, widget):
        self.gettime = self.builder.get_object("timeentry")
        self.laptime = self.gettime.get_text()
        self.laptimeview = self.builder.get_object("laptimeview")
        buffer = self.laptimeview.get_buffer()
        buffer.place_cursor(buffer.get_end_iter())
        # Print lap times in laptime box
        buffer.insert_at_cursor(" " + self.laptime + "\n")
        self.laptimeview.scroll_to_mark(buffer.get_insert(), 0, use_align=False, xalign=0.5, yalign=0.5)
    def on_mnu_lap_activate(self, widget):
        self.on_lapbutton_clicked(self)

    # "Clear lap times" button clicked
    def on_clearbutton_clicked(self, widget):
        self.laptimeview = self.builder.get_object("laptimeview")
        buffer = self.laptimeview.get_buffer()
        # Clear all lap times
        buffer.set_text("")
    def on_mnu_clear_activate(self, widget):
        self.on_clearbutton_clicked(self)

    # "Save lap times" button clicked
    def on_savelaptimesbutton_clicked(self,widget):
        dialog = Gtk.FileChooserDialog(_("Please choose a file"), self,
            Gtk.FileChooserAction.SAVE,
            (Gtk.STOCK_CANCEL, Gtk.ResponseType.CANCEL,
             Gtk.STOCK_SAVE, Gtk.ResponseType.OK))
        dialog.set_do_overwrite_confirmation(True)
        filter_txt = Gtk.FileFilter()
        filter_txt.set_name(_("text files"))
        filter_txt.add_mime_type("text/plain")
        dialog.add_filter(filter_txt)
        # Set file name 
        now = datetime.datetime.now()
        file_name = "Lap Times "+str(now.strftime("%x %X")+".txt")
        dialog.set_current_name(file_name)
        
        response = dialog.run()

        if response == Gtk.ResponseType.OK:
            filename = dialog.get_filename()
            laptimefile = open(filename, "w+")
            laptimeviewbuffer = self.builder.get_object("laptimeview").get_buffer()
            laptimefile.write(laptimeviewbuffer.get_text(laptimeviewbuffer.get_start_iter(), laptimeviewbuffer.get_end_iter(), include_hidden_chars=True))

            laptimefile.close()
            # Notify about saving
            icon = get_media_file("easy-stopwatch.svg")
            Notify.init ("save confirmation")
            notification = Notify.Notification.new ("Easy Stopwatch", "Lap times saved", icon)
            notification.show ()
        dialog.destroy()

    def on_mnu_save_activate(self, widget):
        self.on_savelaptimesbutton_clicked(self)

    # "Help" button clicked
    def on_helpbutton_clicked(self, widget):
        show_uri(self, "ghelp:%s" % get_help_uri())

        
    # "About" button clicked
    def on_aboutbutton_clicked(self, widget):
        aboutdialog = AboutEasyStopwatchDialog()
        result = aboutdialog.run()
        aboutdialog.destroy()
        if result != Gtk.ResponseType.OK:
            return
        

