from cuttlefish.params import StringParam, SelectParam, IntParam, BoolParam
from cuttlefish.plugins import CuttlePlugin
from cuttlefish.events import CuttleEvent, DBusEvent
from cuttlefish.actions import CuttleAction

import threading
import dbus


def _convert_ssid(SSIDbyteArray):
	ssid = ""
	for byte in SSIDbyteArray:
		ssid += chr(byte)
	return ssid


class WlanConnects(DBusEvent, CuttlePlugin):
	NAME = "Connect to WLAN"
	DESCP = "React when connecting to a specified wireless network"
	CATEGORY = "Network"
	PARAMS = {
		'ssid' : ''
	}

	#http://projects.gnome.org/NetworkManager/developers/api/09/spec.html#type-NM_STATE
	NM_STATE_CONNECTED_GLOBAL = 70

	class Editor(CuttlePlugin.Editor):
		def begin(self):
			ssids = {}
			bus = dbus.SystemBus()
			setsobj = bus.get_object('org.freedesktop.NetworkManager', '/org/freedesktop/NetworkManager/Settings')
			settings = dbus.Interface(setsobj, 'org.freedesktop.NetworkManager.Settings')
			for cspath in settings.ListConnections():
				csobj = bus.get_object('org.freedesktop.NetworkManager', cspath)
				connecSettings = dbus.Interface(csobj, 'org.freedesktop.NetworkManager.Settings.Connection')
				config = connecSettings.GetSettings()
				if '802-11-wireless' in config:
					ssid = _convert_ssid(config['802-11-wireless']['ssid'])
					ssids[ssid] = ssid

			return {
					'ssid' : SelectParam('SSID of the wireless network', ssids, str)
				}
			

	def __init__(self):
		DBusEvent.__init__(self, self._on_connec_state_changed, "StateChanged", "org.freedesktop.NetworkManager", "/org/freedesktop/NetworkManager", DBusEvent.USE_SYSTEM_BUS)
		CuttlePlugin.__init__(self)

	def triggerOnStartup(self):
		nmobj = self._bus.get_object('org.freedesktop.NetworkManager', '/org/freedesktop/NetworkManager')
		nmprops = dbus.Interface(nmobj, 'org.freedesktop.DBus.Properties')
		self._on_connec_state_changed(nmprops.Get('org.freedesktop.NetworkManager', 'State'))

	def _on_connec_state_changed(self, state):
		self.logger.debug("state %d" % state)
		if state == self.NM_STATE_CONNECTED_GLOBAL:
			ssid = self._get_ssid()
			if ssid and ssid == self._params['ssid']:
				self.trigger()
			
	def _get_ssid(self):
		nmobj = self._bus.get_object('org.freedesktop.NetworkManager', '/org/freedesktop/NetworkManager')
		nmprops = dbus.Interface(nmobj, 'org.freedesktop.DBus.Properties')
		for acpath in nmprops.Get('org.freedesktop.NetworkManager', 'ActiveConnections'):
			acobj = self._bus.get_object('org.freedesktop.NetworkManager', acpath)
			acprops = dbus.Interface(acobj, 'org.freedesktop.DBus.Properties')
			
			appath = acprops.Get('org.freedesktop.NetworkManager.Connection.Active', 'SpecificObject')	
			apobj = self._bus.get_object('org.freedesktop.NetworkManager', appath)
			approps = dbus.Interface(apobj, 'org.freedesktop.DBus.Properties')
		
			try:
				return _convert_ssid(approps.Get('org.freedesktop.NetworkManager.AccessPoint', 'Ssid'))
			except:
				continue
			
		return False


class WlanDisconnects(DBusEvent, CuttlePlugin):
	NAME = "Disconnect from WLAN/LAN"
	DESCP = "React when losing  the network connection"
	CATEGORY = "Network"
	PARAMS = {
		'delay' : 5
	}

	#http://projects.gnome.org/NetworkManager/developers/api/09/spec.html#type-NM_STATE
	NM_STATE_DISCONNECTED = 20


	class Editor(CuttlePlugin.Editor):
		def begin(self):
			return {'delay' : IntParam('Only if disconnected longer than (in sec)')}	

	def __init__(self):
		DBusEvent.__init__(self, self._on_connec_state_changed, "StateChanged", "org.freedesktop.NetworkManager", "/org/freedesktop/NetworkManager", DBusEvent.USE_SYSTEM_BUS)
		CuttlePlugin.__init__(self)
		self._reconnect = threading.Event()
		
	def _on_connec_state_changed(self, state):
		if state == self.NM_STATE_DISCONNECTED:
			if self._params['delay'] <= 0:
				self.trigger()
			else:
				self._reconnect.clear()
				threading.Thread(target=self._await).start()
		else:
			self._reconnect.set()

	def _await(self):
		if not self._reconnect.wait(self._params['delay']):
			self.trigger()



class ChangeWLAN(CuttleAction):
	CATEGORY = "Network"

	def __init__(self, active):
		CuttleAction.__init__(self)
		self._active = active 

	def execute(self):
		bus = dbus.SystemBus();
		nmobj = bus.get_object('org.freedesktop.NetworkManager', '/org/freedesktop/NetworkManager')
		nmprops = dbus.Interface(nmobj, 'org.freedesktop.DBus.Properties')
		nmprops.Set('org.freedesktop.NetworkManager', 'WirelessEnabled', self._active)



class TurnOnWLAN(ChangeWLAN, CuttlePlugin):
	NAME = 'Activate WLAN'
	DESCP = 'Turn on wireless networking'

	def __init__(self):
		ChangeWLAN.__init__(self, True)
		CuttlePlugin.__init__(self)
		

class TurnOffWLAN(ChangeWLAN, CuttlePlugin):
	NAME = 'Deactivate WLAN'
	DESCP = 'Turn off wireless networking'

	def __init__(self):
		ChangeWLAN.__init__(self, False)
		CuttlePlugin.__init__(self)
