from cuttlefish.actions import CuttleAction
from cuttlefish.events import CuttleEvent, DBusEvent
from cuttlefish.plugins import CuttlePlugin
from cuttlefish.params import SelectParam, FileParam

import dbus

def _get_default_adapter_path(bus):
	bluezobj = bus.get_object('org.bluez', '/')
	bluezmgr = dbus.Interface(bluezobj, 'org.bluez.Manager')
	
	return bluezmgr.DefaultAdapter()


class ChangeBluezPower(CuttleAction):
	CATEGORY = "Bluetooth"

	def __init__(self, powered):
		CuttleAction.__init__(self)
		self._powered = powered

		
	def execute(self):
		bus = dbus.SystemBus()	
		adapobj = bus.get_object('org.bluez', _get_default_adapter_path(bus))
		adapter = dbus.Interface(adapobj, 'org.bluez.Adapter')
		adapter.SetProperty('Powered', self._powered)


class TurnOnBluez(ChangeBluezPower, CuttlePlugin):
	NAME = "Activate Bluetooth"
	DESCP = "Turn on the default bluetooth adapter"

	def __init__(self):
		ChangeBluezPower.__init__(self, True)


class TurnOffBluez(ChangeBluezPower, CuttlePlugin):
	NAME = "Deactivate Bluetooth"
	DESCP = "Turn off the default bluetooth adapter"

	def __init__(self):
		ChangeBluezPower.__init__(self, False)



class BluezPowerChanged(DBusEvent):
	CATEGORY = "Bluetooth"

	def __init__(self, powered):
		DBusEvent.__init__(self, self.on_prop_changed, 'PropertyChanged', 'org.bluez.Adapter', lambda : _get_default_adapter_path(self._bus), DBusEvent.USE_SYSTEM_BUS)
		CuttlePlugin.__init__(self)
		self._powered = powered
		
	def on_prop_changed(self, prop, value):
		if prop == 'Powered' and value == self._powered:
			self.trigger()


class BluezActivated(BluezPowerChanged, CuttlePlugin):
	NAME = "Bluetooth activated"
	DESCP = "React when bluetooth is turned on"

	def __init__(self):
		BluezPowerChanged.__init__(self, True)
		CuttlePlugin.__init__(self)


class BluezDeactivated(BluezPowerChanged, CuttlePlugin):
	NAME = "Bluetooth deactivated"
	DESCP = "React when bluetooth is turned off"

	def __init__(self):
		BluezPowerChanged.__init__(self, False)
		CuttlePlugin.__init__(self)



class BluezDeviceChanged(DBusEvent):
	CATEGORY = "Bluetooth"
	PARAMS = {'dev' : ''}

	class Editor(CuttlePlugin.Editor):
		def begin(self):
			bus = dbus.SystemBus()
			adapobj = bus.get_object('org.bluez', _get_default_adapter_path(bus))
			adapter = dbus.Interface(adapobj, 'org.bluez.Adapter')

			devs = {}
			for devpath in adapter.ListDevices():
				devobj = bus.get_object('org.bluez', devpath)
				device = dbus.Interface(devobj, 'org.bluez.Device')
				props = device.GetProperties()

				devs[str(devpath)] = props['Name']
			
			
			return {'dev' : SelectParam('Monitored device', devs, str)}



	def __init__(self, connected):
		DBusEvent.__init__(self, self.on_prop_changed, 'PropertyChanged', 'org.bluez.Device', lambda : self._params['dev'], DBusEvent.USE_SYSTEM_BUS)
		CuttlePlugin.__init__(self)
		self._connected = connected
		
	def on_prop_changed(self, prop, value):
		if prop == 'Connected' and value == self._connected:
			self.trigger()


class BluezDeviceDisappear(BluezDeviceChanged, CuttlePlugin):
	NAME = "Bluetooth device disconnects"
	DESCP ="React when a specified bluetooth device disconnects"

	def __init__(self):
		BluezDeviceChanged.__init__(self, False)
		CuttlePlugin.__init__(self)


class BluezDeviceFound(BluezDeviceChanged, CuttlePlugin):
	NAME = "Bluetooth device connects"
	DESCP ="React when a specified bluetooth device connects"

	def __init__(self):
		BluezDeviceChanged.__init__(self, True)
		CuttlePlugin.__init__(self)