# -*- Mode: Python; coding: utf-8; indent-tabs-mode: nil; tab-width: 4 -*-
### BEGIN LICENSE
# Copyright (C) 2012 John Vrbanac <john.vrbanac (at) linux.com>
# This program is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License version 3, as published 
# by the Free Software Foundation.
# 
# This program is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranties of 
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
# PURPOSE.  See the GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License along 
# with this program.  If not, see <http://www.gnu.org/licenses/>.
### END LICENSE

from gi.repository import Gtk  # pylint: disable=E0611
from cuckoo_lib.helpers import get_builder
import os
import locale
from locale import gettext as _
locale.textdomain('cuckoo')

from cuckoo.AudioPlayer import AudioPlayer
from cuckoo.CuckooAlarmData import CuckooAlarmData


class EditAlarmDialog(Gtk.Dialog):
    __gtype_name__ = "EditAlarmDialog"
    __activated = False
    audioplayer = None
    response = None

    def __new__(cls):
        builder = get_builder('EditAlarmDialog')
        new_object = builder.get_object('edit_alarm_dialog')
        new_object.finish_initializing(builder)
        return new_object

    def finish_initializing(self, builder):
        # Get a reference to the builder and set up the signals.
        self.builder = builder
        self.ui = builder.get_ui(self)

        self.hour_adjustment = self.builder.get_object("hour_adjustment")
        self.minute_adjustment = self.builder.get_object("minute_adjustment")
        self.minute_spinbutton = self.builder.get_object("minute_spinbutton")
        self.ampm_combobox = self.builder.get_object("ampm_combobox")
        self.btn_test = self.builder.get_object("btn_test")
        self.audiochooser = self.builder.get_object("audiochooser")
        self.volume_btn = self.builder.get_object("volumebtn")

        self.ampm_combobox.set_active(0)

        self.audioplayer = AudioPlayer()
        self.audioplayer.connect(AudioPlayer.STATE_CHANGED, self.on_player_state_changed)
        self.audiochooser.set_hexpand(False)
        self.volume_btn.set_hexpand(False)
        self.volume_btn.set_halign(Gtk.Align.START)

    def set_data(self, data):
        if data.get_ampm() == "PM":
            self.ampm_combobox.set_active(1)
        else:
            self.ampm_combobox.set_active(0)

        self.hour_adjustment.set_value(data.get_hour())
        self.minute_adjustment.set_value(data.get_minute())
        self.audiochooser.set_uri(data.get_audio_filename())
        self.volume_btn.set_value((data.get_volume() / 1.75) * 100)
        self.__activated = data.get_activated()

    def get_data(self):
        data = CuckooAlarmData()

        if self.ampm_combobox.get_active() == 1:
            data.set_ampm("PM")
        else:
            data.set_ampm("AM")

        data.set_hour(int(self.hour_adjustment.get_value()))
        data.set_minute(int(self.minute_adjustment.get_value()))
        data.set_volume(float((self.volume_btn.get_value() / 100.0) * 1.75))
        data.set_audio_filename(self.audiochooser.get_uri())
        data.set_activated(self.__activated)

        return data

    """ Handlers """
    def on_edit_alarm_dialog_response(self, widget, response_id):
        self.response = response_id

    def on_minute_spinbutton_output(self, widget, data=None):
        self.minute_spinbutton.set_text('{:02d}'.format(int(self.minute_adjustment.get_value())))
        return False

    def on_btn_ok_clicked(self, widget, data=None):
        self.btn_test.set_active(False)
        self.destroy()
        pass

    def on_audiochooser_file_set(self, widget, data=None):
        self.btn_test.set_active(False)

    def on_player_state_changed(self, target, state):
        if state == AudioPlayer.PLAYING:
            self.btn_test.set_active(True)
        elif state == AudioPlayer.STOPPED:
            self.btn_test.set_active(False)

    def on_volumebtn_value_changed(self, widget, data=None):
        value = widget.get_adjustment().get_value() / 100
        self.audioplayer.set_volume(1.75 * value)
        #pass

    def on_btn_test_toggled(self, widget, data=None):

        play_button = self.builder.get_object("test_play_image")
        stop_button = self.builder.get_object("test_stop_image")
        button_label = self.builder.get_object("btn_test_label")

        result = widget.get_active()

        play_button.set_visible(not result)
        stop_button.set_visible(result)

        if result:
            self.audioplayer.play(self.audiochooser.get_uri())
            button_label.set_text("Stop")
        else:
            self.audioplayer.stop()
            button_label.set_text("Test")

    def on_btn_cancel_clicked(self, widget, data=None):
        self.btn_test.set_active(False)
        self.destroy()
        pass
