# -*- Mode: Python; coding: utf-8; indent-tabs-mode: nil; tab-width: 4 -*-
### BEGIN LICENSE
# Copyright (C) 2012 John Vrbanac <john.vrbanac (at) linux.com>
# This program is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License version 3, as published 
# by the Free Software Foundation.
# 
# This program is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranties of 
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
# PURPOSE.  See the GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License along 
# with this program.  If not, see <http://www.gnu.org/licenses/>.
### END LICENSE

import locale
from locale import gettext as _
locale.textdomain('cuckoo')

from gi.repository import Gtk, Gdk, GdkPixbuf  # pylint: disable=E0611
import logging
logger = logging.getLogger('cuckoo')

import urlparse

from cuckoo_lib import Window
from cuckoo_lib.helpers import get_media_file
from cuckoo.CuckooAlarmList import CuckooAlarmList
from cuckoo.CuckooConfig import CuckooConfig
from cuckoo.CuckooAlarmButton import CuckooAlarmButton
from cuckoo.AboutCuckooDialog import AboutCuckooDialog
from cuckoo.AlarmController import AlarmController
from cuckoo.EditAlarmDialog import EditAlarmDialog
from cuckoo.PreferencesCuckooDialog import PreferencesCuckooDialog


class CuckooWindow(Window):
    __gtype_name__ = "CuckooWindow"
    config = None
    edit_alarm_dialog = None
    minimized_to_tray = False
    last_selected_alarm = None
    alarm_buttons = []
    alarm_controller = None

    def toggle_tray(self, toggle):
        self.set_skip_taskbar_hint(toggle)
        if toggle:
            self.iconify()
        else:
            self.set_opacity(1)
            self.deiconify()
            self.present()

        self.minimized_to_tray = toggle

    #  Builds the AlarmWidget buttons
    def create_alarm_button(self, data=None):
        logger.debug('Creating alarm button')
        alarm_layout = CuckooAlarmButton()
        alarm_layout.button.connect("clicked", self.alarm_button_clicked, alarm_layout)
        alarm_layout.button.connect("button_press_event", self.alarm_button_press)

        if not data == None:
            alarm_layout.set_data(data)
        else:
            # Setup Default values on add
            default_vol = self.config.get_data().get_default_volume()
            default_file = self.config.get_data().get_default_ring()

            alarm_layout.get_data().set_volume(default_vol)
            alarm_layout.get_data().set_audio_filename(default_file)

        self.alarm_buttons.append(alarm_layout)
        self.available_alarms_box.add(alarm_layout)
        self.available_alarms_box.show_all()

        return alarm_layout

    #  Builds new alarmlist and saves it data to Controller
    def save_alarm_data(self):
        # Saving alarm data
        alarmlist_data = []
        for data in self.alarm_buttons:
            alarmlist_data.append(data.get_data())
        AlarmController.Alarmlist.convert_and_save(alarmlist_data)

    def load_data(self, filename=None):
        # Clearing out alarm display box
        for child in self.alarm_buttons:
            self.available_alarms_box.remove(child)

        self.alarm_buttons = []
        self.config = CuckooConfig()

        self.config.load()
        AlarmController.Alarmlist.load(filename)

        # Build all buttons from controller data
        for data in AlarmController.Alarmlist.get_alarm_data_array():
            self.create_alarm_button(data)

        if len(self.alarm_buttons) <= 0:
            self.toggle_edit_delete_buttons(False)
        else:
            self.toggle_edit_delete_buttons(True)

    def reload_data(self):
        logger.debug('Save and reloading data')
        self.save_alarm_data()
        self.load_data()

    def show_alarmlist_dialog(self, save_mode=False):
        dialog = None

        # TODO: Needs to be condensed
        if not save_mode:
            dialog = Gtk.FileChooserDialog(
                "Open Alarm List",
                self,
                Gtk.FileChooserAction.OPEN)
            dialog.add_buttons(Gtk.STOCK_OPEN, Gtk.ResponseType.OK, "Cancel", Gtk.ResponseType.CANCEL)
        else:
            dialog = Gtk.FileChooserDialog(
                "Save Alarm List",
                self,
                Gtk.FileChooserAction.SAVE)
            dialog.add_buttons(Gtk.STOCK_SAVE, Gtk.ResponseType.OK, "Cancel", Gtk.ResponseType.CANCEL)

        # Adding Plain Text Filter
        filter_text = Gtk.FileFilter()
        filter_text.set_name("Alarm List Files")
        filter_text.add_mime_type("text/plain")
        dialog.add_filter(filter_text)

        response = dialog.run()

        if response == Gtk.ResponseType.OK:
            if not save_mode:
                self.load_data(dialog.get_filename())
                self.save_alarm_data()
            else:
                self.alarm_controller.Alarmlist.convert_and_save(None, dialog.get_filename())

        # Force destroy after response to address close button issue
        dialog.destroy()        

    """ Handlers """
    def on_save_alarm_list_item_activate(self, widget, data=None):
        self.show_alarmlist_dialog(True)

    def on_open_alarm_list_item_activate(self, widget, data=None):
        self.show_alarmlist_dialog(False)

    def on_activate_all_items(self, widget, data=None):
        self.alarm_controller.toggle_all(True)
        self.load_data()

    def on_deactivate_all_items(self, widget, data=None):
        self.alarm_controller.toggle_all(False)
        self.load_data()

    def alarm_activated(self, builder, data=None):
        self.reload_data()

    # Handle Double Click
    def alarm_button_press(self, widget, event):
        if event.type == 5:  # Gdk.EventType.2BUTTON_PRESS
            self.edit_alarm_clicked(widget)

    def alarm_button_clicked(self, widget, data):
        self.last_selected_alarm = data

    def remove_alarm_clicked(self, builder):
        if (self.last_selected_alarm != None):
            # Force stop alarm if playing
            self.alarm_controller.stop_alarm(self.last_selected_alarm.get_data())
            self.available_alarms_box.remove(self.last_selected_alarm)
            self.alarm_buttons.remove(self.last_selected_alarm)
            self.last_selected_alarm = None
        self.reload_data()

    def add_alarm_clicked(self, builder):
        alarm_box = self.create_alarm_button()

        # Kick off editor
        self.last_selected_alarm = alarm_box
        self.edit_alarm_clicked(builder)

    def edit_alarm_clicked(self, widget):
        if not self.last_selected_alarm == None and self.edit_alarm_dialog == None:
            logger.debug('create new editor dialog')
            self.edit_alarm_dialog = self.EditAlarmDialog()  # pylint: disable=E1102
            self.edit_alarm_dialog.connect('destroy', self.on_edit_dialog_destroyed)
            self.edit_alarm_dialog.show()
            self.edit_alarm_dialog.set_data(self.last_selected_alarm.get_data())

            # Disable all window actions until the edit dialog closes
            self.set_sensitive(False)

    def on_cuckoo_window_destroy(self, widget, data=None):
        self.config.save()
        self.save_alarm_data()

    def on_edit_dialog_destroyed(self, widget, data=None):
        logger.debug('editor dialog destroyed')
        if self.edit_alarm_dialog.response == Gtk.ResponseType.OK:
            self.last_selected_alarm.set_data(self.edit_alarm_dialog.get_data())

            # Force stop alarm if playing
            self.alarm_controller.stop_alarm(self.last_selected_alarm.get_data())

            self.reload_data()

        self.edit_alarm_dialog = None

        # Enable all window actions again
        self.toggle_edit_delete_buttons(True)
        self.set_sensitive(True)

    def on_mnu_show_activate(self, settings, data=None):
        self.toggle_tray(not self.minimized_to_tray)

    def on_mnu_preferences_activate(self, widget, data=None):
        if self.preferences_dialog is not None:
            logger.debug('show existing preferences_dialog')
            self.preferences_dialog.present()
        elif self.PreferencesDialog is not None:
            logger.debug('create new preferences_dialog')
            self.preferences_dialog = self.PreferencesDialog()  # pylint: disable=E1102
            self.preferences_dialog.connect('destroy', self.on_preferences_dialog_destroyed)
            self.preferences_dialog.show()
            self.preferences_dialog.set_data(self.config.get_data())

    def on_preferences_dialog_destroyed(self, widget, data=None):
        logger.debug('on_preferences_dialog_destroyed')
        # to determine whether to create or present preferences_dialog
        self.config.set_data(self.preferences_dialog.get_data())
        self.preferences_dialog = None

    def finish_initializing(self, builder):  # pylint: disable=E1002
        """Set up the main window"""
        super(CuckooWindow, self).finish_initializing(builder)

        self.set_opacity(0)

        self.EditAlarmDialog = EditAlarmDialog
        self.AboutDialog = AboutCuckooDialog
        self.PreferencesDialog = PreferencesCuckooDialog

        self.builder = builder
        self.available_alarms_box = builder.get_object("available_alarms_box")
        self.toolbar = builder.get_object("toolbar")
        self.addbutton = self.builder.get_object("add_button")
        self.editbutton = self.builder.get_object("edit_button")
        self.deletebutton = self.builder.get_object("delete_button")

        self.deletemenuitem = self.builder.get_object("mnu_delete_item")
        self.editmenuitem = self.builder.get_object("mnu_edit_item")

        # Uncomment for default icons on buttons
        #settings = Gtk.Settings.get_default()
        #settings.props.gtk_button_images = True

        # Load CSS
        screen = Gdk.Screen.get_default()
        css_provider = Gtk.CssProvider()
        css_provider.load_from_path(urlparse.urlparse(get_media_file("style.css")).path)
        context = Gtk.StyleContext()
        context.add_provider_for_screen(screen, css_provider, Gtk.STYLE_PROVIDER_PRIORITY_USER)

        self.alarm_controller = AlarmController(self)

        toolbar_style_context = self.toolbar.get_style_context()
        toolbar_style_context.add_class(Gtk.STYLE_CLASS_PRIMARY_TOOLBAR)

        self.config = CuckooConfig()
        self.load_data()

        self.startup_visible_id = self.connect("visibility_notify_event", self.startup_minimized_handler)

    def startup_minimized_handler(self, window, event):
        minimized = self.config.get_data().get_start_minimized()
        self.toggle_tray(minimized)

        self.disconnect(self.startup_visible_id)
        self.startup_visible_id = None

    def toggle_edit_delete_buttons(self, toggle):
        self.editbutton.set_sensitive(toggle)
        self.deletebutton.set_sensitive(toggle)
        self.deletemenuitem.set_sensitive(toggle)
        self.editmenuitem.set_sensitive(toggle)

        edit_active_icon = self.builder.get_object("edit_icon")
        edit_disabled_icon = self.builder.get_object("edit_disabled_icon")
        delete_active_icon = self.builder.get_object("delete_icon")
        delete_disabled_icon = self.builder.get_object("delete_disabled_icon")

        if toggle:
            self.editbutton.set_icon_widget(edit_active_icon)
            self.deletebutton.set_icon_widget(delete_active_icon)
        else:
            self.editbutton.set_icon_widget(edit_disabled_icon)
            self.deletebutton.set_icon_widget(delete_disabled_icon)
