# -*- Mode: Python; coding: utf-8; indent-tabs-mode: nil; tab-width: 4 -*-
### BEGIN LICENSE
# Copyright (C) 2012 John Vrbanac <john.vrbanac (at) linux.com>
# This program is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License version 3, as published 
# by the Free Software Foundation.
# 
# This program is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranties of 
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
# PURPOSE.  See the GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License along 
# with this program.  If not, see <http://www.gnu.org/licenses/>.
### END LICENSE

from cuckoo.CuckooAlarmData import CuckooAlarmData

import xdg.BaseDirectory
import ConfigParser
import os


class CuckooAlarmList():
    __default_filename = "alarms.cfg"
    __alarm_inc = 0
    __config_folder = ""
    __config_path = ""
    __config = None
    __alarm_data = []

    def __init__(self):
        config_home = xdg.BaseDirectory.xdg_config_home

        self.__config = ConfigParser.ConfigParser()
        self.__config_folder = os.path.join(config_home, "cuckoo")
        self.__config_path = os.path.join(self.__config_folder, self.__default_filename)

        self.check_and_create()

    def check_and_create(self):
        if not os.path.exists(self.__config_folder):
            os.mkdir(self.__config_folder)

    def generate_alarm_section(self, data):
        name = "alarm-" + str(self.__alarm_inc)
        self.__config.add_section(name)
        self.__config.set(name, "hour", str(data.get_hour()))
        self.__config.set(name, "minute", str(data.get_minute()))
        self.__config.set(name, "period", str(data.get_ampm()))
        self.__config.set(name, "filename", data.get_audio_filename())
        self.__config.set(name, "volume", str(data.get_volume()))
        self.__config.set(name, "activated", str(data.get_activated()))

        self.__alarm_inc += 1

        return name

    def get_alarm_data_array(self):
        return self.__alarm_data

    def read_alarm_section(self, name):
        data = CuckooAlarmData()
        data.set_hour(self.__config.get(name, "hour"))
        data.set_minute(self.__config.get(name, "minute"))
        data.set_ampm(self.__config.get(name, "period"))
        data.set_audio_filename(self.__config.get(name, "filename"))
        data.set_volume(self.__config.getfloat(name, "volume"))
        data.set_activated(self.__config.getboolean(name, "activated"))

        return data

    def toggle_all(self, toggle):
        for alarm in self.__alarm_data:
            alarm.set_activated(toggle)

    def sort_by_date(self, data):
        return sorted(data, key=lambda alarm: alarm.convert_to_time())

    def convert_and_save(self, array_of_data=None, filename=None):
        self.__alarm_inc = 0

        if array_of_data == None:
            array_of_data = self.__alarm_data

        self.__config = ConfigParser.ConfigParser()
        for data in self.sort_by_date(array_of_data):
            self.generate_alarm_section(data)

        self.save(filename)

    def parse_all_data(self):
        for section in self.__config.sections():
            self.__alarm_data.append(self.read_alarm_section(section))

    def load(self, filename=None):
        if filename == None:
            filename = self.__config_path

        self.__config = ConfigParser.ConfigParser()
        self.__alarm_data = []
        if not os.path.exists(filename):
            return
        self.__config.read(filename)
        self.parse_all_data()

    def save(self, filename=None):
        if filename == None:
            filename = self.__config_path

        sfile = open(filename, "w")
        self.__config.write(sfile)
