# -*- Mode: Python; coding: utf-8; indent-tabs-mode: nil; tab-width: 4 -*-
### BEGIN LICENSE
# Copyright (C) 2012 Jorge Alda jorgealda115@gmail.com
# This program is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License version 3, as published 
# by the Free Software Foundation.
# 
# This program is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranties of 
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
# PURPOSE.  See the GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License along 
# with this program.  If not, see <http://www.gnu.org/licenses/>.
### END LICENSE
from cmath import *
import gettext
from gettext import gettext as _
gettext.textdomain('alg3py')

class Alg3Exception(Exception):
    def __init__(self, value):
        self.parameter = value
    def __str__(self):
        return repr(self.parameter)

class Matrix(dict):
    def __init__ (self, rows, columns):
        indict = {}
        dict.__init__(self, indict)
        self.rows = rows
        self.columns = columns
        self.numbers = {}
        self.__editing = False

    def __setattr__ (self, name, value):
        if not self.__dict__.has_key('_Matrix__editing'):
            return dict.__setattr__(self, name, value)
        elif name == '_Matrix__editing':
            return dict.__setattr__(self, name, value)
        elif self.__editing == True:
            return dict.__setattr__(self, name, value)
        else:
            raise Alg3Exception(_("You're not allowed to change the matrix this way"))

    def __repr__ (self):
        output = """"""
        for row in range(0,self.rows):
            for col in range(0,self.columns):
                if (row, col) in self.numbers:
                    output = output + str(self.numbers[(row, col)]) + " "
                else:
                    output = output + "0 "
            output = output + "\n"
        return output

    def __setitem__ (self, index, value):
        if len(index) != 2:
            raise Alg3Exception(_("The index must be bidimensional"))
        if index[0] < 0 or index[0] >= self.rows or index[1] < 0 or index[1] >= self.columns:
            raise Alg3Exception(_("Index out of range"))
        self.__editing = True
        if value != 0:
            self.numbers[(index[0], index[1])] = value
        else:
            if (index[0], index[1]) in self.numbers:
                del self.numbers[(index[0], index[1])]
        self.__editing = False

    def __getitem__ (self, index):
        if len(index) != 2:
            raise Alg3Exception(_("The index must be bidimensional"))
        if index[0] < 0 or index[0] >= self.rows or index[1] < 0 or index[1] >= self.columns:
            raise Alg3Exception(_("Index out of range"))
        if (index[0], index[1]) in self.numbers:
            return self.numbers[(index[0], index[1])]
        else:
            return 0

    def copy(self):
        res = Matrix(self.rows, self.columns)
        for r in range(0, self.rows):
            for c in range(0, self.columns):
                res[(r, c)] = self[(r, c)]
        return res

    def __add__ (A, B):
        if A.rows != B.rows or A.columns != B.columns:
            raise Alg3Exception(_("Matrices have different dimensions"))
        res = Matrix(A.rows, A.columns)
        for row in range(0,A.rows):
            for column in range(0, A.columns):
                value = A[(row, column)] + B[(row, column)]
                if value != 0:
                    res[(row, column)] = value
        return res

    def __sub__ (A, B):
        if A.rows != B.rows or A.columns != B.columns:
            raise Alg3Exception(_("Matrices have different dimensions"))
        res = Matrix(A.rows, A.columns)
        for row in range(0,A.rows):
            for column in range(0, A.columns):
                value = A[(row, column)] - B[(row, column)]
                if value != 0:
                    res[(row, column)] = value
        return res

    def __mul__ (A,B):
        if isinstance(B, Matrix):
            if A.columns != B.rows:
                raise Alg3Exception(_("Matrices can't be multiplied"))
            res = Matrix(A.rows, B.columns)
            for k in range(0, A.rows):
                for l in range(0, B.columns):
                    value = 0
                    for m in range(0, A.columns):
                        value += A[(k,m)] * B[(m,l)]
                    if value != 0:
                        res[(k, l)] = value
            return res
        elif isinstance(B, Vect):
            if A.columns == B.dimension:
                res = Vect(A.rows)
                for c in range (0, A.columns):
                    for r in range(0, A.rows):
                        res[r] = res[r] + A[(r, c)] * B[c]
                return res
            else:
                raise Alg3Exception(_('Matrix and vector with different sizes'))
        else:
            res = Matrix(A.rows, A.columns)
            if B != 0:
                for row in range(0, A.rows):
                    for col in range(0, A.columns):
                        if A[(row, col)] != 0:
                            res[(row, col)] = A[(row, col)] * B
            return res


    def __mod__ (A, B):
        res = Matrix(A.rows*B.rows, A.columns*B.columns)
        for k in range(0, res.rows):
            for l in range(0, res.columns):
                res[(k, l)] = A[(k/B.rows, l/B.columns)] * B[(k%B.rows, l%B.columns)]
        return res

    def simplify(A):
        for r in range(0, A.rows):
            for c in range(0, A.columns):
                if (A[(r,c)]).imag == 0:
                    A[(r,c)] = (A[(r,c)]).real

    def trace(self):
        if self.rows != self.columns:
            raise Alg3Exception(_('Matrix must be square'))
        else:
            res = 0
            for c in range(0, self.columns):
                res = res + self[(c,c)]
            return res

    def transpose(self):
        res = Matrix(self.columns, self.rows)
        for r in range(0, self.rows):
            for c in range(0, self.columns):
                res[(c, r)] = self[(r, c)]
        return res

    def adjoint(self):
        res = Matrix(self.columns, self.rows)
        for r in range(0, self.rows):
            for c in range(0, self.columns):
                res[(c, r)] = self[(r, c)].conjugate()
        return res

    def minor(self, index):
        if len(index) != 2:
            raise Alg3Exception(_('Index must be bidimensional'))
        elif (index[0] > self.rows) or (index[1] > self.columns):
            raise Alg3Exception(_('Index out of range'))
        else:
            res = Matrix(self.rows-1, self.columns-1)
            for r in range(0, self.rows - 1):
                if r < index[0]:
                    for c in range(0, self.columns - 1):
                        if c < index[1]:
                            res[(r, c)] = self[(r, c)]
                        else:
                            res[(r, c)] = self[(r, c+1)]
                else:
                    for c in range(0, self.columns-1):
                        if c < index[1]:
                            res[(r, c)] = self[(r+1, c)]
                        else:
                            res[(r, c)] = self[(r+1, c+1)]
            return res

    def determinant(self):
        if self.columns != self.rows:
            raise Alg3Exception(_('Matrix must be square'))
        else:
            if self.rows == 1:
                return self[(0,0)]
            else:
                res = 0
                sign = 1
                for c in range(0, self.columns):
                    res = res + sign * self[(0, c)] * self.minor((0, c)).determinant()
                    sign = -sign
                return res

    def adjugate(self):
        if self.columns != self.rows:
            raise Alg3Exception(_('Matrix must be square'))
        else:
            res = Matrix(self.rows, self.columns)
            for r in range(0, self.rows):
                for c in range(0, self.columns):
                    s = (r + c)%2
                    if s == 0:
                        sign = 1
                    else:
                        sign = -1
                    res[(r, c)] = sign * self.minor((c, r)).determinant()
            return res

    def inverse(self):
        if self.columns != self.rows:
            raise Alg3Exception(_('Matrix must be square'))
        else:
            det = self.determinant()
            if det == 0:
                raise Alg3Exception(_('Matrix not invertible'))
            else:
                return self.adjugate() * (1.0 / self.determinant())

    def extract_column(self, col):
        res = Vect(self.rows)
        for r in range(0, res.dimension):
            res[r] = self[(r, col)]
        return res

    def extract_row(self, row):
        res = Vect(self.columns)
        for c in range(0, res.dimension):
            res[c] = self[(row, c)]
        return res

#---------------------------------------------------------------------------------------------------

class Vect(dict):
    def __init__ (self, dim):
        indict = {}
        dict.__init__(self, indict)
        self.dimension = dim
        self.numbers = {}
        self.__editing = False

    def __setattr__ (self, name, value):
        if not self.__dict__.has_key('_Vect__editing'):
            return dict.__setattr__(self, name, value)
        elif name == '_Vect__editing':
            return dict.__setattr__(self, name, value)
        elif self.__editing == True:
            return dict.__setattr__(self, name, value)
        else:
            raise Alg3Exception(_("You're not allowed to change the vector this way"))

    def __repr__ (self):
        output = ""
        for element in range(0,self.dimension):
            if element in self.numbers:
                output = output + str(self.numbers[element]) + "  "
            else:
                output = output + "0  "
        return output

    def __setitem__ (self, index, value):
        if index < 0 or index >= self.dimension:
            raise Alg3Exception(_("Index out of range"))
        self.__editing = True
        if value != 0:
            self.numbers[index] = value
        else:
            if index in self.numbers:
                del self.numbers[index]
        self.__editing = False

    def __getitem__ (self, index):
        if index < 0 or index >= self.dimension:
            raise Alg3Exception(_("Index out of range"))
        if index in self.numbers:
            return self.numbers[index]
        else:
            return 0

    def copy(self):
        res = Vect(self.dimension)
        for elem in range(0, self.dimension):
            res[elem] = self[elem]
        return res

    def __add__ (A, B):
        if A.dimension != B.dimension:
            raise Alg3Exception(_('Vectors have different dimensions'))
        else:
            res = Vect(A.dimension)
            for elem in range(0, A.dimension):
                res[elem] = A[elem] + B[elem]
            return res

    def __sub__ (A, B):
        if A.dimension != B.dimension:
            raise Alg3Exception(_('Vectors have different dimensions'))
        else:
            res = Vect(A.dimension)
            for elem in range(0, A.dimension):
                res[elem] = A[elem] - B[elem]
            return res

    def __mul__ (A, B):
        if isinstance(B, Vect):
            if A.dimension != B.dimension:
                raise Alg3Exception(_('Vectors have different dimensions'))
            else:
                res = 0
                for elem in range(0, A.dimension):
                    res = res + (A[elem].conjugate() * B[elem])
                return res   
        else:
            res = Vect(A.dimension)
            for elem in range(0, A.dimension):
                res[elem] = A[elem] * B
            return res

    def __len__ (self):
        return self.dimension

    def __abs__ (self):
        res = 0
        for d in range(0, self.dimension):
            res = res + self[d].conjugate() * self[d]
        return sqrt(res)
    
    def convert2rmatrix(self):
        res = Matrix(1, self.dimension)
        for d in range(0, self.dimension):
            res[(0, d)] = self[d]
        return res

    def convert2cmatrix(self):
        res = Matrix(self.dimension, 1)
        for d in range(0, self.dimension):
            res[(d, 0)] = self[d]
        return res

    def __xor__ (A, B):
        if A.dimension == B.dimension and A.dimension == 3:
            res = Vect(3)
            res[0] = A[1] * B[2] - A[2] * B[1]
            res[1] = A[2] * B[0] - A[0] * B[2]
            res[2] = A[0] * B[1] - A[1] * B[0]
            return res

    def simplify(self):
        for d in range(0, self.dimension):
            if self[d].imag == 0:
                self[d] = self[d].real


#---------------------------------------------------------------------------------------------


def save(A, path):
    f = open(path, "w")
    if isinstance(A, Matrix):
        f.write('alg3py -- Matrix\n')
    elif isinstance(A, Vect):
        f.write('alg3py -- Vector\n')
    else:
        f.write('alg3py -- Scalar\n')
    f.write(str(A))
    f.close()

def load_Mat(path):
    if isinstance(path, str):
        f = open(path, "r")
    else:
        f = path
    l = f.readlines()
    if l[0] == 'alg3py -- Matrix\n':
        for t in range(1, len(l)):
            l[t] = l[t].split()
        res = Matrix(len(l)-1, len(l[1]))
        for r in range(1, len(l)):
            for c in range(0, len(l[1]) ):
                try:
                    res[(r-1, c)] = complex(l[r][c])
                except:
                    print(str(r) + '\t' + str(c) + '\n')
        res.simplify()
        return res
    elif l[0] == 'alg3py -- Scalar\n':
        l[1] = l[1].split()
        return complex(l[1][0])
    elif l[0] == 'alg3py -- Vector\n':
        l[1] = l[1].split()
        res = Vect(len(l[1]))
        for d in range(0, res.dimension):
            res[d] = complex(l[1][d])
        return res
    else:
        raise Alg3Exception(_('Courrupt file!'))

def export_tex(A, path):
    if isinstance(A, Matrix):
        f = open(path, "w")
        f.write("\\[ \\left( \\begin{array}{" + 'c'*A.columns + "}\n")
        for r in range(0, A.rows - 1):
            for c in range(0, A.columns - 1):
                f.write(str(A[(r, c)]) + " & ")
            f.write(str(A[r, A.columns-1]) + " \\\\ \n")
        for c in range(0, A.columns - 1):
            f.write(str(A[(A.rows - 1, c)]) + " & ")
        f.write(str(A[A.rows-1, A.columns-1]) + "\n\\end{array} \\right) \\]" )
        f.close()
    elif isinstance(A, Vect):
        f = open(path, "w")
        f.write('\\[\\left( ')
        for d in range(0, A.dimension-1):
            f.write(str(A[d]) + ',~')
        f.write(str(A[A.dimension - 1]) + ' \\right)\\]')
    else:
        f = open(path, "w")
        f.write('\\[' + str(A) + '\\] ')

def export_MathML(A, path):
    if isinstance(A, Matrix):
        f = open(path, "w")
        f.write('<?xml version="1.0" encoding="UTF-8"?>\n<!DOCTYPE math PUBLIC "-//W3C//DTD MathML 2.0//EN" "http://www.w3c.org/Math/DTD/mathml2/mathml2.dtd">\n\t' + "<math xmlns='http://www.w3.org/1998/Math/MathML'>\n\t\t<mo>(</mo> <mtable>\n\t\t\t")
        for r in range(0, A.rows):
            f.write('<mtr> ')
            for c in range(0, A.columns):
                f.write('<mn>' + str(A[r, c]) + '</mn>')
            f.write('</mtr>\n\t\t\t')
        f.write('</mtable> <mo>)</mo>\n\t</math>')
        f.close()
    elif isinstance(A, Vect):
        f = open(path, "w")
        f.write('<?xml version="1.0" encoding="UTF-8"?>\n<!DOCTYPE math PUBLIC "-//W3C//DTD MathML 2.0//EN" "http://www.w3c.org/Math/DTD/mathml2/mathml2.dtd">\n\t' + "<math xmlns='http://www.w3.org/1998/Math/MathML'>\n\t\t<mo>(</mo> <mtable>\n\t\t\t")
        for r in range(0, A.dimension):
            f.write('<mtr> ')
            f.write('<mn>' + str(A[r]) + '</mn>')
            f.write('</mtr>\n\t\t\t')
        f.write('</mtable> <mo>)</mo>\n\t</math>')
        f.close()
    else:
        f = open(path, "w")
        f.write('<?xml version="1.0" encoding="UTF-8"?>\n<!DOCTYPE math PUBLIC "-//W3C//DTD MathML 2.0//EN" "http://www.w3c.org/Math/DTD/mathml2/mathml2.dtd">\n\t' + "<math xmlns='http://www.w3.org/1998/Math/MathML'>\n\t\t<mn>" + str(A) + '</mn>\n\t</math>')
        f.close()

def text2Matrix(text):
    l = text.splitlines()
    for t in range(0, len(l)):
        l[t] = l[t].split()
    res = Matrix(len(l), len(l[0]))
    for r in range(0, len(l)):
        for c in range(0, len(l[0]) ):
            res[(r, c)] = complex(l[r][c])
    res.simplify()
    return res

def matrix_from_cols(vects):
    if isinstance(vects, list):
        try:
            res = Matrix(len(vects[0]), len(vects))
            for r in range(0, res.rows):
                for c in range(0, res.columns):
                    res[(r,c)] = vects[c][r]
            return res
        except:
            raise Alg3Exception(_("Not a valid list of Vect's"))
